#ifndef SL_CAMERA_INFO_H_
#define SL_CAMERA_INFO_H_

namespace sl
{
struct NetInfo
{
    std::string ip;
    std::string model;
    std::string hardwareVersion;
    std::string softwareVersion;
    std::string username;
    std::string password;
    int version = 0; // 由softwareVersion转化至整数而来
    int port = 0;
};

struct CameraResolution
{
    CameraResolution() : width(1920), height(1080), size("1920*1080") {}

    CameraResolution(int inWidth, int inHeight)
        : width(inWidth),
          height(inHeight),
          size(std::to_string(inWidth).append("*").append(
              std::to_string(inHeight)))
    {
    }

    CameraResolution(int inWidth, int inHeight, const std::string &inSize)
        : width(inWidth), height(inHeight), size(inSize)
    {
    }

    CameraResolution &operator=(const CameraResolution &other)
    {
        width = other.width;
        height = other.height;
        size = other.size;
        return *this;
    }

    bool operator==(const CameraResolution &other)
    {
        if (width == other.width && height == other.height &&
            size.compare(other.size) == 0)
            return true;

        return false;
    }

    bool operator!=(const CameraResolution &other) { return !(*this == other); }

    int width;
    int height;
    std::string size;
};

struct CommonAttribute
{
    int m_cur;
    int m_min;
    int m_max;
    int m_def;
    int m_res;
    CommonAttribute() : m_cur(-1), m_min(0), m_max(0), m_def(0), m_res(1) {}
};

// 视频流格式
// MJPEG所有相机均支持(除F801W), H264目前仅A3xx/F801W支持
enum class StreamFormat
{
    kMJPEG,
    kH264
};

enum class PicStreamFormat
{
    kJPEG
};

// 解码后数据格式
// MJPEG, H264解码后原生数据格式为YUV420P
enum class PixFormat
{
    kYUV420P,
    kRGB888,
    kBGR888,
    kARGB8888,
    kABGR8888,
    kRGBA8888,
    kBGRA8888
};

enum class ZoomMode
{
    ZoomAbsolute = 1,
    ZoomRelative,
};

struct CameraPicStreamSettings
{
    CameraResolution resolution;
    PicStreamFormat picStreamFormat = PicStreamFormat::kJPEG;
    // highest/higher/high/middle/low/lower/lowest
    std::string quality;
    int interval = 333;
};

struct CameraStreamSettings
{
    NetInfo netInfo;
    CameraResolution resolution;
    StreamFormat streamFormat = StreamFormat::kMJPEG;
    PixFormat pixFormat = PixFormat::kYUV420P;
    // cbr(定码率) / vbr(可变码率)
    std::string rcMode;
    // highest/higher/high/middle/low/lower/lowest
    std::string quality;
    int frameRate = 0;
    int bitrate = 0;
    int keyFrameInterval = 0;
};

enum FocusMode
{
    kFocusAuto = 0,
    kFocusManual,
    // 一键聚焦
    kFocusOnce
};

enum WbMode
{
    kWBAuto = 0,
    kWBManual
};

enum ExposureMode
{
    kExpAuto = 0,
    kExpManual
};

struct FocusRegion
{
    bool enable = false;
    int mode = 0;
    CommonAttribute x;
    CommonAttribute y;
};

struct FocusRegionV2
{
    bool enable = false;
    int x = 0;
    int y = 0;
    int width = 0;
    int height = 0;
};

struct FocusInfo
{
    FocusMode mode = FocusMode::kFocusAuto;
    CommonAttribute value;
    CommonAttribute speed;
    // 1-对应FocusRegion，2-对应FocusRegionV2
    int regionVersion = 1;
    FocusRegion region;
    FocusRegionV2 regionV2;
};

struct ExposureInfo
{
    ExposureMode mode = ExposureMode::kExpAuto;
    CommonAttribute bright;
    CommonAttribute gain;
    CommonAttribute time;
};

struct WhiteBalanceInfo
{
    WbMode mode = WbMode::kWBAuto;
    // 色温
    CommonAttribute temperature;
    CommonAttribute red;
    CommonAttribute green;
    CommonAttribute blue;
};

struct LedPartInfo
{
    // LED灯分区号
    int index = 0;
    // LED开关
    bool enable = true;
    // LED灯亮度
    int level = 0;
};

struct LedInfo
{
    // 分区数量
    int partNumber = 0;
    // 亮度范围
    CommonAttribute levelRange;
    std::vector<LedPartInfo> ledPartVec;
};

struct IpInfo
{
    // 是否具有无线功能
    bool enWireless;
    bool dhcp;
    std::string ip;
    std::string gateway;
    std::string netmask;
    std::string mac;
    std::string dns1;
    std::string dns2;

    bool wirelessDhcp;
    std::string wirelessIp;
    std::string wirelessGateway;
    std::string wirelessNetmask;
    std::string wirelessMac;
    std::string wirelessDns1;
    std::string wirelessDns2;
    // 无线模式，AP/STA
    std::string wirelessType;
    // 无线频段，2.4G/5G
    std::string wirelessNet;
    // AP名称
    std::string wirelessApSsid;
    // AP密码
    std::string wirelessApPasswd;
    // STA名称
    std::string wirelessStaSsid;
    // STA密码
    std::string wirelessStaPasswd;
    std::string wirelessEnc;
    // 下面两个兼容F801W
    //  当wirelessType为AP时，wirelessSsid为AP名称，wirelessPasswd为AP密码
    //  当wirelessType为STA时，wirelessSsid为路由器WIFI名称，wirelessPasswd为路由器WIFI密码
    std::string wirelessSsid;
    std::string wirelessPasswd;

    IpInfo() : enWireless(false), dhcp(false), wirelessDhcp(false) {}

    IpInfo &operator=(const IpInfo &other)
    {
        dhcp = other.dhcp;
        ip = other.ip;
        gateway = other.gateway;
        netmask = other.netmask;
        mac = other.mac;
        dns1 = other.dns1;
        dns2 = other.dns2;

        wirelessDhcp = other.wirelessDhcp;
        wirelessIp = other.wirelessIp;
        wirelessGateway = other.wirelessGateway;
        wirelessNetmask = other.wirelessNetmask;
        wirelessMac = other.wirelessMac;
        wirelessDns1 = other.wirelessDns1;
        wirelessDns2 = other.wirelessDns2;
        wirelessType = other.wirelessType;
        wirelessNet = other.wirelessNet;
        wirelessSsid = other.wirelessSsid;
        wirelessPasswd = other.wirelessPasswd;
        wirelessApSsid = other.wirelessApSsid;
        wirelessApPasswd = other.wirelessApPasswd;
        wirelessStaSsid = other.wirelessStaSsid;
        wirelessStaPasswd = other.wirelessStaPasswd;
        wirelessEnc = other.wirelessEnc;
        enWireless = other.enWireless;
        return *this;
    }

    bool operator==(const IpInfo &other)
    {
        if (dhcp == other.dhcp && ip.compare(other.ip) == 0 &&
            gateway.compare(other.gateway) == 0 &&
            netmask.compare(other.netmask) == 0 &&
            dns1.compare(other.dns1) == 0 && dns2.compare(other.dns2) == 0 &&
            wirelessDhcp == other.wirelessDhcp &&
            wirelessIp.compare(other.wirelessIp) == 0 &&
            wirelessGateway.compare(other.wirelessGateway) == 0 &&
            wirelessNetmask.compare(other.wirelessNetmask) == 0 &&
            wirelessMac.compare(other.wirelessMac) == 0 &&
            wirelessDns1.compare(other.wirelessDns1) == 0 &&
            wirelessDns2.compare(other.wirelessDns2) == 0 &&
            wirelessType.compare(other.wirelessType) == 0 &&
            wirelessNet.compare(other.wirelessNet) == 0 &&
            wirelessSsid.compare(other.wirelessSsid) == 0 &&
            wirelessPasswd.compare(other.wirelessPasswd) == 0 &&
            wirelessApSsid.compare(other.wirelessApSsid) == 0 &&
            wirelessApPasswd.compare(other.wirelessApPasswd) == 0 &&
            wirelessStaSsid.compare(other.wirelessStaSsid) == 0 &&
            wirelessStaPasswd.compare(other.wirelessStaPasswd) == 0 &&
            wirelessEnc.compare(other.wirelessEnc) == 0 &&
            enWireless == other.enWireless)
            return true;

        return false;
    }

    bool operator!=(const IpInfo &other) { return !(*this == other); }
};

struct CameraInfo
{
    std::string model;
    std::string modelSuffix;
    FocusInfo focus;
    ExposureInfo exposure;
    WhiteBalanceInfo whiteBalance;
    LedInfo led;
    IpInfo ipInfo;

    CommonAttribute hue;                  // 色度
    CommonAttribute saturation;           // 饱和度
    CommonAttribute contrast;             // 对比度
    CommonAttribute sharpness;            // 锐度
    CommonAttribute gamma;                // Gamma值
    CommonAttribute drc;                  // 动态范围压缩
    CommonAttribute zoomSpeed;            // 变倍速率
    CommonAttribute zoom;                 // 光学变倍
    CommonAttribute dzoom;                // 数字变倍
    CommonAttribute contrastEnhancement;  // 对比度增强

    bool defog = false;                 // 除雾
    bool powerLineFrequency = false;    // 抗频闪
    bool monochrome = false;            // 黑白
    bool flip = false;                  // 翻转
    bool mirror = false;                // 镜像
    bool cac = false;                   // 色差校正
    bool sharpnessEnhancement = false;  // 锐度增强
    int sceneMode = 0;                  // 场景 0-生物，1-工业，2-金相，3-单筒
};

}  // namespace sl

#endif  // SL_CAMERA_INFO_H_
