﻿#include <SDL.h>
#ifdef _WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

#include <iostream>

#include "net_camera.h"

// 注意：如需使用RGB格式的视频帧，取消以下的注释即可
//       但会消耗额外的资源, 并可能增加画面延时
// #define SLDEMO_RGB

using namespace std;

void mySleep(int ms)
{
#ifdef _WIN32
    Sleep(ms);
#else
    usleep(ms * 1000);
#endif
}

int main(int argc, char *argv[])
{
    int select = -1;
    std::vector<sl::NetInfo> infos;
    sl::NetCamera::initNet();
    mySleep(1000);

    while (true)
    {
        infos = sl::NetCamera::searchAvailableCameras();
        for (size_t i = 0; i < infos.size(); ++i)
        {
            std::cout << "Camera " << i + 1 << ", IP=" << infos[i].ip
                      << ", model=" << infos[i].model
                      << ", version=" << infos[i].version << std::endl;
        }
        std::cout << "请选择相机, 0重新搜索可用相机, -1退出选择" << std::endl;
        std::cin >> select;
        std::cout << "select=" << select << std::endl;

        select = select - 1;

        if (select == (0 - 1))
            continue;

        break;
    }

    if (infos.size() == 0 || select < 0 || select >= infos.size())
    {
        sl::NetCamera::destroyNet();
        return 0;
    }

    sl::CameraStreamSettings settings;
    settings.netInfo = infos[select];
    // 目前所有网络相机账号密码均为此值
    settings.netInfo.username = "admin";
    settings.netInfo.password = "123456";
    settings.resolution = sl::CameraResolution(1920, 1080);
    // H264目前仅A3xx/F801W支持
    // MJPEG所有相机均支持(除F801W)
    // F801W仅支持h264, resolution=1920x1080, framerate=25, bitrate=10000
    // WIFI相机仅支持h264格式，分辨率仅支持1920x1080，码率最高为30000
    settings.streamFormat = sl::StreamFormat::kMJPEG;
#ifndef SLDEMO_RGB
    settings.pixFormat = sl::PixFormat::kYUV420P;
#else
    settings.pixFormat = sl::PixFormat::kRGB888;
#endif
    settings.rcMode = "cbr";
    settings.quality = "high";
    // C314/M114 仅支持30帧
    settings.frameRate = 25;
    settings.bitrate = 86000;
    settings.keyFrameInterval = 120;
    auto camera =
        sl::NetCamera::create(settings.netInfo.model, settings.netInfo.version);
    camera->setCameraStreamSettings(settings);
    camera->login();
    sl::CameraInfo cameraInfo = camera->getNetCameraAllInfo();
    camera->openStream();
    // 图片流设置
    sl::CameraPicStreamSettings picSettings;
    // 图片流分辨率请根据相机实际情况调整
    picSettings.resolution = sl::CameraResolution(1920, 1080);
    picSettings.picStreamFormat = sl::PicStreamFormat::kJPEG;
    picSettings.quality = "high";
    picSettings.interval = 333;
    camera->setPicStream(picSettings);
    //camera->requestPicStream();

    // 视频窗体相关, 无需关心
    SDL_Event event;
    SDL_Window *window = nullptr;
    SDL_Renderer *sdlRenderer = nullptr;
    SDL_Texture *sdlTexture = nullptr;
    SDL_Rect sdlRect;
    const int screen_w = 960;
    const int screen_h = 540;
    sdlRect.x = 0;
    sdlRect.y = 0;
    sdlRect.w = screen_w;
    sdlRect.h = screen_h;
#ifndef SLDEMO_RGB
    Uint32 pixformat = SDL_PIXELFORMAT_IYUV;
#else
    Uint32 pixformat = SDL_PIXELFORMAT_RGB24;
#endif

    SDL_Init(SDL_INIT_VIDEO);
    // Create sdl window
    window = SDL_CreateWindow(
        "Net SDK Demo", SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED,
        screen_w, screen_h, SDL_WINDOW_RESIZABLE);

    for (int i = 0; i < SDL_GetNumRenderDrivers(); ++i)
    {
        SDL_RendererInfo rendererInfo = {};
        SDL_GetRenderDriverInfo(i, &rendererInfo);
        if (rendererInfo.name != std::string("direct3d"))
        {
            continue;
        }

        sdlRenderer = SDL_CreateRenderer(
            window, i, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);
        break;
    }

    if (!sdlRenderer)
        sdlRenderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_PRESENTVSYNC);

    sdlTexture = SDL_CreateTexture(
        sdlRenderer, pixformat, SDL_TEXTUREACCESS_STREAMING,
        settings.resolution.width, settings.resolution.height);

    SDL_bool done = SDL_FALSE;
    Uint32 windowID = SDL_GetWindowID(window);
    Uint64 getFrameFailedCount = 0;
    auto &ipInfo = cameraInfo.ipInfo;
    while (true)
    {
        while (SDL_PollEvent(&event))
        {
            switch (event.type)
            {
                // 在视频窗口按下按键
                case SDL_KEYDOWN:
                    if (event.key.keysym.sym == SDLK_ESCAPE)  // 按ESC键退出
                    {
                        done = SDL_TRUE;
                    }
                    else if (
                        event.key.keysym.sym == SDLK_q)  // 视频画面按下A键触发
                    {
                        static bool changed = true;

                        if (changed)
                            settings.frameRate = 20;
                        else
                            settings.frameRate = 25;

                        camera->changeStreamSettings(settings);
                        changed = !changed;
                    }
                    else if (event.key.keysym.sym == SDLK_w)
                    {
                        // cameraInfo.focus.region.enable = false;
                        // cameraInfo.focus.region.x.m_cur = 7;
                        // cameraInfo.focus.region.y.m_cur = 8;
                        // camera->setFocusRegion(
                        //     cameraInfo.focus.region.enable,
                        //     cameraInfo.focus.region.x.m_cur,
                        //     cameraInfo.focus.region.y.m_cur, 0);
                        bool enable = false;
                        int x;
                        int y;
                        int width;
                        int height;
                        cameraInfo.focus.regionV2.enable = true;
                        cameraInfo.focus.regionV2.x = 100;
                        cameraInfo.focus.regionV2.y = 200;
                        cameraInfo.focus.regionV2.width = 112;
                        cameraInfo.focus.regionV2.height = 72;
                        camera->setFocusRegionV2(
                            cameraInfo.focus.regionV2.enable,
                            cameraInfo.focus.regionV2.x,
                            cameraInfo.focus.regionV2.y,
                            cameraInfo.focus.regionV2.width,
                            cameraInfo.focus.regionV2.height);
                        bool focusing = false;
                        camera->getFocusState(focusing);
                        std::cout << "Focusing=" << focusing << std::endl;
                        camera->getFocusRegionV2(enable, x, y, width, height);
                    }
                    else if (event.key.keysym.sym == SDLK_e)
                    {
                        cameraInfo.powerLineFrequency = true;
                        camera->setPowerLineFrequency(cameraInfo.powerLineFrequency);
                        camera->changeStreamSettings(settings);
                    }
                    else if (event.key.keysym.sym == SDLK_r)
                    {
                        cameraInfo.flip = false;
                        camera->setFlip(cameraInfo.flip);
                    }
                    else if (event.key.keysym.sym == SDLK_t)
                    {
                        ipInfo.dhcp = false;
                        ipInfo.ip = "192.168.3.230";
                        ipInfo.gateway = "192.168.3.1";
                        ipInfo.netmask = "255.255.255.0";
                        camera->setIpv4(
                            ipInfo.dhcp, ipInfo.ip, ipInfo.gateway,
                            ipInfo.netmask, ipInfo.dns1, ipInfo.dns2,
                            ipInfo.mac);
                    }
                    else if (event.key.keysym.sym == SDLK_y)
                    {
                        cameraInfo.exposure.mode = sl::ExposureMode::kExpAuto;
                        cameraInfo.exposure.bright.m_cur =
                            (cameraInfo.exposure.bright.m_cur + 15) % cameraInfo.exposure.bright.m_max;
                        camera->setExposure(
                            cameraInfo.exposure.mode,
                            cameraInfo.exposure.bright.m_cur,
                            cameraInfo.exposure.gain.m_cur,
                            cameraInfo.exposure.time.m_cur);
                    }
                    else if (event.key.keysym.sym == SDLK_u)
                    {
                        cameraInfo.exposure.mode = sl::ExposureMode::kExpManual;
                        cameraInfo.exposure.gain.m_cur =
                            (cameraInfo.exposure.gain.m_cur + 3) % cameraInfo.exposure.gain.m_max;
                        cameraInfo.exposure.time.m_cur =
                            (cameraInfo.exposure.time.m_cur + 15) % cameraInfo.exposure.time.m_max;
                        camera->setExposure(
                            cameraInfo.exposure.mode,
                            cameraInfo.exposure.bright.m_cur,
                            cameraInfo.exposure.gain.m_cur,
                            cameraInfo.exposure.time.m_cur);
                    }
                    else if (event.key.keysym.sym == SDLK_i)
                    {
                        cameraInfo.whiteBalance.mode = sl::WbMode::kWBAuto;
                        camera->setWb(
                            cameraInfo.whiteBalance.mode,
                            cameraInfo.whiteBalance.red.m_cur,
                            cameraInfo.whiteBalance.green.m_cur,
                            cameraInfo.whiteBalance.blue.m_cur);
                    }
                    else if (event.key.keysym.sym == SDLK_o)
                    {
                        cameraInfo.whiteBalance.mode = sl::WbMode::kWBManual;
                        cameraInfo.whiteBalance.red.m_cur =
                            (cameraInfo.whiteBalance.red.m_cur + 100) % cameraInfo.whiteBalance.red.m_max;
                        cameraInfo.whiteBalance.green.m_cur =
                            (cameraInfo.whiteBalance.green.m_cur + 100) % cameraInfo.whiteBalance.green.m_max;
                        cameraInfo.whiteBalance.blue.m_cur =
                            (cameraInfo.whiteBalance.blue.m_cur + 100) % cameraInfo.whiteBalance.red.m_max;
                        camera->setWb(
                            cameraInfo.whiteBalance.mode,
                            cameraInfo.whiteBalance.red.m_cur,
                            cameraInfo.whiteBalance.green.m_cur,
                            cameraInfo.whiteBalance.blue.m_cur);
                    }
                    else if (event.key.keysym.sym == SDLK_p)
                    {
                        cameraInfo.focus.mode = sl::FocusMode::kFocusAuto;
                        camera->setFocus(
                            cameraInfo.focus.mode, cameraInfo.focus.value.m_cur,
                            cameraInfo.focus.speed.m_def);
                    }
                    else if (event.key.keysym.sym == SDLK_a)
                    {
                        cameraInfo.focus.mode = sl::FocusMode::kFocusManual;
                        cameraInfo.focus.value.m_cur =
                            (cameraInfo.focus.value.m_cur + 50) %
                            cameraInfo.focus.value.m_max;
                        camera->setFocus(
                            cameraInfo.focus.mode, cameraInfo.focus.value.m_cur,
                            cameraInfo.focus.speed.m_def);
                    }
                    else if (event.key.keysym.sym == SDLK_s)
                    {
                        cameraInfo.zoom.m_cur = (cameraInfo.zoom.m_cur + 10) %
                                                cameraInfo.zoom.m_max;
                        cameraInfo.dzoom.m_cur = 0;
                        camera->setZoom(
                            cameraInfo.zoom.m_cur, cameraInfo.dzoom.m_cur,
                            cameraInfo.zoomSpeed.m_cur,
                            sl::ZoomMode::ZoomAbsolute);
                    }
                    else if (event.key.keysym.sym == SDLK_d)
                    {
                        static int level = 0;
                        for (int i = 0; i < cameraInfo.led.ledPartVec.size();
                             ++i)
                            camera->setLed(
                                cameraInfo.led.ledPartVec[i].index, level);
                        level = (level + 10) % cameraInfo.led.levelRange.m_max;
                    }
                    else if (event.key.keysym.sym == SDLK_f)
                    {
                        camera->upgrade(
                            "D:/Users/13470/updates/m114_update_v1.2.3.lpk");
                    }
                    else if (event.key.keysym.sym == SDLK_g)
                    {
                        std::string status;
                        double percentage = 0;
                        camera->getUpgradeStatus(status, percentage);
                        std::cout << "Upgrade status=" << status
                                  << ", percentage=" << percentage << std::endl;
                    }
                    else if (event.key.keysym.sym == SDLK_h)
                    {
                        // 设置无线IP相关信息, 仅在STA模式下生效(除Mac地址外)
                        ipInfo.wirelessDhcp = true;
                        // ipInfo.wirelessDhcp = false;
                        // ipInfo.wirelessIp = "192.168.30.222";
                        // ipInfo.wirelessGateway = "192.168.30.1";
                        // ipInfo.wirelessNetmask = "255.255.255.0";
                        camera->setWirelessIPv4(
                            ipInfo.wirelessDhcp, ipInfo.wirelessIp,
                            ipInfo.wirelessGateway, ipInfo.wirelessNetmask,
                            ipInfo.wirelessDns1, ipInfo.wirelessDns2,
                            ipInfo.wirelessMac);
                    }
                    else if (event.key.keysym.sym == SDLK_j)
                    {
                        // 设置相机至AP模式
                        ipInfo.wirelessType = "ap";
                        ipInfo.wirelessNet = "5g";
                        ipInfo.wirelessSsid = ipInfo.wirelessApSsid =
                            "SL_WIFI_Camera1";
                        ipInfo.wirelessPasswd = ipInfo.wirelessApPasswd =
                            "123456789";
                        ipInfo.wirelessEnc = "wpa-wpa2 psk";
                        camera->setWireless(
                            ipInfo.wirelessType, ipInfo.wirelessNet,
                            ipInfo.wirelessApSsid, ipInfo.wirelessApPasswd,
                            ipInfo.wirelessStaSsid, ipInfo.wirelessStaPasswd,
                            ipInfo.wirelessEnc);
                    }
                    else if (event.key.keysym.sym == SDLK_k)
                    {
                        // 设置相机至STA模式
                        ipInfo.wirelessType = "sta";
                        ipInfo.wirelessNet = "5g";
                        // 需根据实际路由器来进行设置
                        ipInfo.wirelessSsid = ipInfo.wirelessStaSsid =
                            "HUAWEI_5G";
                        ipInfo.wirelessPasswd = ipInfo.wirelessStaPasswd =
                            "SL123456";
                        ipInfo.wirelessEnc = "wpa-wpa2 psk";
                        camera->setWireless(
                            ipInfo.wirelessType, ipInfo.wirelessNet,
                            ipInfo.wirelessApSsid, ipInfo.wirelessApPasswd,
                            ipInfo.wirelessStaSsid, ipInfo.wirelessStaPasswd,
                            ipInfo.wirelessEnc);
                    }
                    else if (event.key.keysym.sym == SDLK_c)
                    {
                        // 图片流拍照
                        bool saved = camera->savePic("D:/test_pic.jpeg");
                        std::cout << "saved=" << saved << std::endl;
                    }
                    // else if .....
                    break;
                case SDL_QUIT:
                    done = SDL_TRUE;
                case SDL_WINDOWEVENT:
                    if (event.window.windowID == windowID &&
                        event.window.event == SDL_WINDOWEVENT_RESIZED)
                    {
                        sdlRect.w = event.window.data1;
                        sdlRect.h = event.window.data2;
                        std::cout << "Resize window. Width=" << sdlRect.w
                                  << "height=" << sdlRect.h << std::endl;
                    }
                    else if (
                        event.window.windowID == windowID &&
                        event.window.event == SDL_WINDOWEVENT_MINIMIZED)
                    {
                        while (SDL_WaitEvent(&event))
                        {
                            if (event.window.windowID == windowID &&
                                event.window.event == SDL_WINDOWEVENT_RESTORED)
                            {
                                break;
                            }
                        }
                    }
                    break;
            }
        }

        if (done)
            break;

        l_md_data_t *frameData = nullptr;
        if (camera->getVideoFrameData(frameData) && frameData)
        {
#ifndef SLDEMO_RGB
            SDL_UpdateYUVTexture(
                sdlTexture, nullptr, reinterpret_cast<Uint8 *>(frameData->p_y),
                frameData->pitch_y, reinterpret_cast<Uint8 *>(frameData->p_u),
                frameData->pitch_u, reinterpret_cast<Uint8 *>(frameData->p_v),
                frameData->pitch_v);
#else
            SDL_UpdateTexture(
                sdlTexture, nullptr,
                reinterpret_cast<Uint8 *>(frameData->p_rgb),
                frameData->pitch_888);
#endif

            /*std::cout << "Width=" << frameData->w << ", height=" <<
               frameData->h << ", format=" << frameData->type << std::endl;*/
            getFrameFailedCount = 0;
        }

        l_sdk_dec_free_md_data(frameData);

        if (++getFrameFailedCount > 300)
        {
            done = SDL_TRUE;
            std::cerr << "Net camera disconnect!" << std::endl;
        }

        SDL_RenderClear(sdlRenderer);
        SDL_RenderCopy(sdlRenderer, sdlTexture, nullptr, &sdlRect);
        SDL_RenderPresent(sdlRenderer);
    }

    SDL_DestroyTexture(sdlTexture);
    SDL_DestroyRenderer(sdlRenderer);
    SDL_DestroyWindow(window);
    SDL_Quit();

    camera->closePicStream();
    camera->closeStream();
    camera->logout();
    sl::NetCamera::destroyNet();
    return 0;
}