#include "net_camera.h"

#include <stdio.h>

#include <iostream>
#include <regex>

#include "f801w_net_camera.h"
#include "new_net_camera.h"

namespace
{
int firstDecId = 1;

int versionFromSoftwareVer(const std::string &softwareVer)
{
    int version = 0;
    std::regex rx("(\\d+)");

    auto begin = std::sregex_iterator(softwareVer.begin(), softwareVer.end(), rx);
    auto end = std::sregex_iterator();

    for (auto it = begin; it != end; ++it)
    {
        std::smatch match = *it;
        version = version * 10 + atoi(match.str().c_str());
    }
    return version;
}
}  // namespace

namespace sl
{
std::unique_ptr<NetCamera> NetCamera::create(
    const std::string &cameraModel, int version)
{
    // C312/C313/C314/L313/L314 1.8.0版本，L116 1.7.0版本开始仅支持新版协议
    if (cameraModel.compare("M112_IPC") == 0 ||
        cameraModel.compare("M114_IPC") == 0 ||
        cameraModel.compare("M202_IPC") == 0)
        return std::make_unique<NetCamera>();
    else if (cameraModel.compare("F801W_IPC") == 0)
        return std::make_unique<F801WNetCamera>();
    else if ((cameraModel.compare("C312_IPC") == 0 ||
        cameraModel.compare("C313_IPC") == 0 ||
        cameraModel.compare("C314_IPC") == 0 ||
        cameraModel.compare("L313_IPC") == 0 ||
        cameraModel.compare("L314_IPC") == 0) && version < 180)
        return std::make_unique<NetCamera>();
    else if ((cameraModel.compare("L116_IPC") == 0) && version < 170)
        return std::make_unique<NetCamera>();

    return std::make_unique<NewNetCamera>();
}

NetCamera::NetCamera()
{
    m_streamDecId = firstDecId++;
}

CameraStreamSettings NetCamera::getCameraStreamSettings() const
{
    return m_cameraStreamSettings;
}

void NetCamera::setCameraStreamSettings(const CameraStreamSettings &settings)
{
    m_cameraStreamSettings = settings;
}

void NetCamera::initNet()
{
    l_sdk_init("");
    l_sdk_discover_open("");
    l_sdk_discover_run(true);
}

void NetCamera::destroyNet()
{
    l_sdk_discover_run(false);
    l_sdk_discover_close();
    l_sdk_quit();
}

std::vector<NetInfo> NetCamera::searchAvailableCameras()
{  // example
    //[
    //{
    //    "ip": "192.168.1.222",
    //        "discover" : {
    //        "hw_ver": "h1.2.0",
    //            "txt_enc" : "",
    //            "sw_ver" : "v1.2.0",
    //            "dev_type" : "ipc",
    //            "name" : "IPC",
    //            "port" : 80,
    //            "model" : "C313_IPC",
    //            "mac" : "00:13:09:FE:45:78",
    //            "mac_wireless" : "00:13:09:FE:45:79",
    //            "chnn_num" : 1,
    //            "md_enc" : ""
    //    },
    //        "sn": "YDFE4EFDFESHEDFR"
    //},
    //{
    //    "ip": "192.168.1.240",
    //    "discover" : {
    //        "hw_ver": "h1.3.0",
    //        "md_enc" : "",
    //        "sw_ver" : "v1.3.0",
    //        "dev_type" : "ipc",
    //        "model" : "L116_IPC",
    //        "chnn_num" : 1,
    //        "name" : "IPC",
    //        "mac" : "00:13:09:FE:45:78",
    //        "mac_wireless" : "00:13:09:FE:45:79",
    //        "port" : 80,
    //        "txt_enc" : ""
    //    },
    //    "sn": "YDFE4EFDFESHEDFR"
    //}
    //]
    char *p_res = nullptr;
    std::vector<NetInfo> infos;
    l_sdk_discover_get_devs(&p_res);
    if (!p_res)
        return infos;

    std::string res(p_res);
    l_sdk_free(p_res);

    nlohmann::json resJson = nlohmann::json::parse(res);
    // std::cout << "searchAvailableCameras:" << resJson.dump() << std::endl;
    for (size_t i = 0; i < resJson.size(); ++i)
    {
        try
        {
            NetInfo info;
            resJson[i].at("ip").get_to(info.ip);
            resJson[i].at("discover").at("model").get_to(info.model);
            resJson[i].at("discover").at("sw_ver").get_to(info.softwareVersion);
            resJson[i].at("discover").at("hw_ver").get_to(info.hardwareVersion);
            resJson[i].at("discover").at("port").get_to(info.port);
            info.version = versionFromSoftwareVer(info.softwareVersion);
            infos.push_back(info);
        }
        catch (const nlohmann::detail::exception &e)
        {
            std::cerr << "searchAvailableCameras failed, json throw an error:"
                      << e.what() << std::endl;
        }
    }

    return infos;
}

nlohmann::json NetCamera::getRequest(const nlohmann::json &requestJson)
{
    std::string request = requestJson.dump();
    char *resultStr = nullptr;
    int ret = l_sdk_request(m_id, request.c_str(), &resultStr);
    if (ret != 0 && !resultStr)
    {
        std::cout << "Get requset failed" << std::endl;
        return nlohmann::json();
    }

    nlohmann::json resJson;
    int code = -1;
    if (resultStr)
    {
        resJson = nlohmann::json::parse(resultStr);
        try
        {
            std::string cmd = requestJson.at("cmd").get<std::string>();
            // std::cout << "getRequest res=" << resJson.dump() << std::endl;
            code = resJson.at(cmd).at("code").get<int>();
        }
        catch (const nlohmann::detail::exception &e)
        {
            std::cerr << "getRequest failed, json throw an error:" << e.what()
                      << std::endl;
        }
        l_sdk_free(resultStr);
    }

    if (code != 0)
    {
        std::cout << "Get requset failed" << request << ", coed=" << code
                  << std::endl;
        return nlohmann::json();
    }

    return resJson;
}

bool NetCamera::setRequest(const nlohmann::json &requestJson)
{
    std::string request = requestJson.dump();
    char *resultStr = nullptr;
    int ret = l_sdk_request(m_id, request.c_str(), &resultStr);

    int code = -1;
    if (resultStr)
    {
        try
        {
            nlohmann::json res = nlohmann::json::parse(resultStr);
            std::string cmd = requestJson.at("cmd").get<std::string>();
            code = res.at(cmd).at("code").get<int>();
            // std::cout << res.dump() << std::endl;
        }
        catch (const nlohmann::detail::exception &e)
        {
            std::cerr << "setRequest failed, json throw an error:" << e.what()
                      << std::endl;
        }
        l_sdk_free(resultStr);
    }

    if (ret == 0 && code == 0)
        return true;

    std::cout << "Set request failed! " << request << ", coed=" << code
              << std::endl;
    return false;
}

bool NetCamera::login()
{
    nlohmann::json request = {
        {"ip", m_cameraStreamSettings.netInfo.ip},
        {"port", m_cameraStreamSettings.netInfo.port},
        {"cmd", "login"},
        {"login",
         {{"username", m_cameraStreamSettings.netInfo.username},
          {"passwd", m_cameraStreamSettings.netInfo.password}}}};

    if (l_sdk_login(&m_id, request.dump().c_str()) != 0)
    {
        return false;
    }
    std::cout << "login success! id=" << m_id << std::endl;
    return true;
}

bool NetCamera::logout()
{
    if (l_sdk_logout(m_id) != 0)
        return false;
    return true;
}

bool NetCamera::openStream()
{
    // 打开一个解码器
    // default md_fmt    MJPEG
    // default pix_fmt   YUV420P_SEPARATE
    nlohmann::json decJson;
    std::string strStreamFormat = "mjpeg";
    // 视频流格式, 切换视频流格式时, 需要重新创建解码器
    switch (m_cameraStreamSettings.streamFormat)
    {
        case StreamFormat::kMJPEG:
            decJson["md_fmt"] = "MJPEG";
            strStreamFormat = "mjpeg";
            break;
        case StreamFormat::kH264:
            decJson["md_fmt"] = "H264";
            strStreamFormat = "h264";
            break;
        default:
            break;
    }
    // 解码后数据格式
    switch (m_cameraStreamSettings.pixFormat)
    {
        case PixFormat::kYUV420P:
            decJson["pix_fmt"] = "YUV420P_SEPARATE";
            break;
        case PixFormat::kRGB888:
            decJson["pix_fmt"] = "RGB888";
            break;
        case PixFormat::kRGBA8888:
            decJson["pix_fmt"] = "RGBA8888";
            break;
        case PixFormat::kBGR888:
            decJson["pix_fmt"] = "BGR888";
            break;
        case PixFormat::kBGRA8888:
            decJson["pix_fmt"] = "BGRA8888";
            break;
        case PixFormat::kABGR8888:
            decJson["pix_fmt"] = "ABGR8888";
            break;
        case PixFormat::kARGB8888:
            decJson["pix_fmt"] = "ARGB8888";
            break;
        default:
            break;
    }
    // std::cout << "decJson=" << decJson.dump() << std::endl;
    // 多个视频流的时，需创建多个解码器，dec_id需要不同，>0即可
    int ret = l_sdk_dec_open(m_streamDecId, decJson.dump().c_str());

    if (ret != 0)
    {
        std::cout << "Create decoder failed!" << std::endl;
        return false;
    }

    ret = l_sdk_dec_bind(m_streamDecId, m_id, m_chnn, m_idx, 0);
    if (ret != 0)
    {
        l_sdk_dec_close(m_streamDecId);
        std::cout << "Bind decoder failed!" << std::endl;
        return false;
    }

    getNetCameraAllInfo();

    // h264码率存在限制
    if (m_cameraStreamSettings.streamFormat != StreamFormat::kMJPEG &&
        m_cameraStreamSettings.bitrate > 30000)
        m_cameraStreamSettings.bitrate = 30000;

    setStream(
        strStreamFormat, m_cameraStreamSettings.rcMode,
        m_cameraStreamSettings.resolution.size, m_cameraStreamSettings.quality,
        m_cameraStreamSettings.frameRate, m_cameraStreamSettings.bitrate,
        m_cameraStreamSettings.keyFrameInterval);

    // 登录成功, 后请求流
    return requestStream();
}

bool NetCamera::closeStream()
{
    l_sdk_dec_unbind(m_streamDecId);
    l_sdk_dec_close(m_streamDecId);
    nlohmann::json request = {
        {"cmd", "close_stream"},
        {"close_stream", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return setRequest(request);
}

bool NetCamera::changeStreamSettings(const CameraStreamSettings &settings)
{
    bool streamFormatChanged =
        (m_cameraStreamSettings.streamFormat != settings.streamFormat);

    m_cameraStreamSettings = settings;

    nlohmann::json decJson;
    std::string strStreamFormat = "h264";
    // 视频流格式, 切换视频流格式时, 需要重新创建解码器
    switch (m_cameraStreamSettings.streamFormat)
    {
        case StreamFormat::kMJPEG:
            decJson["md_fmt"] = "MJPEG";
            strStreamFormat = "mjpeg";
            break;
        case StreamFormat::kH264:
            decJson["md_fmt"] = "H264";
            strStreamFormat = "h264";
            break;
        default:
            break;
    }

    if (streamFormatChanged)
    {
        std::cout << "Stream format changed!" << std::endl;
        l_sdk_dec_unbind(m_streamDecId);
        l_sdk_dec_close(m_streamDecId);
    }

    // h264码率存在限制
    if (m_cameraStreamSettings.streamFormat != StreamFormat::kMJPEG &&
        m_cameraStreamSettings.bitrate > 30000)
        m_cameraStreamSettings.bitrate = 30000;

    setStream(
        strStreamFormat, m_cameraStreamSettings.rcMode,
        m_cameraStreamSettings.resolution.size, m_cameraStreamSettings.quality,
        m_cameraStreamSettings.frameRate, m_cameraStreamSettings.bitrate,
        m_cameraStreamSettings.keyFrameInterval);

    if (streamFormatChanged)
    {
        switch (m_cameraStreamSettings.pixFormat)
        {
            case PixFormat::kYUV420P:
                decJson["pix_fmt"] = "YUV420P_SEPARATE";
                break;
            case PixFormat::kRGB888:
                decJson["pix_fmt"] = "RGB888";
                break;
            case PixFormat::kRGBA8888:
                decJson["pix_fmt"] = "RGBA8888";
                break;
            case PixFormat::kBGR888:
                decJson["pix_fmt"] = "BGR888";
                break;
            case PixFormat::kBGRA8888:
                decJson["pix_fmt"] = "BGRA8888";
                break;
            case PixFormat::kABGR8888:
                decJson["pix_fmt"] = "ABGR8888";
                break;
            case PixFormat::kARGB8888:
                decJson["pix_fmt"] = "ARGB8888";
                break;
            default:
                break;
        }

        l_sdk_dec_open(m_streamDecId, decJson.dump().c_str());
        l_sdk_dec_bind(m_streamDecId, m_id, m_chnn, m_idx, 0);
    }

    return true;
}

CameraInfo NetCamera::getNetCameraAllInfo()
{
    getSupportFuncInfo();
    getRangeInfo();
    getStreamInfo();
    getPicStream();

    getControlInfo();
    getFocusRegionInfo();
    getExposureInfo();
    getWbInfo();
    getFlipMirrorStatus();
    getGammaValue();
    getDefogStatus();
    getPowerLineFrequencyStatus();
    getSystemBaseInfo();
    getIpv4Info();
    // 是否具有WIFI功能
    if (m_cameraInfo.ipInfo.enWireless)
    {
        getWirelessIpv4Info();
        getWirelessInfo();
    }
    getCACInfo();
    getDRCInfo();
    getSceneInfo();
    getWbTemperatureInfo();

    return m_cameraInfo;
}

bool NetCamera::getVideoFrameData(l_md_data_t *&frameData)
{
    if (l_sdk_dec_get_md_data(m_streamDecId, &frameData) != 0)
        return false;

    assert(nullptr != frameData);
    if (!frameData)
        return false;

    if (frameData->w != m_cameraStreamSettings.resolution.width ||
        frameData->h != m_cameraStreamSettings.resolution.height)
    {
        l_sdk_dec_free_md_data(frameData);
        frameData = nullptr;
        return false;
    }
    return true;
}

bool NetCamera::setFocus(FocusMode mode, int value, int speed)
{
    m_cameraInfo.focus.mode = mode;
    // C31X相机默认范围为-25-375
    // M系列value值无效, 置0即可
    if (m_cameraStreamSettings.netInfo.model.compare("M202_IPC") ||
        m_cameraStreamSettings.netInfo.model.compare("M114_IPC") ||
        m_cameraStreamSettings.netInfo.model.compare("M112_IPC"))
    {
        m_cameraInfo.focus.value.m_cur = 0;
    }
    else
    {
        m_cameraInfo.focus.value.m_cur = value;
    }
    m_cameraInfo.focus.speed.m_cur = speed;
    int focusMode = convertNetFocusMode(mode);
    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        m_cameraInfo.saturation.m_cur, m_cameraInfo.hue.m_cur, focusMode,
        m_cameraInfo.focus.value.m_cur,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::setFocusRegion(bool enable, int x, int y, int mode)
{
    m_cameraInfo.focus.region.enable = enable;
    m_cameraInfo.focus.region.x.m_cur = x;
    m_cameraInfo.focus.region.y.m_cur = y;
    m_cameraInfo.focus.region.mode = mode;
    nlohmann::json request = {
        {"cmd", "set_img_roi"},
        {"set_img_roi",
         {{"chnn", m_chnn},
          {"roi", enable},
          {"roi_x", x},
          {"roi_y", y},
          {"roi_mode", mode}}}};
    return setRequest(request);
}

bool NetCamera::setFocusRegionV2(
    bool enable, int x, int y, int width, int height)
{
    std::cerr << "Current camera is not support set focus region v2"
              << std::endl;
    return false;
}

bool NetCamera::getFocusRegionV2(
    bool &enable, int &x, int &y, int &width, int &height)
{
    std::cerr << "Current camera is not support get focus region v2"
              << std::endl;
    return false;
}

bool NetCamera::focusAdd()
{
    if (m_cameraInfo.focus.mode != FocusMode::kFocusManual)
        return false;

    int focusValue = 1;
    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        m_cameraInfo.saturation.m_cur, m_cameraInfo.hue.m_cur,
        convertNetFocusMode(m_cameraInfo.focus.mode), focusValue,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::focusSub()
{
    if (m_cameraInfo.focus.mode != FocusMode::kFocusManual)
        return false;

    int focusValue = -1;
    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        m_cameraInfo.saturation.m_cur, m_cameraInfo.hue.m_cur,
        convertNetFocusMode(m_cameraInfo.focus.mode), focusValue,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::getFocusState(bool &focusing)
{
    std::cerr << "Current camera is not support get focus state" << std::endl;
    return false;
}

bool NetCamera::setExposure(
    ExposureMode mode, int compensation, int gain, int shutterTime)
{
    m_cameraInfo.exposure.mode = mode;
    m_cameraInfo.exposure.bright.m_cur = compensation;
    m_cameraInfo.exposure.gain.m_cur = gain;
    m_cameraInfo.exposure.time.m_cur = shutterTime;

    setMEExposure(
        m_cameraInfo.exposure.mode, m_cameraInfo.exposure.gain.m_cur,
        m_cameraInfo.exposure.time.m_cur);

    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        m_cameraInfo.saturation.m_cur, m_cameraInfo.hue.m_cur,
        convertNetFocusMode(m_cameraInfo.focus.mode),
        m_cameraInfo.focus.value.m_cur,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::setWb(WbMode mode, int red, int green, int blue)
{
    std::string wbMode = convertNetWhiteBalanceMode(mode);
    m_cameraInfo.whiteBalance.red.m_cur = red;
    m_cameraInfo.whiteBalance.green.m_cur = green;
    m_cameraInfo.whiteBalance.blue.m_cur = blue;

    nlohmann::json request = {
        {"cmd", "set_img_awb"},
        {"set_img_awb",
         {{"chnn", m_chnn},
          {"awb", wbMode},
          {"b", m_cameraInfo.whiteBalance.blue.m_cur},
          {"gb", m_cameraInfo.whiteBalance.green.m_cur},
          {"gr", m_cameraInfo.whiteBalance.green.m_cur},
          {"r", m_cameraInfo.whiteBalance.red.m_cur}}}};
    return setRequest(request);
}

bool NetCamera::setWbTemperature(int temperature)
{
    std::cerr << "Current camera is not support set wb temperature"
              << std::endl;
    return false;
}

bool NetCamera::setZoom(
    int ezoomValue, int dzoomValue, int speed, ZoomMode mode)
{
    if (ezoomValue > m_cameraInfo.zoom.m_max)
        ezoomValue = m_cameraInfo.zoom.m_max;

    if (dzoomValue > m_cameraInfo.dzoom.m_max)
        dzoomValue = m_cameraInfo.dzoom.m_max;

    m_cameraInfo.zoom.m_cur = ezoomValue;
    if (m_cameraInfo.zoom.m_cur == m_cameraInfo.zoom.m_max)
        m_cameraInfo.dzoom.m_cur = dzoomValue;
    else
        m_cameraInfo.dzoom.m_cur = 0;

    m_cameraInfo.zoomSpeed.m_cur = speed;
    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        m_cameraInfo.saturation.m_cur, m_cameraInfo.hue.m_cur,
        convertNetFocusMode(m_cameraInfo.focus.mode),
        m_cameraInfo.focus.value.m_cur,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::setFlip(bool enable)
{
    m_cameraInfo.flip = enable;
    return setFlipMirror(m_cameraInfo.flip, m_cameraInfo.mirror);
}

bool NetCamera::setMirror(bool enable)
{
    m_cameraInfo.mirror = enable;
    return setFlipMirror(m_cameraInfo.flip, m_cameraInfo.mirror);
}

bool NetCamera::setCAC(bool enable)
{
    m_cameraInfo.cac = enable;
    nlohmann::json request = {
        {"cmd", "set_img_cac"},
        {"set_img_cac", {{"chnn", m_chnn}, {"cac", m_cameraInfo.cac}}}};
    return setRequest(request);
}

bool NetCamera::setPowerLineFrequency(bool enable)
{
    m_cameraInfo.powerLineFrequency = enable;
    nlohmann::json request = {
        {"cmd", "set_img_flick_hz"},
        {"set_img_flick_hz",
         {{"chnn", m_chnn}, {"flick", m_cameraInfo.powerLineFrequency}}}};
    return setRequest(request);
}

bool NetCamera::setMonochrome(bool enable)
{
    m_cameraInfo.monochrome = enable;
    int tempSaturation = 0;
    // 通过调节饱和度来实现黑白效果
    if (!m_cameraInfo.monochrome)
        tempSaturation = m_cameraInfo.saturation.m_cur;

    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        tempSaturation, m_cameraInfo.hue.m_cur,
        convertNetFocusMode(m_cameraInfo.focus.mode),
        m_cameraInfo.focus.value.m_cur,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::setDefog(bool enable)
{
    m_cameraInfo.defog = enable;
    nlohmann::json request = {
        {"cmd", "set_motor_demist"},
        {"set_motor_demist",
         {{"chnn", m_chnn}, {"demist", m_cameraInfo.defog}}}};
    return setRequest(request);
}

bool NetCamera::setLed(int index, int level)
{
    std::cout << "Led not support!" << std::endl;
    return false;
}

bool NetCamera::setContrastValue(int value)
{
    m_cameraInfo.contrast.m_cur = value;
    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        m_cameraInfo.saturation.m_cur, m_cameraInfo.hue.m_cur,
        convertNetFocusMode(m_cameraInfo.focus.mode),
        m_cameraInfo.focus.value.m_cur,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::setSaturationValue(int value)
{
    m_cameraInfo.saturation.m_cur = value;
    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        m_cameraInfo.saturation.m_cur, m_cameraInfo.hue.m_cur,
        convertNetFocusMode(m_cameraInfo.focus.mode),
        m_cameraInfo.focus.value.m_cur,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::setHueValue(int value)
{
    m_cameraInfo.hue.m_cur = value;
    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        m_cameraInfo.saturation.m_cur, m_cameraInfo.hue.m_cur,
        convertNetFocusMode(m_cameraInfo.focus.mode),
        m_cameraInfo.focus.value.m_cur,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::setSharpnessValue(int value)
{
    m_cameraInfo.sharpness.m_cur = value;
    return control(
        m_cameraInfo.exposure.bright.m_cur, m_cameraInfo.contrast.m_cur,
        m_cameraInfo.saturation.m_cur, m_cameraInfo.hue.m_cur,
        convertNetFocusMode(m_cameraInfo.focus.mode),
        m_cameraInfo.focus.value.m_cur,
        m_cameraInfo.zoom.m_cur + m_cameraInfo.dzoom.m_cur,
        m_cameraInfo.sharpness.m_cur);
}

bool NetCamera::setGammaValue(int value)
{
    m_cameraInfo.gamma.m_cur = value;
    nlohmann::json request = {
        {"cmd", "set_img_hdr_gamma"},
        {"set_img_hdr_gamma",
         {{"chnn", m_chnn}, {"level", m_cameraInfo.gamma.m_cur}}}};
    return setRequest(request);
}

bool NetCamera::setDRCValue(int value)
{
    m_cameraInfo.drc.m_cur = value;
    nlohmann::json request = {
        {"cmd", "set_img_drc"},
        {"set_img_drc", {{"chnn", m_chnn}, {"drc", m_cameraInfo.drc.m_cur}}}};
    return setRequest(request);
}

bool NetCamera::setScene(int mode)
{
    m_cameraInfo.sceneMode = mode;
    nlohmann::json request = {
        {"cmd", "set_img_mode"},
        {"set_img_mode", {{"chnn", m_chnn}, {"mode", m_cameraInfo.sceneMode}}}};
    return setRequest(request);
}

bool NetCamera::setSharpnessEnhancement(bool enable)
{
    std::cerr << "Current camera is not support set sharpness enhancement"
              << std::endl;
    return false;
}

bool NetCamera::setContrastEnhancementValue(int value)
{
    std::cerr << "Current camera is not support set contrast enhancement"
              << std::endl;
    return false;
}

bool NetCamera::setIpv4(
    bool dhcp, const std::string &ip, const std::string &gateway,
    const std::string &netmask, const std::string &dns1,
    const std::string &dns2, const std::string &mac)
{
    // mac地址仅对F801W生效
    nlohmann::json request = {
        {"cmd", "set_ipv4"},
        {"set_ipv4",
         {{"chnn", m_chnn},
          {"dhcp", dhcp},
          {"ip", ip},
          {"gateway", gateway},
          {"netmask", netmask},
          {"dns1", dns1},
          {"dns2", dns2},
          {"mac", mac}}}};
    return setRequest(request);
}

bool NetCamera::setWirelessIPv4(
    bool dhcp, const std::string &ip, const std::string &gateway,
    const std::string &netmask, const std::string &dns1,
    const std::string &dns2, const std::string &mac)
{
    nlohmann::json request = {
        {"cmd", "set_wireless_ipv4"},
        {"set_wireless_ipv4",
         {{"chnn", m_chnn},
          {"dhcp", dhcp},
          {"ip", ip},
          {"gateway", gateway},
          {"netmask", netmask},
          {"dns1", dns1},
          {"dns2", dns2},
          {"mac", mac}}}};
    return setRequest(request);
}

bool NetCamera::setWireless(
    const std::string &type, const std::string &net, const std::string &apSsid,
    const std::string &apPasswd, const std::string &staSsid,
    const std::string &staPasswd, const std::string &enc)
{
    // 兼容老型号处理
    bool apMode = (type.compare("AP") == 0 || type.compare("ap") == 0);
    std::string ssid = apMode ? apSsid : staSsid;
    std::string passwd = apMode ? apPasswd : staPasswd;

    nlohmann::json request = {
        {"cmd", "set_wireless"},
        {"set_wireless",
         {{"chnn", m_chnn},
          {"type", type},
          {"net", net},
          {"ssid", ssid},
          {"passwd", passwd},
          {"ap_ssid", apSsid},
          {"ap_passwd", apPasswd},
          {"sta_ssid", staSsid},
          {"sta_passwd", staPasswd},
          {"enc", enc}}}};
    return setRequest(request);
}

bool NetCamera::upgrade(const std::string &upgradeFilePath)
{
    nlohmann::json requestJson = {
        {"cmd", "upgrade"},
        {"ip", m_cameraStreamSettings.netInfo.ip},
        {"port", m_cameraStreamSettings.netInfo.port},
        {"path", upgradeFilePath},
        {"upgrade",
         {{"username", m_cameraStreamSettings.netInfo.username},
          {"passwd", m_cameraStreamSettings.netInfo.password}}}};

    std::string request = requestJson.dump();
    char *resultStr = nullptr;
    // 网络升级时, 登录Id必须设置为0
    int ret = l_sdk_request(0, request.c_str(), &resultStr);

    int code = -1;
    if (resultStr)
    {
        nlohmann::json res = nlohmann::json::parse(resultStr);
        try
        {
            std::string cmd = requestJson.at("cmd").get<std::string>();
            code = res.at(cmd).at("code").get<int>();
        }
        catch (const nlohmann::detail::exception &e)
        {
            std::cerr << "upgrade failed, json throw an error:" << e.what()
                      << std::endl;
        }
        l_sdk_free(resultStr);
    }

    if (ret == 0 && code == 0)
        return true;

    return false;
}

bool NetCamera::control(
    int bright, int contrast, int saturation, int hue, int focusMode,
    int focusValue, int zoomValue, int sharpness)
{
    nlohmann::json request = {
        {"cmd", "set_image"},
        {"set_image",
         {{"chnn", m_chnn},
          {"bright", bright},
          {"contrast", contrast},
          {"saturation", saturation},
          {"hue", hue},
          {"focus_mode", focusMode},
          {"focus_value", focusValue},
          {"ezoom_value", zoomValue},
          {"sharpness", sharpness}}}};
    return setRequest(request);
}

bool NetCamera::setStream(
    const std::string &streamFormat, const std::string &rcMode,
    const std::string &resolution, const std::string &quality, int frameRate,
    int bitrate, int keyFrameInterval)
{
    nlohmann::json request = {
        {"cmd", "set_stream"},
        {"set_stream",
         {{"chnn", m_chnn},
          {"idx", m_idx},
          {"fmt", streamFormat},
          {"rc_mode", rcMode},
          {"wh", resolution},
          {"quality", quality},
          {"frame_rate", frameRate},
          {"bitrate", bitrate},
          {"i_interval", keyFrameInterval}}}};
    std::cout << "setStream: " << request.dump() << std::endl;
    return setRequest(request);
}

bool NetCamera::requestStream()
{
    nlohmann::json request = {
        {"cmd", "open_stream"},
        {"open_stream", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return setRequest(request);
}

bool NetCamera::setMEExposure(ExposureMode mode, int gain, int shutterTime)
{
    int exposureMode = convertNetExposureMode(mode);
    nlohmann::json request = {
        {"cmd", "set_img_meexposure"},
        {"set_img_meexposure",
         {{"chnn", m_chnn},
          {"expo_type", exposureMode},
          {"shuttertime", shutterTime},
          {"a_gain", gain}}}};
    return setRequest(request);
}

bool NetCamera::setFlipMirror(bool flip, bool mirror)
{
    nlohmann::json request = {
        {"cmd", "set_img_mirror_flip"},
        {"set_img_mirror_flip",
         {{"chnn", m_chnn}, {"flip", flip}, {"mirror", mirror}}}};
    return setRequest(request);
}

void NetCamera::getSupportFuncInfo()
{
    nlohmann::json request = {{"cmd", "support"}};
    nlohmann::json res = getRequest(request);
    std::cout << "getSupportFuncInfo, " << res.dump() << std::endl;
}

void NetCamera::getControlRangeInfo()
{
    nlohmann::json info = getControlRangeInfoJson();
    try
    {
        m_cameraInfo.exposure.bright.m_def =
            info.at("default_image").at("bright").get<int>();
        m_cameraInfo.exposure.bright.m_min = info.at("default_image")
                                                 .at("range")
                                                 .at("bright")
                                                 .at("min")
                                                 .get<int>();
        m_cameraInfo.exposure.bright.m_max = info.at("default_image")
                                                 .at("range")
                                                 .at("bright")
                                                 .at("max")
                                                 .get<int>();

        m_cameraInfo.hue.m_def = info.at("default_image").at("hue").get<int>();
        m_cameraInfo.hue.m_min =
            info.at("default_image").at("range").at("hue").at("min").get<int>();
        m_cameraInfo.hue.m_max =
            info.at("default_image").at("range").at("hue").at("max").get<int>();
        m_cameraInfo.contrast.m_def =
            info.at("default_image").at("contrast").get<int>();
        m_cameraInfo.contrast.m_min = info.at("default_image")
                                          .at("range")
                                          .at("contrast")
                                          .at("min")
                                          .get<int>();
        m_cameraInfo.contrast.m_max = info.at("default_image")
                                          .at("range")
                                          .at("contrast")
                                          .at("max")
                                          .get<int>();

        m_cameraInfo.saturation.m_def =
            info.at("default_image").at("saturation").get<int>();
        m_cameraInfo.saturation.m_min = info.at("default_image")
                                            .at("range")
                                            .at("saturation")
                                            .at("min")
                                            .get<int>();
        m_cameraInfo.saturation.m_max = info.at("default_image")
                                            .at("range")
                                            .at("saturation")
                                            .at("max")
                                            .get<int>();

        m_cameraInfo.sharpness.m_def =
            info.at("default_image").at("sharpness").get<int>();
        m_cameraInfo.sharpness.m_min = info.at("default_image")
                                           .at("range")
                                           .at("sharpness")
                                           .at("min")
                                           .get<int>();
        m_cameraInfo.sharpness.m_max = info.at("default_image")
                                           .at("range")
                                           .at("sharpness")
                                           .at("max")
                                           .get<int>();

        m_cameraInfo.focus.value.m_def =
            info.at("default_image").at("focus_value").get<int>();
        m_cameraInfo.focus.value.m_min = info.at("default_image")
                                             .at("range")
                                             .at("focus_value")
                                             .at("min")
                                             .get<int>();
        m_cameraInfo.focus.value.m_max = info.at("default_image")
                                             .at("range")
                                             .at("focus_value")
                                             .at("max")
                                             .get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getControlRangeInfo failed, " << e.what() << std::endl;
    }
}

void NetCamera::getFocusRegionRangeInfo()
{
    // {
    //     "cmd": "default_img_roi",
    //     "default_img_roi": {
    //         "chnn": 0,
    //         "code": 0,
    //         "range": {
    //             "roi_big_x": {
    //                 "max": 4,
    //                 "min": 0
    //             },
    //             "roi_big_y": {
    //                 "max": 4,
    //                 "min": 0
    //             },
    //             "roi_x": {
    //                 "max": 16,
    //                 "min": 0
    //             },
    //             "roi_y": {
    //                 "max": 14,
    //                 "min": 0
    //             }
    //         },
    //         "roi": true,
    //         "roi_mode": 0,
    //         "roi_x": 8,
    //         "roi_y": 7
    //     }
    // }
    nlohmann::json info = getFocusRegionRangeInfoJson();
    try
    {
        m_cameraInfo.focus.region.x.m_def =
            info.at("default_img_roi").at("roi_x").get<int>();
        m_cameraInfo.focus.region.x.m_min = info.at("default_img_roi")
                                                .at("range")
                                                .at("roi_x")
                                                .at("min")
                                                .get<int>();
        m_cameraInfo.focus.region.x.m_max = info.at("default_img_roi")
                                                .at("range")
                                                .at("roi_x")
                                                .at("max")
                                                .get<int>();

        m_cameraInfo.focus.region.y.m_def =
            info.at("default_img_roi").at("roi_y").get<int>();
        m_cameraInfo.focus.region.y.m_min = info.at("default_img_roi")
                                                .at("range")
                                                .at("roi_y")
                                                .at("min")
                                                .get<int>();
        m_cameraInfo.focus.region.y.m_max = info.at("default_img_roi")
                                                .at("range")
                                                .at("roi_y")
                                                .at("max")
                                                .get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getFocusRegionRangeInfo failed, " << e.what()
                  << std::endl;
    }
}

void NetCamera::getMExposureRangeInfo()
{
    nlohmann::json info = getMExposureRangeInfoJson();
    try
    {
        m_cameraInfo.exposure.gain.m_def =
            info.at("default_img_meexposure").at("a_gain").get<int>();
        m_cameraInfo.exposure.gain.m_min = info.at("default_img_meexposure")
                                               .at("range")
                                               .at("a_gain")
                                               .at("min")
                                               .get<int>();
        m_cameraInfo.exposure.gain.m_max = info.at("default_img_meexposure")
                                               .at("range")
                                               .at("a_gain")
                                               .at("max")
                                               .get<int>();

        m_cameraInfo.exposure.time.m_def =
            info.at("default_img_meexposure").at("shuttertime").get<int>();
        m_cameraInfo.exposure.time.m_min = info.at("default_img_meexposure")
                                               .at("range")
                                               .at("shuttertime")
                                               .at("min")
                                               .get<int>();
        m_cameraInfo.exposure.time.m_max = info.at("default_img_meexposure")
                                               .at("range")
                                               .at("shuttertime")
                                               .at("max")
                                               .get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getMExposureRangeInfo failed, " << e.what() << std::endl;
    }
}

void NetCamera::getWbRangeInfo()
{
    nlohmann::json info = getWbRangeInfoJson();
    try
    {
        m_cameraInfo.whiteBalance.red.m_def =
            info.at("default_img_awb").at("r").get<int>();
        m_cameraInfo.whiteBalance.red.m_min =
            info.at("default_img_awb").at("range").at("r").at("min").get<int>();
        m_cameraInfo.whiteBalance.red.m_max =
            info.at("default_img_awb").at("range").at("r").at("max").get<int>();

        m_cameraInfo.whiteBalance.green.m_def =
            info.at("default_img_awb").at("gr").get<int>();
        m_cameraInfo.whiteBalance.green.m_min = info.at("default_img_awb")
                                                    .at("range")
                                                    .at("gr")
                                                    .at("min")
                                                    .get<int>();
        m_cameraInfo.whiteBalance.green.m_max = info.at("default_img_awb")
                                                    .at("range")
                                                    .at("gr")
                                                    .at("max")
                                                    .get<int>();

        m_cameraInfo.whiteBalance.blue.m_def =
            info.at("default_img_awb").at("b").get<int>();
        m_cameraInfo.whiteBalance.blue.m_min =
            info.at("default_img_awb").at("range").at("b").at("min").get<int>();
        m_cameraInfo.whiteBalance.blue.m_max =
            info.at("default_img_awb").at("range").at("b").at("max").get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getWbRangeInfoJson failed, " << e.what() << std::endl;
    }
}

void NetCamera::getGammaRangeInfo()
{
    nlohmann::json info = getGammaRangeInfoJson();
    try
    {
        m_cameraInfo.gamma.m_def =
            info.at("default_img_hdr_gamma").at("level").get<int>();
        m_cameraInfo.gamma.m_min = info.at("default_img_hdr_gamma")
                                          .at("range")
                                          .at("level")
                                          .at("min")
                                          .get<int>();
        m_cameraInfo.gamma.m_max = info.at("default_img_hdr_gamma")
                                          .at("range")
                                          .at("level")
                                          .at("max")
                                          .get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getGammaRangeInfoJson failed, " << e.what() << std::endl;
    }
}

void NetCamera::getDRCRangeInfo()
{
    nlohmann::json info = getDRCRangeInfoJson();
    try
    {
        m_cameraInfo.drc.m_def =
            info.at("default_img_drc").at("drc").get<int>();
        m_cameraInfo.drc.m_min = info.at("default_img_drc")
                                     .at("range")
                                     .at("drc")
                                     .at("min")
                                     .get<int>();
        m_cameraInfo.drc.m_max = info.at("default_img_drc")
                                     .at("range")
                                     .at("drc")
                                     .at("max")
                                     .get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getDRCRangeInfoJson failed, " << e.what() << std::endl;
    }
}

void NetCamera::getWbTemperatureRangeInfo()
{
    nlohmann::json info = getWbTemperatureRangeInfoJson();
    try
    {
        m_cameraInfo.whiteBalance.temperature.m_def =
            info.at("default_img_color_temp").at("color_temp").get<int>();
        m_cameraInfo.whiteBalance.temperature.m_min =
            info.at("default_img_color_temp")
                .at("range")
                .at("color_temp")
                .at("min")
                .get<int>();
        m_cameraInfo.whiteBalance.temperature.m_max =
            info.at("default_img_color_temp")
                .at("range")
                .at("color_temp")
                .at("max")
                .get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getWbTemperatureRangeInfo failed, " << e.what()
                  << std::endl;
    }
}

void NetCamera::getStreamInfo()
{
    nlohmann::json streamInfoJson = getStreamJson();
    if (streamInfoJson.is_null())
        return;

    std::cout << "getStreamInfo, " << streamInfoJson.dump() << std::endl;
}

void NetCamera::getRangeInfo()
{
    // nlohmann::json nameJson = getNameJson();
    // nlohmann::json ntpJson = getNtpJson();

    // nlohmann::json defaultStreamJson = getDefaultStreamJson();
    // nlohmann::json defaultIpv4Json = getDefaultIpv4Json();
    // nlohmann::json defaultWirelessJson = getDefaultWirelessJson();
    // nlohmann::json defaultWirelessIpv4Json = getDefaultWirelessIpv4Json();
    // nlohmann::json defaultNameJson = getDefaultNameJson();
    // nlohmann::json defaultNtpJson = getDefaultNtpJson();

    // 目前仅C313 1.5.0版本及L116 1.6.0版本后支持获取范围及默认值接口
    getControlRangeInfo();
    getFocusRegionRangeInfo();
    getMExposureRangeInfo();
    getWbRangeInfo();
    getGammaRangeInfo();
    getDRCRangeInfo();
    getWbTemperatureRangeInfo();

    m_cameraInfo.focus.value.m_def = 175;
    m_cameraInfo.focus.value.m_max = 375;
    m_cameraInfo.focus.value.m_min = -25;
    m_cameraInfo.focus.value.m_res = 1;

    m_cameraInfo.exposure.bright.m_def = 56;
    m_cameraInfo.exposure.bright.m_max = 250;
    m_cameraInfo.exposure.bright.m_min = 0;
    m_cameraInfo.exposure.bright.m_res = 1;

    m_cameraInfo.exposure.time.m_def = 150;
    m_cameraInfo.exposure.time.m_max = 333;
    m_cameraInfo.exposure.time.m_min = 1;
    m_cameraInfo.exposure.time.m_res = 1;

    m_cameraInfo.exposure.gain.m_def = 0;
    m_cameraInfo.exposure.gain.m_max = 33;
    m_cameraInfo.exposure.gain.m_min = 0;
    m_cameraInfo.exposure.gain.m_res = 1;

    // RGB默认值
    // C313/C312/L116  r=628, g=256, b=421
    // C314            r=661, g=256, b=372
    // M114            r=512, g=256, b=478
    // M112/M202       r=490, g=256, b=369
    m_cameraInfo.whiteBalance.red.m_def = 628;
    m_cameraInfo.whiteBalance.red.m_max = 4095;
    m_cameraInfo.whiteBalance.red.m_min = 1;
    m_cameraInfo.whiteBalance.red.m_res = 1;

    m_cameraInfo.whiteBalance.green.m_def = 256;
    m_cameraInfo.whiteBalance.green.m_max = 4095;
    m_cameraInfo.whiteBalance.green.m_min = 1;
    m_cameraInfo.whiteBalance.green.m_res = 1;

    m_cameraInfo.whiteBalance.blue.m_def = 421;
    m_cameraInfo.whiteBalance.blue.m_max = 4095;
    m_cameraInfo.whiteBalance.blue.m_min = 1;
    m_cameraInfo.whiteBalance.blue.m_res = 1;

    m_cameraInfo.hue.m_def = 50;
    m_cameraInfo.hue.m_max = 100;
    m_cameraInfo.hue.m_min = 0;
    m_cameraInfo.hue.m_res = 1;

    m_cameraInfo.saturation.m_def = 128;
    m_cameraInfo.saturation.m_max = 255;
    m_cameraInfo.saturation.m_min = 0;
    m_cameraInfo.saturation.m_res = 1;

    m_cameraInfo.contrast.m_def = 50;
    m_cameraInfo.contrast.m_max = 100;
    m_cameraInfo.contrast.m_min = 0;
    m_cameraInfo.contrast.m_res = 1;

    m_cameraInfo.sharpness.m_def = 8;
    m_cameraInfo.sharpness.m_max = 15;
    m_cameraInfo.sharpness.m_min = 0;
    m_cameraInfo.sharpness.m_res = 1;

    m_cameraInfo.gamma.m_def = 1;
    m_cameraInfo.gamma.m_min = 0;
    m_cameraInfo.gamma.m_max = 3;
    m_cameraInfo.gamma.m_res = 1;

    m_cameraInfo.drc.m_def = 100;
    m_cameraInfo.drc.m_max = 255;
    m_cameraInfo.drc.m_min = 0;
    m_cameraInfo.drc.m_res = 1;

    m_cameraInfo.zoom.m_def = 0;
    m_cameraInfo.zoom.m_min = 0;
    // M202: zoom(0-144), dzoom(0-30)
    // M112/M114: zoom(0-130), dzoom(0-50)
    if (m_cameraStreamSettings.netInfo.model.compare("M202_IPC"))
        m_cameraInfo.zoom.m_max = 144;
    else
        m_cameraInfo.zoom.m_max = 130;

    m_cameraInfo.zoom.m_res = 1;

    m_cameraInfo.dzoom.m_def = 0;
    m_cameraInfo.dzoom.m_min = 0;

    if (m_cameraStreamSettings.netInfo.model.compare("M202_IPC"))
        m_cameraInfo.dzoom.m_max = 30;
    else
        m_cameraInfo.dzoom.m_max = 50;

    m_cameraInfo.dzoom.m_res = 1;
}

void NetCamera::getControlInfo()
{
    nlohmann::json imgRecJson = getControlInfoJson();
    if (imgRecJson.is_null())
        return;

    try
    {
        int focusMode = imgRecJson.at("get_image").at("focus_mode").get<int>();
        m_cameraInfo.focus.mode = convertFromNetFocusMode(focusMode);

        m_cameraInfo.focus.value.m_cur =
            imgRecJson.at("get_image").at("focus_value").get<int>();
        m_cameraInfo.exposure.bright.m_cur =
            imgRecJson.at("get_image").at("bright").get<int>();
        m_cameraInfo.contrast.m_cur =
            imgRecJson.at("get_image").at("contrast").get<int>();
        m_cameraInfo.saturation.m_cur =
            imgRecJson.at("get_image").at("saturation").get<int>();
        m_cameraInfo.monochrome = (m_cameraInfo.saturation.m_cur == 0);
        m_cameraInfo.hue.m_cur =
            imgRecJson.at("get_image").at("hue").get<int>();
        m_cameraInfo.sharpness.m_cur =
            imgRecJson.at("get_image").at("sharpness").get<int>();
        m_cameraInfo.zoom.m_cur =
            imgRecJson.at("get_image").at("ezoom_value").get<int>();

        if (m_cameraInfo.zoom.m_cur > m_cameraInfo.zoom.m_max)
        {
            m_cameraInfo.dzoom.m_cur =
                m_cameraInfo.zoom.m_cur - m_cameraInfo.zoom.m_max;
            m_cameraInfo.zoom.m_cur = m_cameraInfo.zoom.m_max;
        }
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getControlInfo failed, json throw an error:" << e.what()
                  << std::endl;
    }
}

void NetCamera::getWirelessInfo()
{
    nlohmann::json wirelessInfoJson = getWirelessInfoJson();
    if (wirelessInfoJson.is_null())
        return;
    try
    {
        m_cameraInfo.ipInfo.wirelessType =
            wirelessInfoJson.at("wireless").at("type").get<std::string>();
        m_cameraInfo.ipInfo.wirelessNet =
            wirelessInfoJson.at("wireless").at("net").get<std::string>();
        bool apMode =
            (m_cameraInfo.ipInfo.wirelessType.compare("AP") == 0 ||
             m_cameraInfo.ipInfo.wirelessType.compare("ap") == 0);
        if (wirelessInfoJson.at("wireless").contains("ap_ssid"))
        {
            m_cameraInfo.ipInfo.wirelessApSsid = wirelessInfoJson.at("wireless")
                                                     .at("ap_ssid")
                                                     .get<std::string>();
            m_cameraInfo.ipInfo.wirelessApPasswd =
                wirelessInfoJson.at("wireless")
                    .at("ap_passwd")
                    .get<std::string>();
            m_cameraInfo.ipInfo.wirelessStaSsid =
                wirelessInfoJson.at("wireless")
                    .at("sta_ssid")
                    .get<std::string>();
            m_cameraInfo.ipInfo.wirelessStaPasswd =
                wirelessInfoJson.at("wireless")
                    .at("sta_passwd")
                    .get<std::string>();
            if (apMode)
            {
                m_cameraInfo.ipInfo.wirelessSsid =
                    m_cameraInfo.ipInfo.wirelessApSsid;
                m_cameraInfo.ipInfo.wirelessPasswd =
                    m_cameraInfo.ipInfo.wirelessApPasswd;
            }
            else
            {
                m_cameraInfo.ipInfo.wirelessSsid =
                    m_cameraInfo.ipInfo.wirelessStaSsid;
                m_cameraInfo.ipInfo.wirelessPasswd =
                    m_cameraInfo.ipInfo.wirelessStaPasswd;
            }
        }
        else
        {
            m_cameraInfo.ipInfo.wirelessSsid =
                wirelessInfoJson.at("wireless").at("ssid").get<std::string>();
            m_cameraInfo.ipInfo.wirelessPasswd =
                wirelessInfoJson.at("wireless").at("passwd").get<std::string>();
            if (apMode)
            {
                m_cameraInfo.ipInfo.wirelessApSsid =
                    m_cameraInfo.ipInfo.wirelessSsid;
                m_cameraInfo.ipInfo.wirelessApPasswd =
                    m_cameraInfo.ipInfo.wirelessPasswd;
            }
            else
            {
                m_cameraInfo.ipInfo.wirelessStaSsid =
                    m_cameraInfo.ipInfo.wirelessSsid;
                m_cameraInfo.ipInfo.wirelessStaPasswd =
                    m_cameraInfo.ipInfo.wirelessPasswd;
            }
        }
        std::string wirelessEnc =
            wirelessInfoJson.at("wireless").at("enc").get<std::string>();
    }
    catch (const std::exception &e)
    {
        std::cerr << "getWirelessInfo failed, json throw an error:" << e.what()
                  << std::endl;
    }
    std::cout << "getWirelessInfo: " << wirelessInfoJson.dump() << std::endl;
}

void NetCamera::getWirelessIpv4Info()
{
    nlohmann::json wirelessIpv4InfoJson = getWirelessIpv4InfoJson();
    if (wirelessIpv4InfoJson.is_null())
        return;
    try
    {
        m_cameraInfo.ipInfo.wirelessDhcp =
            wirelessIpv4InfoJson.at("wireless_ipv4").at("dhcp").get<bool>();
        m_cameraInfo.ipInfo.wirelessIp =
            wirelessIpv4InfoJson.at("wireless_ipv4")
                .at("ip")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessGateway =
            wirelessIpv4InfoJson.at("wireless_ipv4")
                .at("gateway")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessNetmask =
            wirelessIpv4InfoJson.at("wireless_ipv4")
                .at("netmask")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessDns1 =
            wirelessIpv4InfoJson.at("wireless_ipv4")
                .at("dns1")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessDns2 =
            wirelessIpv4InfoJson.at("wireless_ipv4")
                .at("dns2")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessMac =
            wirelessIpv4InfoJson.at("wireless_ipv4")
                .at("mac")
                .get<std::string>();
    }
    catch (const std::exception &e)
    {
        std::cerr << "getWirelessIpv4Info failed, json throw an error:"
                  << e.what() << std::endl;
    }
    std::cout << "getWirelessIpv4Info: " << wirelessIpv4InfoJson.dump()
              << std::endl;
}

void NetCamera::getExposureInfo()
{
    nlohmann::json exposureRecJson = getExposureInfoJson();
    if (exposureRecJson.is_null())
        return;
    // std::cout << "exposureRecJson: " << exposureRecJson.dump() << std::endl;
    try
    {
        int mode =
            exposureRecJson.at("img_meexposure").at("expo_type").get<int>();
        m_cameraInfo.exposure.mode = convertFromNetExposureMode(mode);
        m_cameraInfo.exposure.time.m_cur =
            exposureRecJson.at("img_meexposure").at("shuttertime").get<int>();
        m_cameraInfo.exposure.gain.m_cur =
            exposureRecJson.at("img_meexposure").at("a_gain").get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getExposureInfo failed, json throw an error:" << e.what()
                  << std::endl;
    }
}

void NetCamera::getWbMode()
{
    nlohmann::json awbRecJson = getWbModeJson();
    if (awbRecJson.is_null())
        return;
    try
    {
        std::string wbMode =
            awbRecJson.at("img_awb").at("awb").get<std::string>();
        m_cameraInfo.whiteBalance.mode = convertFromNetWhiteBalanceMode(wbMode);
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getWbMode failed, json throw an error:" << e.what()
                  << std::endl;
    }
}

void NetCamera::getWbInfo()
{
    getWbMode();
    nlohmann::json wbInfoJson = getWbInfoJson();
    if (wbInfoJson.is_null())
        return;
    // std::cout << "wbInfoJson: " << wbInfoJson.dump() << std::endl;
    try
    {
        m_cameraInfo.whiteBalance.red.m_cur =
            wbInfoJson.at("info_img_awb").at("r").get<int>();
        m_cameraInfo.whiteBalance.green.m_cur =
            wbInfoJson.at("info_img_awb").at("gb").get<int>();
        m_cameraInfo.whiteBalance.blue.m_cur =
            wbInfoJson.at("info_img_awb").at("b").get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getWbInfo failed, json throw an error:" << e.what()
                  << std::endl;
    }
}

void NetCamera::getFlipMirrorStatus()
{
    nlohmann::json MirrorFlipRecJson = getFlipMirrorStatusJson();
    if (MirrorFlipRecJson.is_null())
        return;
    try
    {
        m_cameraInfo.mirror =
            MirrorFlipRecJson.at("img_mirror_flip").at("mirror").get<bool>();
        m_cameraInfo.flip =
            MirrorFlipRecJson.at("img_mirror_flip").at("flip").get<bool>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getFlipMirrorStatus failed, json throw an error:"
                  << e.what() << std::endl;
    }
}

void NetCamera::getPowerLineFrequencyStatus()
{
    nlohmann::json flickRecJson = getPowerLineFrequencyStatusJson();
    if (flickRecJson.is_null())
        return;

    try
    {
        m_cameraInfo.powerLineFrequency =
            flickRecJson.at("img_flick_hz").at("flick").get<bool>();
    }
    catch (const std::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NetCamera::getGammaValue()
{
    nlohmann::json hdrRecJson = getGammaValueJson();
    if (hdrRecJson.is_null())
        return;
    try
    {
        m_cameraInfo.gamma.m_cur =
            hdrRecJson.at("img_hdr_gamma").at("level").get<int>();
    }
    catch (const std::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NetCamera::getDefogStatus()
{
    nlohmann::json defogRecJson = getDefogStatusJson();
    if (defogRecJson.is_null())
        return;
    // std::cout << "defogRecJson: " << defogRecJson.dump() << std::endl;
    try
    {
        m_cameraInfo.defog =
            defogRecJson.at("motor_demist").at("demist").get<bool>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getDefogStatus failed, json throw an error:" << e.what()
                  << std::endl;
    }
}

void NetCamera::getIpv4Info()
{
    nlohmann::json ipv4InfoJson = getIpv4InfoJson();
    if (ipv4InfoJson.is_null())
        return;

    try
    {
        m_cameraInfo.ipInfo.dhcp =
            ipv4InfoJson.at("ipv4").at("dhcp").get<bool>();
        m_cameraInfo.ipInfo.ip =
            ipv4InfoJson.at("ipv4").at("ip").get<std::string>();
        m_cameraInfo.ipInfo.gateway =
            ipv4InfoJson.at("ipv4").at("gateway").get<std::string>();
        m_cameraInfo.ipInfo.netmask =
            ipv4InfoJson.at("ipv4").at("netmask").get<std::string>();
        m_cameraInfo.ipInfo.dns1 =
            ipv4InfoJson.at("ipv4").at("dns1").get<std::string>();
        m_cameraInfo.ipInfo.dns2 =
            ipv4InfoJson.at("ipv4").at("dns2").get<std::string>();
        if (ipv4InfoJson.at("ipv4").contains("mac"))
            m_cameraInfo.ipInfo.mac =
                ipv4InfoJson.at("ipv4").at("mac").get<std::string>(); 
    }
    catch (const std::exception &e)
    {
        std::cerr << "getIpv4Info failed, json throw an error:" << e.what()
                  << std::endl;
    }
    std::cout << "getIpv4Info: " << ipv4InfoJson.dump() << std::endl;
}

void NetCamera::getSystemBaseInfo()
{
    nlohmann::json systemBaseJson = getSystemBaseJson();
    if (systemBaseJson.is_null())
        return;

    try
    {
        m_cameraInfo.model = systemBaseJson.at("system").at("model");

        if (systemBaseJson.at("system").contains("model_suffix"))
            m_cameraInfo.modelSuffix =
                systemBaseJson.at("system").at("model_suffix");

        if (systemBaseJson.at("system").contains("wifi"))
            m_cameraInfo.ipInfo.enWireless =
                systemBaseJson.at("system").at("wifi").get<bool>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getSystemBaseInfo failed, json throw an error:"
                  << e.what() << std::endl;
    }

    std::cout << "getSystemBaseInfo: " << systemBaseJson.dump() << std::endl;
}

bool NetCamera::getUpgradeStatus(std::string &status, double &percentage)
{
    nlohmann::json upgradeStatusJson = getUpgradeStatusJson();
    if (upgradeStatusJson.is_null())
        return false;

    // std::cout << upgradeStatusJson.dump() << std::endl;
    try
    {
        status = upgradeStatusJson.at("status_upgrade")
                     .at("status")
                     .get<std::string>();
        percentage = upgradeStatusJson.at("status_upgrade")
                         .at("percentage")
                         .get<double>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getUpgradeStatus failed, json throw an error:" << e.what()
                  << std::endl;
        return false;
    }
    return true;
}

bool NetCamera::getPicStream()
{
    nlohmann::json request = {
        {"cmd", "stream_pic"},
        {"stream_pic", {{"chnn", 0}, {"idx", m_picIdx}}}};
    nlohmann::json result = getRequest(request);
    if (!result.is_null())
        std::cout << "getPicStream: " << result.dump() << std::endl;

    return true;
}

bool NetCamera::setPicStream(const CameraPicStreamSettings &settings)
{
    m_cameraPicStreamSettings = settings;

    std::string picFormat;
    if (m_cameraPicStreamSettings.picStreamFormat == PicStreamFormat::kJPEG)
        picFormat = "jpeg";

    nlohmann::json request = {
        {"cmd", "set_stream_pic"},
        {"set_stream_pic",
         {{"chnn", m_chnn},
          {"idx", m_picIdx},
          {"fmt", picFormat},
          {"wh", m_cameraPicStreamSettings.resolution.size},
          {"quality", m_cameraPicStreamSettings.quality},
          {"interval_ms", m_cameraPicStreamSettings.interval}}}};
    std::cout << "setPicStream: " << request.dump() << std::endl;
    return setRequest(request);
}

bool NetCamera::requestPicStream()
{
    nlohmann::json request = {
        {"cmd", "open_stream"},
        {"open_stream", {{"chnn", m_chnn}, {"idx", m_picIdx}}}};
    return setRequest(request);
}

bool NetCamera::closePicStream()
{
    nlohmann::json request = {
        {"cmd", "close_stream"},
        {"close_stream", {{"chnn", m_chnn}, {"idx", m_picIdx}}}};
    return setRequest(request);
}

bool NetCamera::savePic(const std::string &filePath)
{
    l_md_buf_t *buffer = nullptr;
    int ret = l_sdk_md_get(m_id, m_chnn, m_picIdx, 0, &buffer);
    if (ret != 0)
        return false;

    bool saved = false;
    FILE *file = fopen(filePath.c_str(), "wb");
    if (file)
    {
        fwrite(
            buffer->p_buf + buffer->start, 1, buffer->end - buffer->start,
            file);
        fclose(file);
        saved = true;
    }

    l_sdk_md_buf_sub(buffer);

    return saved;
}

int NetCamera::convertNetFocusMode(FocusMode mode)
{
    switch (mode)
    {
        case FocusMode::kFocusManual:
            return 1;
        case FocusMode::kFocusAuto:
            return 0;
        case FocusMode::kFocusOnce:
            return 2;
        default:
            return -1;
    }
    return -1;
}

int NetCamera::convertNetExposureMode(ExposureMode mode)
{
    switch (mode)
    {
        case ExposureMode::kExpManual:
            return 1;
        case ExposureMode::kExpAuto:
            return 0;
        default:
            return -1;
    }
    return -1;
}

std::string NetCamera::convertNetWhiteBalanceMode(WbMode mode)
{
    std::string wbMode = "auto";
    switch (mode)
    {
        case WbMode::kWBManual:
            wbMode = "manual";
            break;
        case WbMode::kWBAuto:
            wbMode = "auto";
            break;
        default:
            break;
    }
    return wbMode;
}

int NetCamera::convertNetWhiteBalanceMode2(WbMode mode)
{
    switch (mode)
    {
        case WbMode::kWBManual:
            return 1;
        case WbMode::kWBAuto:
            return 0;
        default:
            return -1;
    }
    return -1;
}

FocusMode NetCamera::convertFromNetFocusMode(int mode)
{
    if (mode == 0)
        return FocusMode::kFocusAuto;
    else if (mode == 1)
        return FocusMode::kFocusManual;
    else if (mode == 2)
        return FocusMode::kFocusOnce;

    return FocusMode::kFocusAuto;
}

ExposureMode NetCamera::convertFromNetExposureMode(int mode)
{
    if (mode == 0)
        return ExposureMode::kExpAuto;
    else if (mode == 1)
        return ExposureMode::kExpManual;

    return ExposureMode::kExpAuto;
}

WbMode NetCamera::convertFromNetWhiteBalanceMode(std::string mode)
{
    if (mode.compare("manual") == 0)
        return WbMode::kWBAuto;
    else if (mode.compare("auto") == 0)
        return WbMode::kWBManual;

    return WbMode::kWBAuto;
}

WbMode NetCamera::convertFromNetWhiteBalanceMode(int mode)
{
    if (mode == 0)
        return WbMode::kWBAuto;
    else if (mode == 1)
        return WbMode::kWBManual;

    return WbMode::kWBAuto;
}

void NetCamera::getFocusRegionInfo()
{
    nlohmann::json focusRegionJson = getFocusRegionInfoJson();
    if (focusRegionJson.is_null())
        return;
    // std::cout << "focusRegionJson: " << focusRegionJson.dump() << std::endl;
    try
    {
        m_cameraInfo.focus.region.enable =
            focusRegionJson.at("img_roi").at("roi").get<bool>();
        m_cameraInfo.focus.region.mode =
            focusRegionJson.at("img_roi").at("roi_mode").get<int>();
        m_cameraInfo.focus.region.x.m_cur =
            focusRegionJson.at("img_roi").at("roi_x").get<int>();
        m_cameraInfo.focus.region.y.m_cur =
            focusRegionJson.at("img_roi").at("roi_y").get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getFocusRegionInfo failed, json throw an error:"
                  << e.what() << std::endl;
    }
}

void NetCamera::getCACInfo()
{
    nlohmann::json cacInfo = getCACInfoJson();
    if (cacInfo.is_null())
        return;

    try
    {
        m_cameraInfo.cac = cacInfo.at("img_cac").at("cac").get<bool>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getCACInfo failed, throw an error:" << e.what()
                  << std::endl;
    }
}

void NetCamera::getSceneInfo()
{
    nlohmann::json sceneInfo = getSceneInfoJson();
    if (sceneInfo.is_null())
        return;

    try
    {
        m_cameraInfo.sceneMode = sceneInfo.at("img_mode").at("mode").get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getSceneInfo failed, throw an error:" << e.what()
                  << std::endl;
    }
}

void NetCamera::getWbTemperatureInfo()
{
    nlohmann::json info = getWbTemperatureInfoJson();
    if (info.is_null())
        return;

    try
    {
        m_cameraInfo.whiteBalance.temperature.m_cur =
            info.at("img_color_temp").at("color_temp").get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getWbTemperatureInfo failed, throw an error:" << e.what()
                  << std::endl;
    }
}

void NetCamera::getDRCInfo()
{
    nlohmann::json drcInfo = getDRCInfoJson();
    if (drcInfo.is_null())
        return;

    try
    {
        m_cameraInfo.drc.m_cur = drcInfo.at("img_drc").at("drc").get<int>();
    }
    catch (const nlohmann::detail::exception &e)
    {
        std::cerr << "getDRCInfo failed, throw an error:" << e.what()
                  << std::endl;
    }
}

nlohmann::json NetCamera::getControlInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_image"},
        {"get_image", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getWirelessInfoJson()
{
    nlohmann::json request = {{"cmd", "wireless"}};
    return getRequest(request);
}

nlohmann::json NetCamera::getWirelessIpv4InfoJson()
{
    nlohmann::json request = {{"cmd", "wireless_ipv4"}};
    return getRequest(request);
}

nlohmann::json NetCamera::getExposureInfoJson()
{
    nlohmann::json request = {
        {"cmd", "img_meexposure"}, {"img_meexposure", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getWbModeJson()
{
    nlohmann::json request = {
        {"cmd", "img_awb"}, {"img_awb", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getWbInfoJson()
{
    nlohmann::json request = {
        {"cmd", "info_img_awb"}, {"info_img_awb", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getFlipMirrorStatusJson()
{
    nlohmann::json request = {
        {"cmd", "img_mirror_flip"}, {"img_mirror_flip", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getPowerLineFrequencyStatusJson()
{
    nlohmann::json request = {
        {"cmd", "img_flick_hz"}, {"img_flick_hz", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getGammaValueJson()
{
    nlohmann::json request = {
        {"cmd", "img_hdr_gamma"},
        {"img_hdr_gamma", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getDefogStatusJson()
{
    nlohmann::json request = {
        {"cmd", "motor_demist"}, {"motor_demist", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getIpv4InfoJson()
{
    nlohmann::json request = {{"cmd", "ipv4"}};
    return getRequest(request);
}

nlohmann::json NetCamera::getSystemBaseJson()
{
    nlohmann::json request = {{"cmd", "system"}};
    return getRequest(request);
}

nlohmann::json NetCamera::getUpgradeStatusJson()
{
    nlohmann::json requestJson = {
        {"cmd", "status_upgrade"},
        {"ip", m_cameraStreamSettings.netInfo.ip},
        {"port", m_cameraStreamSettings.netInfo.port}};

    std::string request = requestJson.dump();
    char *resultStr = nullptr;
    // 网络升级时, 登录Id必须设置为0
    int ret = l_sdk_request(0, request.c_str(), &resultStr);
    if (ret != 0 && !resultStr)
    {
        std::cout << "getUpgradeStatusJson failed" << std::endl;
        return nlohmann::json();
    }

    nlohmann::json resJson;
    int code = -1;
    if (resultStr)
    {
        resJson = nlohmann::json::parse(resultStr);
        try
        {
            std::string cmd = requestJson.at("cmd").get<std::string>();
            code = resJson.at(cmd).at("code").get<int>();
        }
        catch (const nlohmann::detail::exception &e)
        {
            std::cerr << "getUpgradeStatusJson failed, json throw an error:"
                      << e.what() << std::endl;
        }
        l_sdk_free(resultStr);
    }

    if (code != 0)
    {
        std::cout << "getUpgradeStatusJson failed" << std::endl;
        return nlohmann::json();
    }

    return resJson;
}

nlohmann::json NetCamera::getFocusRegionInfoJson()
{
    nlohmann::json request = {
        {"cmd", "img_roi"}, {"img_roi", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getCACInfoJson()
{
    nlohmann::json request = {
        {"cmd", "img_cac"}, {"img_cac", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getDRCInfoJson()
{
    nlohmann::json request = {
        {"cmd", "img_drc"}, {"img_drc", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getSceneInfoJson()
{
    nlohmann::json request = {
        {"cmd", "img_mode"}, {"img_mode", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getWbTemperatureInfoJson()
{
    nlohmann::json request = {
        {"cmd", "img_color_temp"},
        {"img_color_temp", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getNameJson()
{
    nlohmann::json request = {{"cmd", "name"}};
    return getRequest(request);
}

nlohmann::json NetCamera::getNtpJson()
{
    nlohmann::json request = {{"cmd", "ntp"}};
    return getRequest(request);
}

nlohmann::json NetCamera::getStreamJson()
{
    nlohmann::json request = {
        {"cmd", "stream"}, {"stream", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getDefaultStreamJson()
{
    nlohmann::json request = {
        {"cmd", "default_stream"},
        {"default_stream", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getDefaultIpv4Json()
{
    nlohmann::json request = {
        {"cmd", "default_ipv4"},
        {"default_ipv4", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getDefaultWirelessJson()
{
    nlohmann::json request = {
        {"cmd", "default_wireless"},
        {"default_wireless", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getDefaultWirelessIpv4Json()
{
    nlohmann::json request = {
        {"cmd", "default_wireless_ipv4"},
        {"default_wireless_ipv4", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getDefaultNameJson()
{
    nlohmann::json request = {{"cmd", "default_name"}};
    return getRequest(request);
}

nlohmann::json NetCamera::getDefaultNtpJson()
{
    nlohmann::json request = {{"cmd", "default_ntp"}};
    return getRequest(request);
}

nlohmann::json NetCamera::getControlRangeInfoJson()
{
    nlohmann::json request = {
        {"cmd", "default_image"}, {"default_image", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getExposureRangeInfoJson()
{
    nlohmann::json request = {
        {"cmd", "default_img_exposure"},
        {"default_img_exposure", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getFocusRegionRangeInfoJson()
{
    nlohmann::json request = {
        {"cmd", "default_img_roi"}, {"default_img_roi", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getMExposureRangeInfoJson()
{
    nlohmann::json request = {
        {"cmd", "default_img_meexposure"},
        {"default_img_meexposure", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getWbRangeInfoJson()
{
    nlohmann::json request = {
        {"cmd", "default_img_awb"}, {"default_img_awb", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getGammaRangeInfoJson()
{
    nlohmann::json request = {
        {"cmd", "default_img_hdr_gamma"},
        {"default_img_hdr_gamma", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getDRCRangeInfoJson()
{
    nlohmann::json request = {
        {"cmd", "default_img_drc"}, {"default_img_drc", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NetCamera::getWbTemperatureRangeInfoJson()
{
    nlohmann::json request = {
        {"cmd", "default_img_color_temp"},
        {"default_img_color_temp", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

}  // namespace sl
