#ifndef SL_NET_CAMERA_H_
#define SL_NET_CAMERA_H_

#include <memory>
#include <nlohmann/json.hpp>
#include <string>
#include <vector>

#include "camera_info.h"
#include "l_sdk.h"

namespace sl
{
class NetCamera
{
public:
    static std::unique_ptr<NetCamera> create(
        const std::string &cameraModel, int version);
    NetCamera();
    virtual ~NetCamera() = default;

    CameraStreamSettings getCameraStreamSettings() const;
    void setCameraStreamSettings(const CameraStreamSettings &settings);

    // 初始化SDK环境
    static void initNet();
    // 销毁SDK环境
    static void destroyNet();
    // 搜索同一网段下的所有网络相机
    static std::vector<NetInfo> searchAvailableCameras();

    // 登录至指定IP相机
    bool login();
    // 退出登录
    bool logout();

    // 开始取流
    bool openStream();
    // 停止取流
    bool closeStream();
    // 变更视频流参数, 分辨率、帧率等
    bool changeStreamSettings(const CameraStreamSettings &settings);
    // 获取一帧视频帧
    bool getVideoFrameData(l_md_data_t *&frameData);

    // 获取当前已登录相机的各参数
    virtual CameraInfo getNetCameraAllInfo();

    // 设置聚焦
    // mode - 模式
    // value 手动聚焦值
    // speed 聚焦速度 (暂未使用)
    virtual bool setFocus(FocusMode mode, int value, int speed = 3);
    // 设置聚焦区域
    // enable - 是否启用
    // x x轴位置 (小框, 0-16; 大框, 0-4)
    // y y轴位置 (小框, 0-14; 大框, 0-4)
    // mode 聚焦区域大小(0-小 1-大 目前仅使用小)
    virtual bool setFocusRegion(bool enable, int x, int y, int mode = 0);
    // 设置聚焦区域, 可自由设置聚焦区域位置及大小
    // enable 是否启用
    // x      区域左上角x坐标
    // y      区域左上角y坐标
    // width  区域宽度
    // height 区域高度
    virtual bool setFocusRegionV2(
        bool enable, int x, int y, int width, int height);
    virtual bool getFocusRegionV2(
        bool &enable, int &x, int &y, int &width, int &height);
    // M系列独有功能，增加/减少1点聚焦值
    virtual bool focusAdd();
    virtual bool focusSub();

    // 获取当前聚焦状态
    // focusing true-正在聚焦, false-聚焦完成
    virtual bool getFocusState(bool &focusing);

    // exposureMode 曝光模式
    // compensation 目标亮度
    // gain 增益
    // shutterTime 快门
    virtual bool setExposure(
        ExposureMode mode, int compensation, int gain, int shutterTime);
    // 设置白平衡参数
    // wbMode 白平衡模式
    // red 红色
    // green 绿色
    // blue 蓝色
    virtual bool setWb(WbMode mode, int red, int green, int blue);
    // 设置白平衡色温
    // 色温和 RGB值是互斥关系，不可同时设置。修改色温，相机内部实现为调节RGB分量
    virtual bool setWbTemperature(int temperature);
    // 设置变倍
    // ezoomValue - 光学变倍值
    // dzoomValue - 数字变倍值
    // speed  变倍速率  0-低  1-中  2-高
    // zoomMode ZoomAbsolute 绝对变倍模式，ezoom值控制变倍位置；
    //          ZoomRelative 相对变倍模式，当ezoom值为0-停止变倍，1-缩小，-1-放大
    virtual bool setZoom(
        int ezoomValue, int dzoomValue, int speed,
        ZoomMode mode = ZoomMode::ZoomAbsolute);
    // 垂直翻转
    virtual bool setFlip(bool enable);
    // 水平翻转
    virtual bool setMirror(bool enable);
    // 消紫边
    virtual bool setCAC(bool enable);
    // 抗频闪
    // enable true-启用
    // 注意: 启用后, 如原帧率为1080p 60fps / 4k 30fps,
    // 应调用changeStreamSettings修改帧率至1080p 50fps / 4k 25fps
    // 否则可能出现帧率异常的情况
    virtual bool setPowerLineFrequency(bool enable);
    // 黑白
    virtual bool setMonochrome(bool enable);
    // 加热除雾
    virtual bool setDefog(bool enable);
    // 设置补光灯
    // index - 补光灯分块序号
    // level - 亮度等级
    virtual bool setLed(int index, int level);

    // 对比度
    virtual bool setContrastValue(int value);
    // 饱和度
    virtual bool setSaturationValue(int value);
    // 色度
    virtual bool setHueValue(int value);
    // 锐度
    virtual bool setSharpnessValue(int value);
    // Gamma
    virtual bool setGammaValue(int value);
    // 消反光
    virtual bool setDRCValue(int value);
    // 场景切换, 0-生物, 1-工业, 2-金相, 3-单筒 (仅L116支持，支持工业/金相/单筒)
    virtual bool setScene(int mode);
    // 锐度增强
    virtual bool setSharpnessEnhancement(bool enable);
    // 对比度增强
    virtual bool setContrastEnhancementValue(int value);

    // 设置相机有线IP相关信息
    // dhcp     是否启用DHCP, 有线DHCP暂不可用，默认为false
    // ip       相机新IP
    // gateway  网关
    // netmask  子网掩码
    // dns      dns地址
    // mac      MAC地址
    // 如在拉流状态, 修改IP后视频流将会断开
    virtual bool setIpv4(
        bool dhcp, const std::string &ip, const std::string &gateway,
        const std::string &netmask, const std::string &dns1,
        const std::string &dns2, const std::string &mac);

    // 设置相机无线IP相关信息
    // dhcp 是否启用DHCP
    // ip 相机新IP
    // gateway 网关
    // netmask 子网掩码
    // dns dns地址
    // mac MAC地址, 仅支持单播MAC, 即第一个字节最后一位不可为1
    virtual bool setWirelessIPv4(
        bool dhcp, const std::string &ip, const std::string &gateway,
        const std::string &netmask, const std::string &dns1,
        const std::string &dns2, const std::string &mac);

    // 设置相机无线相关信息
    // type        ap/sta 无线模式
    // net         2.4g/5g 无线频段, 目前仅支持5g
    // apSsid      AP模式下，相机AP名称
    // apPasswd    相机AP密码
    // staSsid     STA模式下，路由器无线名称
    // staPasswd   STA模式下，路由器无线密码
    // enc         wpa-wpa2 psk/wpa2 psk/psk/none
    //             AP模式下，为相机AP的加密方式;
    //             STA模式下，为路由器WIFI的加密方式
    virtual bool setWireless(
        const std::string &type, const std::string &net,
        const std::string &apSsid, const std::string &apPasswd,
        const std::string &staSsid, const std::string &staPasswd,
        const std::string &enc);
    // 相机升级功能
    // upgradeFilePath 相机升级文件绝对路径
    // 升级文件名称规范 例: C313Net - c313_update.lpk
    virtual bool upgrade(const std::string &upgradeFilePath);
    virtual bool getUpgradeStatus(std::string &status, double &percentage);

    // 获取当前图片流参数，包括范围等信息
    virtual bool getPicStream();
    // 设置图片流接口
    virtual bool setPicStream(const CameraPicStreamSettings &settings);
    // 开启图片流
    virtual bool requestPicStream();
    // 关闭图片流
    virtual bool closePicStream();
    // 图片流开启后才可拍照
    // 保存图片到指定路径下，附带文件后缀名，目前仅支持jpeg
    virtual bool savePic(const std::string &filePath);

    // Helper func
    int convertNetFocusMode(FocusMode mode);
    int convertNetExposureMode(ExposureMode mode);
    std::string convertNetWhiteBalanceMode(WbMode mode);
    int convertNetWhiteBalanceMode2(WbMode mode);

    FocusMode convertFromNetFocusMode(int mode);
    ExposureMode convertFromNetExposureMode(int mode);
    WbMode convertFromNetWhiteBalanceMode(std::string mode);
    WbMode convertFromNetWhiteBalanceMode(int mode);

protected:
    nlohmann::json getRequest(const nlohmann::json &requestJson);
    bool setRequest(const nlohmann::json &requestJson);

    // 调节视频流接口
    // streamFormat     视频流格式
    // rcMode           码率控制模式  vbr-可变码率 cbr-恒定码率
    // resolution       分辨率
    // quality          视频质量  highest/higher/high/meddle/low/lower/lowest
    // 对应新版协议能力集中的数字     6      5      4    3     2    1     0
    // 但仍使用字符串发送, 以便兼容老型号
    // frameRate        帧率
    // bitrate          码率
    // keyFrameInterval 关键帧间隔  对MJPEG格式无效
    bool setStream(
        const std::string &streamFormat, const std::string &rcMode,
        const std::string &resolution, const std::string &quality,
        int frameRate, int bitrate, int keyFrameInterval);
    // 请求视频流
    bool requestStream();

    // 控制接口
    // bright     目标亮度(曝光模式为自动时可调)
    // contrast   对比度
    // saturation 饱和度
    // hue        色度
    // gamma      伽马值
    // focusMode  聚焦模式
    // focusValue 聚焦值
    // zoomValue  变倍值
    // sharpness  锐度
    // 注意点: M系列focusValue无效，均设置为0即可
    // Zoom值是光学变倍与数字变倍之和, 光学变倍值为最大后, 才会启用数字变倍
    bool control(
        int bright, int contrast, int saturation, int hue, int focusMode,
        int focusValue, int zoomValue, int sharpness);
    // 设置曝光参数
    // mode 曝光模式
    // 增益和快门仅在手动模式下可调, control接口下目标亮度仅在自动模式下可调
    // gain 增益
    // shutterTime 快门
    bool setMEExposure(ExposureMode mode, int gain, int shutterTime);
    bool setFlipMirror(bool flip, bool mirror);

    void getSupportFuncInfo();
    void getControlRangeInfo();
    void getFocusRegionRangeInfo();
    void getMExposureRangeInfo();
    void getWbRangeInfo();
    void getGammaRangeInfo();
    void getDRCRangeInfo();
    void getWbTemperatureRangeInfo();
    // 获取当前视频流及支持的格式，部分相机支持
    void getStreamInfo();

    // 老型号手动设置各参数范围
    virtual void getRangeInfo();
    // 以下接口用于json转化, 并将参数保存至m_cameraInfo中
    virtual void getWirelessInfo();
    virtual void getWirelessIpv4Info();
    void getFlipMirrorStatus();
    virtual void getControlInfo();
    void getExposureInfo();
    void getWbMode();
    void getWbInfo();
    void getPowerLineFrequencyStatus();
    void getGammaValue();
    virtual void getDefogStatus();
    virtual void getIpv4Info();
    void getSystemBaseInfo();
    void getFocusRegionInfo();
    void getDRCInfo();
    void getCACInfo();
    virtual void getSceneInfo();
    void getWbTemperatureInfo();

    // 控制相关信息 部分参数包含其具体范围
    virtual nlohmann::json getControlInfoJson();
    // 无线ssid, passwd等信息
    virtual nlohmann::json getWirelessInfoJson();
    // 无线ip, gateway, dns等信息
    virtual nlohmann::json getWirelessIpv4InfoJson();
    // 曝光相关信息 曝光模式 快门值 增益 目标亮度
    virtual nlohmann::json getExposureInfoJson();
    // 白平衡模式
    virtual nlohmann::json getWbModeJson();
    // 白平衡 红绿蓝值
    virtual nlohmann::json getWbInfoJson();
    // 镜像翻转状态信息
    virtual nlohmann::json getFlipMirrorStatusJson();
    // 抗频闪状态
    virtual nlohmann::json getPowerLineFrequencyStatusJson();
    // gamma值
    virtual nlohmann::json getGammaValueJson();
    // 除雾状态
    virtual nlohmann::json getDefogStatusJson();
    // 有线IP相关信息
    virtual nlohmann::json getIpv4InfoJson();
    // 相机系统基本信息, 型号, 软硬件版本号等
    virtual nlohmann::json getSystemBaseJson();
    // 获取当前升级进度信息
    virtual nlohmann::json getUpgradeStatusJson();
    // 聚焦区域相关信息
    virtual nlohmann::json getFocusRegionInfoJson();
    // 去紫边
    virtual nlohmann::json getCACInfoJson();
    // 消反光
    virtual nlohmann::json getDRCInfoJson();
    // 场景
    virtual nlohmann::json getSceneInfoJson();
    // 色温
    virtual nlohmann::json getWbTemperatureInfoJson();

    // 名称, 目前都为IPC
    virtual nlohmann::json getNameJson();
    virtual nlohmann::json getNtpJson();
    virtual nlohmann::json getStreamJson();

    // 获取默认值相关信息
    virtual nlohmann::json getDefaultStreamJson();
    virtual nlohmann::json getDefaultIpv4Json();
    virtual nlohmann::json getDefaultWirelessJson();
    virtual nlohmann::json getDefaultWirelessIpv4Json();
    virtual nlohmann::json getDefaultNameJson();
    virtual nlohmann::json getDefaultNtpJson();

    virtual nlohmann::json getControlRangeInfoJson();
    virtual nlohmann::json getFocusRegionRangeInfoJson();
    virtual nlohmann::json getMExposureRangeInfoJson();
    virtual nlohmann::json getWbRangeInfoJson();
    virtual nlohmann::json getGammaRangeInfoJson();
    virtual nlohmann::json getDRCRangeInfoJson();
    virtual nlohmann::json getWbTemperatureRangeInfoJson();
    // 暂不使用此接口，使用default_img中获取的值
    virtual nlohmann::json getExposureRangeInfoJson();

protected:
    int m_id = 0;
    // 以下参数保持为0即可
    int m_idx = 0; // 主码流
    int m_chnn = 0;
    
    // 解码器ID，从1开始
    int m_streamDecId = 1;

    int m_picIdx = 64; // 图片流

    CameraStreamSettings m_cameraStreamSettings;
    CameraInfo m_cameraInfo;
    CameraPicStreamSettings m_cameraPicStreamSettings;
};

}  // namespace sl

#endif  // SL_NET_CAMERA_H_
