#include "new_net_camera.h"

#include <iostream>

#define RESET "\033[0m"
#define BLACK "\033[30m" /* Black */
#define RED "\033[31m"   /* Red */

namespace sl
{
CameraInfo NewNetCamera::getNetCameraAllInfo()
{
    getSystemBaseInfo();
    getStreamInfo();
    getStreamAbility();
    getPicAbility();
    getImageAbility();
    getMotorAbility();
    getLedAbility();

    getImageValueInfo();
    getImageBoolInfo();
    getFocusInfo();
    getZoomInfo();
    getFocusRegionInfo();
    getFocusRegionInfoV2();
    getExposureInfo();
    getWbInfo();
    getLedInfo();
    getIpv4Info();
    // M系列具有除雾功能
    // getDefogStatus();
    getPicStream();

    // 是否具有WIFI功能
    if (m_cameraInfo.ipInfo.enWireless)
    {
        getWirelessIpv4Info();
        getWirelessInfo();
    }

    return m_cameraInfo;
}

bool NewNetCamera::setFocus(FocusMode mode, int value, int speed)
{
    m_cameraInfo.focus.mode = mode;
    m_cameraInfo.focus.value.m_cur = value;
    m_cameraInfo.focus.speed.m_cur = value;

    int focusMode = convertNetFocusMode(mode);
    nlohmann::json request = {
        {"cmd", "set_focus"},
        {"set_focus",
         {{"chnn", m_chnn},
          {"focus_mode", focusMode},
          {"focus_value", m_cameraInfo.focus.value.m_cur},
          {"focus_speed", m_cameraInfo.focus.speed.m_cur}}}};
    return setRequest(request);
}

bool NewNetCamera::setFocusRegion(bool enable, int x, int y, int mode)
{
    m_cameraInfo.focus.region.enable = enable;
    m_cameraInfo.focus.region.x.m_cur = x;
    m_cameraInfo.focus.region.y.m_cur = y;
    m_cameraInfo.focus.region.mode = mode;
    nlohmann::json request = {
        {"cmd", "set_roi"},
        {"set_roi",
         {{"chnn", m_chnn},
          {"roi", m_cameraInfo.focus.region.enable},
          {"roi_x", m_cameraInfo.focus.region.x.m_cur},
          {"roi_y", m_cameraInfo.focus.region.y.m_cur},
          {"roi_mode", mode}}}};
    return setRequest(request);
}

bool NewNetCamera::setFocusRegionV2(
    bool enable, int x, int y, int width, int height)
{
    m_cameraInfo.focus.regionV2.enable = enable;
    m_cameraInfo.focus.regionV2.x = x;
    m_cameraInfo.focus.regionV2.y = y;
    m_cameraInfo.focus.regionV2.width = width;
    m_cameraInfo.focus.regionV2.height = height;

    nlohmann::json request = {
        {"cmd", "set_focus_region"},
        {"set_focus_region",
         {{"chnn", m_chnn},
          {"roi", m_cameraInfo.focus.regionV2.enable},
          {"roi_x", m_cameraInfo.focus.regionV2.x},
          {"roi_y", m_cameraInfo.focus.regionV2.y},
          {"roi_width", m_cameraInfo.focus.regionV2.width},
          {"roi_height", m_cameraInfo.focus.regionV2.height}}}};
    return setRequest(request);
}

bool NewNetCamera::getFocusRegionV2(
    bool &enable, int &x, int &y, int &width, int &height)
{
    getFocusRegionInfoV2();
    enable = m_cameraInfo.focus.regionV2.enable;
    x = m_cameraInfo.focus.regionV2.x;
    y = m_cameraInfo.focus.regionV2.y;
    width = m_cameraInfo.focus.regionV2.width;
    height = m_cameraInfo.focus.regionV2.height;
    return true;
}

bool NewNetCamera::getFocusState(bool &focusing)
{
    nlohmann::json request = {
        {"cmd", "get_focus_state"},
        {"get_focus_state", {{"chnn", m_chnn}, {"idx", m_idx}}}};

    nlohmann::json resJson = getRequest(request);
    if (resJson.is_null())
        return false;
    try
    {
        // std::cout << "resJson: " << resJson.dump() << std::endl;
        focusing = resJson.at("get_focus_state").at("focusing").get<bool>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "getFocusState throw an error:" << e.what() << std::endl;
        return false;
    }

    return true;
}

bool NewNetCamera::focusAdd() 
{
    return setFocusRelative(1);
}

bool NewNetCamera::focusSub()
{
    return setFocusRelative(-1);
}

bool NewNetCamera::setExposure(
    ExposureMode mode, int compensation, int gain, int shutterTime)
{
    m_cameraInfo.exposure.mode = mode;
    m_cameraInfo.exposure.bright.m_cur = compensation;
    m_cameraInfo.exposure.gain.m_cur = gain;
    m_cameraInfo.exposure.time.m_cur = shutterTime;

    int exposureMode = convertNetExposureMode(mode);
    nlohmann::json request = {
        {"cmd", "set_exposure"},
        {"set_exposure",
         {{"chnn", m_chnn},
          {"expo_type", exposureMode},
          {"compensation", m_cameraInfo.exposure.bright.m_cur},
          {"shuttertime", m_cameraInfo.exposure.time.m_cur},
          {"a_gain", m_cameraInfo.exposure.gain.m_cur}}}};
    return setRequest(request);
}

bool NewNetCamera::setWb(WbMode mode, int red, int green, int blue)
{
    m_cameraInfo.whiteBalance.mode = mode;
    m_cameraInfo.whiteBalance.red.m_cur = red;
    m_cameraInfo.whiteBalance.green.m_cur = green;
    m_cameraInfo.whiteBalance.blue.m_cur = blue;

    int wbMode = convertNetWhiteBalanceMode2(mode);
    nlohmann::json request = {
        {"cmd", "set_white_balance"},
        {"set_white_balance",
         {{"chnn", m_chnn},
          {"awb", wbMode},
          {"b", m_cameraInfo.whiteBalance.blue.m_cur},
          {"gb", m_cameraInfo.whiteBalance.green.m_cur},
          {"gr", m_cameraInfo.whiteBalance.green.m_cur},
          {"r", m_cameraInfo.whiteBalance.red.m_cur}}}};
    return setRequest(request);
}

bool NewNetCamera::setWbTemperature(int temperature)
{
    m_cameraInfo.whiteBalance.temperature.m_cur = temperature;

    int wbMode = convertNetWhiteBalanceMode2(m_cameraInfo.whiteBalance.mode);
    nlohmann::json request = {
        {"cmd", "set_white_balance"},
        {"set_white_balance",
         {{"chnn", m_chnn},
          {"awb", wbMode},
          {"b", m_cameraInfo.whiteBalance.blue.m_cur},
          {"gb", m_cameraInfo.whiteBalance.green.m_cur},
          {"gr", m_cameraInfo.whiteBalance.green.m_cur},
          {"r", m_cameraInfo.whiteBalance.red.m_cur},
          {"use_temperature", true},
          {"color_temperature", m_cameraInfo.whiteBalance.temperature.m_cur}}}};
    return setRequest(request);
}

bool NewNetCamera::setZoom(
    int ezoomValue, int dzoomValue, int speed, ZoomMode mode)
{
    if (ezoomValue > m_cameraInfo.zoom.m_max)
        ezoomValue = m_cameraInfo.zoom.m_max;

    if (dzoomValue > m_cameraInfo.dzoom.m_max)
        dzoomValue = m_cameraInfo.dzoom.m_max;

    m_cameraInfo.zoom.m_cur = ezoomValue;
    m_cameraInfo.dzoom.m_cur = dzoomValue;
    m_cameraInfo.zoomSpeed.m_cur = speed;
    nlohmann::json request = {
        {"cmd", "set_zoom"},
        {"set_zoom",
         {{"chnn", m_chnn},
          {"ezoom_value", m_cameraInfo.zoom.m_cur},
          {"dzoom_value", m_cameraInfo.dzoom.m_cur},
          {"ezoom_speed", m_cameraInfo.zoomSpeed.m_cur},
          {"ezoom_mode", static_cast<int>(mode)}}}};
    return setRequest(request);
}

bool NewNetCamera::setFlip(bool enable)
{
    m_cameraInfo.flip = enable;
    return setImageBool(collectImageBoolSettings());
}

bool NewNetCamera::setMirror(bool enable)
{
    m_cameraInfo.mirror = enable;
    return setImageBool(collectImageBoolSettings());
}

bool NewNetCamera::setCAC(bool enable)
{
    m_cameraInfo.cac = enable;
    return setImageBool(collectImageBoolSettings());
}

bool NewNetCamera::setPowerLineFrequency(bool enable)
{
    m_cameraInfo.powerLineFrequency = enable;
    return setImageBool(collectImageBoolSettings());
}

bool NewNetCamera::setMonochrome(bool enable)
{
    m_cameraInfo.monochrome = enable;
    return setImageBool(collectImageBoolSettings());
}

bool NewNetCamera::setLed(int partIndex, int level)
{
    for (int i = 0; i < m_cameraInfo.led.ledPartVec.size(); ++i)
    {
        if (m_cameraInfo.led.ledPartVec[i].index == partIndex)
        {
            m_cameraInfo.led.ledPartVec[i].level = level;
            // a3xx version < 120
            nlohmann::json request = {
                {"cmd", "set_led"},
                {"set_led",
                 {{"chnn", m_chnn},
                  {"index", partIndex},
                  {"level", m_cameraInfo.led.ledPartVec[i].level}}}};
            return setRequest(request);
        }
    }

    return setLeds(
        m_cameraInfo.led.ledPartVec[0].level,
        m_cameraInfo.led.ledPartVec[1].level,
        m_cameraInfo.led.ledPartVec[2].level,
        m_cameraInfo.led.ledPartVec[3].level);
}

bool NewNetCamera::setContrastValue(int value)
{
    m_cameraInfo.contrast.m_cur = value;
    return setImageValue(collectImageValueSettings());
}

bool NewNetCamera::setSaturationValue(int value)
{
    m_cameraInfo.saturation.m_cur = value;
    return setImageValue(collectImageValueSettings());
}

bool NewNetCamera::setHueValue(int value)
{
    m_cameraInfo.hue.m_cur = value;
    return setImageValue(collectImageValueSettings());
}

bool NewNetCamera::setSharpnessValue(int value)
{
    m_cameraInfo.sharpness.m_cur = value;
    return setImageValue(collectImageValueSettings());
}

bool NewNetCamera::setGammaValue(int value)
{
    m_cameraInfo.gamma.m_cur = value;
    return setImageValue(collectImageValueSettings());
}

bool NewNetCamera::setDRCValue(int value)
{
    m_cameraInfo.drc.m_cur = value;
    return setImageValue(collectImageValueSettings());
}

bool NewNetCamera::setDefog(bool enable)
{
    m_cameraInfo.defog = enable;
    nlohmann::json request = {
        {"cmd", "set_motor_demist"},
        {"set_motor_demist",
         {{"chnn", m_chnn}, {"demist", m_cameraInfo.defog}}}};
    return setRequest(request);
}

bool NewNetCamera::setScene(int mode)
{
    m_cameraInfo.sceneMode = mode;
    nlohmann::json request = {
        {"cmd", "set_scene_mode"},
        {"set_scene_mode", {{"chnn", m_chnn}, {"mode", mode}}}};
    return setRequest(request);
}

bool NewNetCamera::setSharpnessEnhancement(bool enable)
{
    m_cameraInfo.sharpnessEnhancement = enable;
    return setImageBool(collectImageBoolSettings());
}

bool NewNetCamera::setContrastEnhancementValue(int value)
{
    m_cameraInfo.contrastEnhancement.m_cur = value;
    return setImageValue(collectImageValueSettings());
}

bool NewNetCamera::getPicStream()
{
    nlohmann::json request = {
        {"cmd", "get_stream_pic"},
        {"get_stream_pic", {{"chnn", 0}, {"idx", m_picIdx}}}};
    nlohmann::json result = getRequest(request);
    if (!result.is_null())
        std::cout << RED << "getPicStream: " << result.dump() << RESET
                  << std::endl;

    return true;
}

bool NewNetCamera::setImageValue(const ImageValueSettings &settings)
{
    nlohmann::json request = {
        {"cmd", "set_image_value"},
        {"set_image_value",
         {{"chnn", m_chnn},
          {"contrast", settings.contrast},
          {"saturation", settings.saturation},
          {"hue", settings.hue},
          {"sharpness", settings.sharpness},
          {"gamma", settings.gamma},
          {"drc", settings.drc},
          {"contrast_enhance", settings.contrastEnhancement}}}};
    return setRequest(request);
}

bool NewNetCamera::setImageBool(const ImageBoolSettings &settings)
{
    nlohmann::json request = {
        {"cmd", "set_image_bool"},
        {"set_image_bool",
         {{"chnn", m_chnn},
          {"flip", settings.flip},
          {"mirror", settings.mirror},
          {"cac", settings.cac},
          {"flick", settings.flick},
          {"monochrome", settings.monochrome},
          {"enhance", settings.sharpnessEnhancement}}}};
    return setRequest(request);
}

bool NewNetCamera::setLeds(int up, int left, int right, int down)
{
    nlohmann::json request = {
        {"cmd", "set_leds"},
        {"set_leds",
         {
             {"chnn", m_chnn},
             {"led0", up},
             {"led1", left},
             {"led2", right},
             {"led3", down},
         }}};
    return setRequest(request);
}

bool NewNetCamera::setFocusRelative(int relative)
{
    nlohmann::json request = {
        {"cmd", "set_focus_relative"},
        {"set_focus_relative", {{"chnn", m_chnn}, {"relative", relative}}}};
    return setRequest(request);
}

ImageBoolSettings NewNetCamera::collectImageBoolSettings() const
{
    ImageBoolSettings settings;
    settings.flip = m_cameraInfo.flip;
    settings.mirror = m_cameraInfo.mirror;
    settings.cac = m_cameraInfo.cac;
    settings.flick = m_cameraInfo.powerLineFrequency;
    settings.monochrome = m_cameraInfo.monochrome;
    settings.sharpnessEnhancement = m_cameraInfo.sharpnessEnhancement;
    return settings;
}

ImageValueSettings NewNetCamera::collectImageValueSettings() const
{
    ImageValueSettings settings;
    settings.contrast = m_cameraInfo.contrast.m_cur;
    settings.saturation = m_cameraInfo.saturation.m_cur;
    settings.hue = m_cameraInfo.hue.m_cur;
    settings.sharpness = m_cameraInfo.sharpness.m_cur;
    settings.gamma = m_cameraInfo.gamma.m_cur;
    settings.drc = m_cameraInfo.drc.m_cur;
    settings.contrastEnhancement = m_cameraInfo.contrastEnhancement.m_cur;
    return settings;
}

void NewNetCamera::getImageValueInfo()
{
    nlohmann::json imgRecJson = getImageValueInfoJson();
    if (imgRecJson.is_null())
        return;
    try
    {
        std::cout << "imgRecJson: " << imgRecJson.dump() << std::endl;
        m_cameraInfo.contrast.m_cur =
            imgRecJson.at("get_image_value").at("contrast").get<int>();
        m_cameraInfo.saturation.m_cur =
            imgRecJson.at("get_image_value").at("saturation").get<int>();
        m_cameraInfo.hue.m_cur =
            imgRecJson.at("get_image_value").at("hue").get<int>();
        m_cameraInfo.sharpness.m_cur =
            imgRecJson.at("get_image_value").at("sharpness").get<int>();
        m_cameraInfo.gamma.m_cur =
            imgRecJson.at("get_image_value").at("gamma").get<int>();
        m_cameraInfo.drc.m_cur =
            imgRecJson.at("get_image_value").at("drc").get<int>();

        // 新增字段
        if (imgRecJson.at("get_image_value").contains("contrast_enhance"))
            m_cameraInfo.contrastEnhancement.m_cur = imgRecJson.at("get_image_value")
                                                  .at("contrast_enhance")
                                                  .get<int>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getImageBoolInfo()
{
    nlohmann::json imgRecJson = getImageBoolInfoJson();
    if (imgRecJson.is_null())
        return;
    try
    {
        std::cout << "imgRecJson: " << imgRecJson.dump() << std::endl;
        m_cameraInfo.mirror =
            imgRecJson.at("get_image_bool").at("mirror").get<bool>();
        m_cameraInfo.flip =
            imgRecJson.at("get_image_bool").at("flip").get<bool>();
        m_cameraInfo.powerLineFrequency =
            imgRecJson.at("get_image_bool").at("flick").get<bool>();
        m_cameraInfo.cac =
            imgRecJson.at("get_image_bool").at("cac").get<bool>();
        m_cameraInfo.monochrome =
            imgRecJson.at("get_image_bool").at("monochrome").get<bool>();

        // 新增字段
        if (imgRecJson.at("get_image_bool").contains("enhance"))
            m_cameraInfo.sharpnessEnhancement =
                imgRecJson.at("get_image_bool").at("enhance").get<bool>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getExposureInfo()
{
    nlohmann::json exposureRecJson = getExposureInfoJson();
    if (exposureRecJson.is_null())
        return;
    try
    {
        std::cout << "exposureRecJson: " << exposureRecJson.dump() << std::endl;
        m_cameraInfo.exposure.mode =
            (ExposureMode)exposureRecJson.at("get_exposure")
                .at("expo_type")
                .get<int>();
        m_cameraInfo.exposure.bright.m_cur =
            exposureRecJson.at("get_exposure").at("compensation").get<int>();
        m_cameraInfo.exposure.time.m_cur =
            exposureRecJson.at("get_exposure").at("shuttertime").get<int>();
        m_cameraInfo.exposure.gain.m_cur =
            exposureRecJson.at("get_exposure").at("a_gain").get<int>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getWbInfo()
{
    nlohmann::json wbInfoJson = getWbInfoJson();
    if (wbInfoJson.is_null())
        return;
    try
    {
        std::cout << "wbInfoJson: " << wbInfoJson.dump() << std::endl;
        m_cameraInfo.whiteBalance.mode =
            (WbMode)wbInfoJson.at("get_white_balance").at("awb").get<int>();
        m_cameraInfo.whiteBalance.red.m_cur =
            wbInfoJson.at("get_white_balance").at("r").get<int>();
        m_cameraInfo.whiteBalance.green.m_cur =
            wbInfoJson.at("get_white_balance").at("gb").get<int>();
        m_cameraInfo.whiteBalance.blue.m_cur =
            wbInfoJson.at("get_white_balance").at("b").get<int>();

        // 新增字段
        if (wbInfoJson.at("get_white_balance").contains("color_temperature"))
        {
            m_cameraInfo.whiteBalance.temperature.m_cur =
                wbInfoJson.at("get_white_balance")
                    .at("color_temperature")
                    .get<int>();
        }
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getFocusInfo()
{
    nlohmann::json focusJson = getFocusInfoJson();
    if (focusJson.is_null())
        return;
    try
    {
        std::cout << "focusJson: " << focusJson.dump() << std::endl;
        m_cameraInfo.focus.mode =
            (FocusMode)focusJson.at("get_focus").at("focus_mode").get<int>();
        m_cameraInfo.focus.value.m_cur =
            focusJson.at("get_focus").at("focus_value").get<int>();
        m_cameraInfo.focus.speed.m_cur =
            focusJson.at("get_focus").at("focus_speed").get<int>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getFocusRegionInfo()
{
    nlohmann::json focusRegionJson = getFocusRegionInfoJson();
    if (focusRegionJson.is_null())
        return;
    try
    {
        std::cout << "focusRegionJson: " << focusRegionJson.dump() << std::endl;
        m_cameraInfo.focus.region.enable =
            focusRegionJson.at("get_roi").at("roi").get<bool>();
        m_cameraInfo.focus.region.mode =
            focusRegionJson.at("get_roi").at("roi_mode").get<int>();
        m_cameraInfo.focus.region.x.m_cur =
            focusRegionJson.at("get_roi").at("roi_x").get<int>();
        m_cameraInfo.focus.region.y.m_cur =
            focusRegionJson.at("get_roi").at("roi_y").get<int>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getFocusRegionInfoV2()
{
    nlohmann::json focusRegionJson = getFocusRegionInfoJsonV2();
    if (focusRegionJson.is_null())
        return;
    try
    {
        std::cout << "focusRegionJsonV2: " << focusRegionJson.dump()
                  << std::endl;
        m_cameraInfo.focus.regionV2.enable =
            focusRegionJson.at("get_focus_region").at("roi").get<bool>();
        m_cameraInfo.focus.regionV2.x =
            focusRegionJson.at("get_focus_region").at("roi_x").get<int>();
        m_cameraInfo.focus.regionV2.y =
            focusRegionJson.at("get_focus_region").at("roi_y").get<int>();
        m_cameraInfo.focus.regionV2.width =
            focusRegionJson.at("get_focus_region").at("roi_width").get<int>();
        m_cameraInfo.focus.regionV2.height =
            focusRegionJson.at("get_focus_region").at("roi_height").get<int>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getZoomInfo()
{
    nlohmann::json zoomJson = getZoomInfoJson();
    if (zoomJson.is_null())
        return;
    try
    {
        std::cout << "zoomJson: " << zoomJson.dump() << std::endl;
        m_cameraInfo.zoom.m_cur =
            zoomJson.at("get_zoom").at("ezoom_value").get<int>();
        m_cameraInfo.dzoom.m_cur =
            zoomJson.at("get_zoom").at("dzoom_value").get<int>();
        m_cameraInfo.dzoom.m_cur =
            zoomJson.at("get_zoom").at("ezoom_speed").get<int>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getLedInfo()
{
    nlohmann::json ledJson = getLedInfoJson();
    if (ledJson.is_null())
        return;
    try
    {
        std::cout << "ledJson: " << ledJson.dump() << std::endl;
        nlohmann::json infoJson = ledJson.at("get_led").at("led_info");
        int infoNum = infoJson.size();
        for (int i = 0; i < infoNum; ++i)
        {
            LedPartInfo info;
            infoJson[i].at("index").get_to(info.index);
            infoJson[i].at("level").get_to(info.level);
            if (infoJson[i].contains("enable"))
            {
                info.enable = infoJson[i].at("enable").get<int>();
            }
            m_cameraInfo.led.ledPartVec.push_back(info);
        }
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getIpv4Info()
{
    nlohmann::json ipv4InfoJson = getIpv4InfoJson();
    if (ipv4InfoJson.is_null())
        return;
    try
    {
        m_cameraInfo.ipInfo.dhcp =
            ipv4InfoJson.at("get_ipv4").at("dhcp").get<bool>();
        m_cameraInfo.ipInfo.ip =
            ipv4InfoJson.at("get_ipv4").at("ip").get<std::string>();
        m_cameraInfo.ipInfo.gateway =
            ipv4InfoJson.at("get_ipv4").at("gateway").get<std::string>();
        m_cameraInfo.ipInfo.netmask =
            ipv4InfoJson.at("get_ipv4").at("netmask").get<std::string>();
        m_cameraInfo.ipInfo.dns1 =
            ipv4InfoJson.at("get_ipv4").at("dns1").get<std::string>();
        m_cameraInfo.ipInfo.dns2 =
            ipv4InfoJson.at("get_ipv4").at("dns2").get<std::string>();
    }
    catch (const std::exception &e)
    {
        std::cerr << "getIpv4Info failed, json throw an error:" << e.what()
                  << std::endl;
    }
    std::cout << "getIpv4Info: " << ipv4InfoJson.dump() << std::endl;
}

void NewNetCamera::getWirelessInfo()
{
    nlohmann::json wirelessInfoJson = getWirelessInfoJson();
    if (wirelessInfoJson.is_null())
        return;
    try
    {
        m_cameraInfo.ipInfo.wirelessType =
            wirelessInfoJson.at("get_wireless").at("type").get<std::string>();
        m_cameraInfo.ipInfo.wirelessNet =
            wirelessInfoJson.at("get_wireless").at("net").get<std::string>();
        bool apMode =
            (m_cameraInfo.ipInfo.wirelessType.compare("AP") == 0 ||
             m_cameraInfo.ipInfo.wirelessType.compare("ap") == 0);
        if (wirelessInfoJson.at("get_wireless").contains("ap_ssid"))
        {
            m_cameraInfo.ipInfo.wirelessApSsid =
                wirelessInfoJson.at("get_wireless")
                    .at("ap_ssid")
                    .get<std::string>();
            m_cameraInfo.ipInfo.wirelessApPasswd =
                wirelessInfoJson.at("get_wireless")
                    .at("ap_passwd")
                    .get<std::string>();
            m_cameraInfo.ipInfo.wirelessStaSsid =
                wirelessInfoJson.at("get_wireless")
                    .at("sta_ssid")
                    .get<std::string>();
            m_cameraInfo.ipInfo.wirelessStaPasswd =
                wirelessInfoJson.at("get_wireless")
                    .at("sta_passwd")
                    .get<std::string>();
            if (apMode)
            {
                m_cameraInfo.ipInfo.wirelessSsid =
                    m_cameraInfo.ipInfo.wirelessApSsid;
                m_cameraInfo.ipInfo.wirelessPasswd =
                    m_cameraInfo.ipInfo.wirelessApPasswd;
            }
            else
            {
                m_cameraInfo.ipInfo.wirelessSsid =
                    m_cameraInfo.ipInfo.wirelessStaSsid;
                m_cameraInfo.ipInfo.wirelessPasswd =
                    m_cameraInfo.ipInfo.wirelessStaPasswd;
            }
        }
        else
        {
            m_cameraInfo.ipInfo.wirelessSsid =
                wirelessInfoJson.at("get_wireless")
                    .at("ssid")
                    .get<std::string>();
            m_cameraInfo.ipInfo.wirelessPasswd =
                wirelessInfoJson.at("get_wireless")
                    .at("passwd")
                    .get<std::string>();
            if (apMode)
            {
                m_cameraInfo.ipInfo.wirelessApSsid =
                    m_cameraInfo.ipInfo.wirelessSsid;
                m_cameraInfo.ipInfo.wirelessApPasswd =
                    m_cameraInfo.ipInfo.wirelessPasswd;
            }
            else
            {
                m_cameraInfo.ipInfo.wirelessStaSsid =
                    m_cameraInfo.ipInfo.wirelessSsid;
                m_cameraInfo.ipInfo.wirelessStaPasswd =
                    m_cameraInfo.ipInfo.wirelessPasswd;
            }
        }
        std::string wirelessEnc =
            wirelessInfoJson.at("get_wireless").at("enc").get<std::string>();
    }
    catch (const std::exception &e)
    {
        std::cerr << "getWirelessInfo failed, json throw an error:" << e.what()
                  << std::endl;
    }
    std::cout << "getWirelessInfo: " << wirelessInfoJson.dump() << std::endl;
}

void NewNetCamera::getWirelessIpv4Info()
{
    nlohmann::json wirelessIpv4InfoJson = getWirelessIpv4InfoJson();
    if (wirelessIpv4InfoJson.is_null())
        return;
    try
    {
        m_cameraInfo.ipInfo.wirelessDhcp =
            wirelessIpv4InfoJson.at("get_wireless_ipv4").at("dhcp").get<bool>();
        m_cameraInfo.ipInfo.wirelessIp =
            wirelessIpv4InfoJson.at("get_wireless_ipv4")
                .at("ip")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessGateway =
            wirelessIpv4InfoJson.at("get_wireless_ipv4")
                .at("gateway")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessNetmask =
            wirelessIpv4InfoJson.at("get_wireless_ipv4")
                .at("netmask")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessDns1 =
            wirelessIpv4InfoJson.at("get_wireless_ipv4")
                .at("dns1")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessDns2 =
            wirelessIpv4InfoJson.at("get_wireless_ipv4")
                .at("dns2")
                .get<std::string>();
        m_cameraInfo.ipInfo.wirelessMac =
            wirelessIpv4InfoJson.at("get_wireless_ipv4")
                .at("mac")
                .get<std::string>();
    }
    catch (const std::exception &e)
    {
        std::cerr << "getWirelessIpv4Info failed, json throw an error:"
                  << e.what() << std::endl;
    }
    std::cout << "getWirelessIpv4Info: " << wirelessIpv4InfoJson.dump()
              << std::endl;
}

void NewNetCamera::getDefogStatus()
{
    nlohmann::json resJson = getDefogStatusJson();
    if (resJson.is_null())
        return;

    try
    {
        m_cameraInfo.defog =
            resJson.at("get_motor_demist").at("demist").get<bool>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "getDefogStatus throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getSceneInfo()
{
    nlohmann::json resJson = getSceneInfoJson();

    try
    {
        m_cameraInfo.sceneMode =
            resJson.at("get_scene_mode").at("mode").get<int>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "getDefogStatus throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getStreamAbility()
{
    nlohmann::json streamAbilityJson = getStreamAbilityJson();
    if (streamAbilityJson.is_null())
        return;
    std::cout << "streamAbilityJson: " << streamAbilityJson.dump() << std::endl;
}

void NewNetCamera::getPicAbility()
{
    nlohmann::json picAbilityJson = getPicAbilityJson();
    if (picAbilityJson.is_null())
        return;

    std::cout << RED << "picAbilityJson: " << picAbilityJson.dump() << RESET
              << std::endl;
}

void NewNetCamera::getImageAbility()
{
    try
    {
        nlohmann::json imageAbilityJson =
            getImageAbilityJson().at("get_image_ability");
        if (imageAbilityJson.is_null())
            return;
        // std::cout << "imageAbilityJson: " << imageAbilityJson.dump()
        //          << std::endl;

        m_cameraInfo.exposure.bright.m_min =
            imageAbilityJson.at("compensation").at("min").get<int>();
        m_cameraInfo.exposure.bright.m_max =
            imageAbilityJson.at("compensation").at("max").get<int>();
        m_cameraInfo.exposure.bright.m_def =
            imageAbilityJson.at("compensation").at("def").get<int>();
        m_cameraInfo.exposure.gain.m_min =
            imageAbilityJson.at("a_gain").at("min").get<int>();
        m_cameraInfo.exposure.gain.m_max =
            imageAbilityJson.at("a_gain").at("max").get<int>();
        m_cameraInfo.exposure.gain.m_def =
            imageAbilityJson.at("a_gain").at("def").get<int>();
        m_cameraInfo.exposure.time.m_min =
            imageAbilityJson.at("shuttertime").at("min").get<int>();
        m_cameraInfo.exposure.time.m_max =
            imageAbilityJson.at("shuttertime").at("max").get<int>();
        m_cameraInfo.exposure.time.m_def =
            imageAbilityJson.at("shuttertime").at("def").get<int>();
        m_cameraInfo.contrast.m_min =
            imageAbilityJson.at("contrast").at("min").get<int>();
        m_cameraInfo.contrast.m_max =
            imageAbilityJson.at("contrast").at("max").get<int>();
        m_cameraInfo.contrast.m_def =
            imageAbilityJson.at("contrast").at("def").get<int>();
        m_cameraInfo.saturation.m_min =
            imageAbilityJson.at("saturation").at("min").get<int>();
        m_cameraInfo.saturation.m_max =
            imageAbilityJson.at("saturation").at("max").get<int>();
        m_cameraInfo.saturation.m_def =
            imageAbilityJson.at("saturation").at("def").get<int>();
        m_cameraInfo.hue.m_min =
            imageAbilityJson.at("hue").at("min").get<int>();
        m_cameraInfo.hue.m_max =
            imageAbilityJson.at("hue").at("max").get<int>();
        m_cameraInfo.hue.m_def =
            imageAbilityJson.at("hue").at("def").get<int>();
        m_cameraInfo.sharpness.m_min =
            imageAbilityJson.at("sharpness").at("min").get<int>();
        m_cameraInfo.sharpness.m_max =
            imageAbilityJson.at("sharpness").at("max").get<int>();
        m_cameraInfo.sharpness.m_def =
            imageAbilityJson.at("sharpness").at("def").get<int>();
        m_cameraInfo.gamma.m_min =
            imageAbilityJson.at("gamma").at("min").get<int>();
        m_cameraInfo.gamma.m_max =
            imageAbilityJson.at("gamma").at("max").get<int>();
        m_cameraInfo.gamma.m_def =
            imageAbilityJson.at("gamma").at("def").get<int>();
        m_cameraInfo.whiteBalance.red.m_min =
            imageAbilityJson.at("r").at("min").get<int>();
        m_cameraInfo.whiteBalance.red.m_max =
            imageAbilityJson.at("r").at("max").get<int>();
        m_cameraInfo.whiteBalance.red.m_def =
            imageAbilityJson.at("r").at("def").get<int>();
        m_cameraInfo.whiteBalance.green.m_min =
            imageAbilityJson.at("gr").at("min").get<int>();
        m_cameraInfo.whiteBalance.green.m_max =
            imageAbilityJson.at("gr").at("max").get<int>();
        m_cameraInfo.whiteBalance.green.m_def =
            imageAbilityJson.at("gr").at("def").get<int>();
        m_cameraInfo.whiteBalance.blue.m_min =
            imageAbilityJson.at("b").at("min").get<int>();
        m_cameraInfo.whiteBalance.blue.m_max =
            imageAbilityJson.at("b").at("max").get<int>();
        m_cameraInfo.whiteBalance.blue.m_def =
            imageAbilityJson.at("b").at("def").get<int>();
        m_cameraInfo.drc.m_min =
            imageAbilityJson.at("drc").at("min").get<int>();
        m_cameraInfo.drc.m_max =
            imageAbilityJson.at("drc").at("max").get<int>();
        m_cameraInfo.drc.m_def =
            imageAbilityJson.at("drc").at("def").get<int>();

        // 后续新增字段
        if (imageAbilityJson.contains("contrast_enhance"))
        {
            m_cameraInfo.contrastEnhancement.m_min =
                imageAbilityJson.at("contrast_enhance").at("min").get<int>();
            m_cameraInfo.contrastEnhancement.m_max =
                imageAbilityJson.at("contrast_enhance").at("max").get<int>();
            m_cameraInfo.contrastEnhancement.m_def =
                imageAbilityJson.at("contrast_enhance").at("def").get<int>();
        }

        if (imageAbilityJson.contains("color_temperature"))
        {
            m_cameraInfo.whiteBalance.temperature.m_min =
                imageAbilityJson.at("color_temperature").at("min").get<int>();
            m_cameraInfo.whiteBalance.temperature.m_max =
                imageAbilityJson.at("color_temperature").at("max").get<int>();
            m_cameraInfo.whiteBalance.temperature.m_def =
                imageAbilityJson.at("color_temperature").at("def").get<int>();
        }
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getMotorAbility()
{
    try
    {
        nlohmann::json motorAbilityJson =
            getMotorAbilityJson().at("get_motor_ability");
        if (motorAbilityJson.is_null())
            return;
        std::cout << "motorAbilityJson: " << motorAbilityJson.dump()
                  << std::endl;
        m_cameraInfo.focus.value.m_min =
            motorAbilityJson.at("focus_value").at("min").get<int>();
        m_cameraInfo.focus.value.m_max =
            motorAbilityJson.at("focus_value").at("max").get<int>();
        m_cameraInfo.focus.value.m_def =
            motorAbilityJson.at("focus_value").at("def").get<int>();
        m_cameraInfo.focus.speed.m_min =
            motorAbilityJson.at("focus_speed").at("min").get<int>();
        m_cameraInfo.focus.speed.m_max =
            motorAbilityJson.at("focus_speed").at("max").get<int>();
        m_cameraInfo.focus.speed.m_def =
            motorAbilityJson.at("focus_speed").at("def").get<int>();
        m_cameraInfo.zoom.m_min =
            motorAbilityJson.at("ezoom_value").at("min").get<int>();
        m_cameraInfo.zoom.m_max =
            motorAbilityJson.at("ezoom_value").at("max").get<int>();
        m_cameraInfo.zoom.m_def =
            motorAbilityJson.at("ezoom_value").at("def").get<int>();
        m_cameraInfo.dzoom.m_min =
            motorAbilityJson.at("dzoom_value").at("min").get<int>();
        m_cameraInfo.dzoom.m_max =
            motorAbilityJson.at("dzoom_value").at("max").get<int>();
        m_cameraInfo.dzoom.m_def =
            motorAbilityJson.at("dzoom_value").at("def").get<int>();
        m_cameraInfo.zoomSpeed.m_min =
            motorAbilityJson.at("ezoom_speed").at("min").get<int>();
        m_cameraInfo.zoomSpeed.m_max =
            motorAbilityJson.at("ezoom_speed").at("max").get<int>();
        m_cameraInfo.zoomSpeed.m_def =
            motorAbilityJson.at("ezoom_speed").at("def").get<int>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

void NewNetCamera::getLedAbility()
{
    try
    {
        nlohmann::json ledAbilityJson =
            getLedAbilityJson().at("get_led_ability");
        if (ledAbilityJson.is_null())
            return;
        // std::cout << "ledAbilityJson: " << ledAbilityJson.dump() <<
        // std::endl;
        m_cameraInfo.led.partNumber =
            ledAbilityJson.at("part_number").get<int>();
        m_cameraInfo.led.levelRange.m_min =
            ledAbilityJson.at("level").at("min").get<int>();
        m_cameraInfo.led.levelRange.m_max =
            ledAbilityJson.at("level").at("max").get<int>();
        m_cameraInfo.led.levelRange.m_def =
            ledAbilityJson.at("level").at("def").get<int>();
    }
    catch (nlohmann::detail::exception &e)
    {
        std::cout << "json throw an error:" << e.what() << std::endl;
    }
}

nlohmann::json NewNetCamera::getImageValueInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_image_value"},
        {"get_image_value", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getImageBoolInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_image_bool"},
        {"get_image_bool", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getExposureInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_exposure"}, {"get_exposure", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getWbInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_white_balance"},
        {"get_white_balance", {{"chnn", m_chnn}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getFocusInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_focus"},
        {"get_focus", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getFocusRegionInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_roi"}, {"get_roi", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getFocusRegionInfoJsonV2()
{
    nlohmann::json request = {
        {"cmd", "get_focus_region"},
        {"get_focus_region", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getZoomInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_zoom"}, {"get_zoom", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getLedInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_led"}, {"get_led", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getIpv4InfoJson()
{
    nlohmann::json request = {{"cmd", "get_ipv4"}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getWirelessInfoJson()
{
    nlohmann::json request = {{"cmd", "get_wireless"}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getWirelessIpv4InfoJson()
{
    nlohmann::json request = {{"cmd", "get_wireless_ipv4"}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getDefogStatusJson()
{
    nlohmann::json request = {
        {"cmd", "get_motor_demist"},
        {"get_motor_demist", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getSceneInfoJson()
{
    nlohmann::json request = {
        {"cmd", "get_scene_mode"},
        {"get_scene_mode", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getStreamJson()
{
    nlohmann::json request = {
        {"cmd", "get_stream"},
        {"get_stream", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getStreamAbilityJson()
{
    nlohmann::json request = {
        {"cmd", "get_stream_ability"},
        {"get_stream_ability", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getPicAbilityJson()
{
    nlohmann::json request = {
        {"cmd", "get_pic_ability"},
        {"get_pic_ability", {{"chnn", m_chnn}, {"idx", m_picIdx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getImageAbilityJson()
{
    nlohmann::json request = {
        {"cmd", "get_image_ability"},
        {"get_image_ability", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getMotorAbilityJson()
{
    nlohmann::json request = {
        {"cmd", "get_motor_ability"},
        {"get_motor_ability", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

nlohmann::json NewNetCamera::getLedAbilityJson()
{
    nlohmann::json request = {
        {"cmd", "get_led_ability"},
        {"get_led_ability", {{"chnn", m_chnn}, {"idx", m_idx}}}};
    return getRequest(request);
}

}  // namespace sl
