﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace netsdkdemo
{
    #region 枚举

    public enum StreamFormat
    {
        kMJPEG,
        kH264
    }

    public enum PixFormat
    {
        kYUV420P,
        kRGB888,
        kBGR888,
        kARGB8888,
        kABGR8888,
        kRGBA8888,
        kBGRA8888
    }

    public enum ZoomMode
    {
        ZoomAbsolute = 1,
        ZoomRelative
    }

    public enum FocusMode
    {
        kFocusAuto,
        kFocusManual,
        kFocusOnce
    }

    public enum WbMode
    {
        kWBAuto,
        kWBManual
    }

    public enum ExposureMode
    {
        kExpAuto,
        kExpManual
    }

    #endregion

    #region 结构体

    [StructLayout(LayoutKind.Sequential)]
    public struct NetInfo
    {
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 64)]
        public string Ip;

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 64)]
        public string Model;

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 64)]
        public string HardwareVersion;

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 64)]
        public string SoftwareVersion;

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 64)]
        public string Username;

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 64)]
        public string Password;

        // 由softwareVersion转化至整数而来
        public int Version;
        public int Port;
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct CommonAttribute
    {
        public int Cur;
        public int Min;
        public int Max;
        public int Def;
        public int Res;

        public CommonAttribute(int cur = -1, int min = 0, int max = 0, int def = 0, int res = 1)
        {
            this.Cur = cur;
            this.Min = min;
            this.Max = max;
            this.Def = def;
            this.Res = res;
        }
    }

    public struct CameraSettings
    {
        public NetInfo netInfo;
        public CameraResolution Resolution;
        public StreamFormat StreamFormat;
        public PixFormat PixFormat;
        public string RcMode;
        public string Quality;
        public int FrameRate;
        public int Bitrate;
        public int KeyFrameInterval;
    }


    public struct FocusRegion
    {
        public bool Enable;
        public int Mode;
        public CommonAttribute X;
        public CommonAttribute Y;
    }

    public struct FocusRegionV2
    {
        public bool Enable;
        public int X;
        public int Y;
        public int Width;
        public int Height;
    };


    // CameraResolution类
    public class CameraResolution
    {
        public int Width;
        public int Height;
        public string Size;

        public CameraResolution() : this(1920, 1080)
        {
        }

        public CameraResolution(int width, int height)
        {
            Width = width;
            Height = height;
            Size = $"{width}*{height}";
        }

        public CameraResolution(int width, int height, string size)
        {
            Width = width;
            Height = height;
            Size = size;
        }

        // Copy constructor
        public CameraResolution(CameraResolution other)
        {
            if (other == null)
                throw new ArgumentNullException(nameof(other));

            Width = other.Width;
            Height = other.Height;
            Size = other.Size;
        }

        public override bool Equals(object obj)
        {
            var resolution = obj as CameraResolution;
            if (resolution == null) return false;

            return Width == resolution.Width && Height == resolution.Height && Size == resolution.Size;
        }

        // public override int GetHashCode()
        // {
        //     return HashCode.Combine(Width, Height, Size);
        // }

        public static bool operator ==(CameraResolution left, CameraResolution right)
        {
            return object.Equals(left, right);
        }

        public static bool operator !=(CameraResolution left, CameraResolution right)
        {
            return !object.Equals(left, right);
        }
    }

    public struct ExposureInfo
    {
        public ExposureMode Mode;
        public CommonAttribute Bright;
        public CommonAttribute Gain;
        public CommonAttribute Time;
    }

    public struct WhiteBalanceInfo
    {
        public WbMode Mode;

        // 色温
        public CommonAttribute Temperature;
        public CommonAttribute Red;
        public CommonAttribute Green;
        public CommonAttribute Blue;
    }

    public struct LedInfo
    {
        public int PartNumber;
        public CommonAttribute LevelRange;
        public List<LedPartInfo> LedPartVec;
    }
    // CommonAttribute结构体转换

    #endregion

    public class FocusInfo
    {
        public FocusMode Mode;
        public CommonAttribute Value;

        public CommonAttribute Speed;

        // 1-对应FocusRegion，2-对应FocusRegionV2
        public int RegionVersion = 1;
        public FocusRegion Region;
        public FocusRegionV2 RegionV2;
    }

    public class LedPartInfo
    {
        // LED灯分区号
        public int Index;

        // LED开关
        public bool Enable = true;

        // LED灯亮度
        public int Level;

        public LedPartInfo(int index, bool enable, int level)
        {
            this.Index = index;
            this.Enable = enable;
            this.Level = level;
        }
    }

    public class IpInfo
    {
        // 是否具有无线功能
        public bool EnWireless;
        public bool Dhcp;
        public string Ip;
        public string Gateway;
        public string Netmask;
        public string Mac;
        public string Dns1;
        public string Dns2;

        public bool WirelessDhcp;
        public string WirelessIp;
        public string WirelessGateway;
        public string WirelessNetmask;
        public string WirelessMac;
        public string WirelessDns1;

        public string WirelessDns2;

        // 无线模式，AP/STA
        public string WirelessType;

        // 无线频段，2.4G/5G
        public string WirelessNet;

        // AP名称
        public string WirelessApSsid;

        // AP密码
        public string WirelessApPasswd;

        // STA名称
        public string WirelessStaSsid;

        // STA密码
        public string WirelessStaPasswd;

        public string WirelessEnc;

        // 下面两个兼容F801W
        //  当WirelessType为AP时，WirelessSsid为AP名称，WirelessPasswd为AP密码
        //  当WirelessType为STA时，WirelessSsid为路由器WIFI名称，WirelessPasswd为路由器WIFI密码
        public string WirelessSsid;
        public string WirelessPasswd;

        // 默认构造函数
        public IpInfo()
        {
            EnWireless = false;
            Dhcp = false;
            WirelessDhcp = false;
        }

        // 重写Equals方法进行相等比较
        public override bool Equals(object obj)
        {
            if (obj is IpInfo other)
            {
                return Dhcp == other.Dhcp &&
                       Ip == other.Ip &&
                       Gateway == other.Gateway &&
                       Netmask == other.Netmask &&
                       Mac == other.Mac &&
                       Dns1 == other.Dns1 &&
                       Dns2 == other.Dns2 &&
                       WirelessDhcp == other.WirelessDhcp &&
                       WirelessIp == other.WirelessIp &&
                       WirelessGateway == other.WirelessGateway &&
                       WirelessNetmask == other.WirelessNetmask &&
                       WirelessMac == other.WirelessMac &&
                       WirelessDns1 == other.WirelessDns1 &&
                       WirelessDns2 == other.WirelessDns2 &&
                       WirelessType == other.WirelessType &&
                       WirelessNet == other.WirelessNet &&
                       WirelessSsid == other.WirelessSsid &&
                       WirelessPasswd == other.WirelessPasswd &&
                       WirelessApSsid == other.WirelessApSsid &&
                       WirelessApPasswd == other.WirelessApPasswd &&
                       WirelessStaSsid == other.WirelessStaSsid &&
                       WirelessStaPasswd == other.WirelessStaPasswd &&
                       WirelessEnc == other.WirelessEnc &&
                       EnWireless == other.EnWireless;
            }

            return false;
        }


        // 重写==和!=运算符
        public static bool operator ==(IpInfo left, IpInfo right)
        {
            return Equals(left, right);
        }

        public static bool operator !=(IpInfo left, IpInfo right)
        {
            return !Equals(left, right);
        }
    }

    public class CameraInfo
    {
        public string Model;
        public string ModelSuffix;
        public FocusInfo Focus = new FocusInfo();
        public ExposureInfo Exposure = new ExposureInfo();
        public WhiteBalanceInfo WhiteBalance = new WhiteBalanceInfo();
        public LedInfo Led = new LedInfo();
        public IpInfo ipInfo = new IpInfo();

        public CommonAttribute Hue;
        public CommonAttribute Saturation;
        public CommonAttribute Contrast;
        public CommonAttribute Sharpness;
        public CommonAttribute Gamma;
        public CommonAttribute Drc;
        public CommonAttribute ZoomSpeed;
        public CommonAttribute Zoom;
        public CommonAttribute Dzoom;
        public CommonAttribute ContrastEnhancement;

        public bool Defog = false;
        public bool PowerlineFrequency = false;
        public bool Monochrome = false;
        public bool Flip = false;
        public bool Mirror = false;
        public bool Cac = false;
        public bool SharpnessEnhancement = false;
        public int SceneMode = 0; // 场景 0-生物，1-工业，2-金相，3-单筒
    }
}