﻿using System;
using System.Runtime.InteropServices;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace netsdkdemo
{
    public class SLcamNet
    {
        private const string SLCamDll = "l_sdk.dll";
        private const CallingConvention cc = CallingConvention.Cdecl;

        #region P/Invoke Methods

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern void l_sdk_init(string config);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern void l_sdk_discover_open(string config);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern void l_sdk_discover_run(bool start);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern void l_sdk_discover_close();

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern void l_sdk_quit();

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern void l_sdk_discover_get_devs(out IntPtr devices);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern void l_sdk_free(IntPtr memory);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int l_sdk_request(int id, string request, out IntPtr response);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int l_sdk_login(out int id, string config);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int l_sdk_logout(int id);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int l_sdk_dec_open(int decId, string config);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int l_sdk_dec_close(int decId);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int l_sdk_dec_bind(int decId, int id, int chnn, int idx, int reserved);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int l_sdk_dec_unbind(int decId);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int l_sdk_dec_get_md_data(int decId, ref IntPtr frameData);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern void l_sdk_dec_free_md_data(IntPtr frameData);

        #endregion

        [StructLayout(LayoutKind.Sequential)]
        public struct l_buf_t
        {
            public IntPtr p_buf; // 缓存指针
            public int buf_len; // 缓存长度

            public int start; // 数据起始位置
            public int end; // 数据结束位置
        }

# if x64
        // 32位时intPtr只占4位，对应的intPtr的offset前移
        // 所有的数据指针和pitch共享同一块地址(即最外层被union所包裹)
        [StructLayout(LayoutKind.Explicit, Pack = 4)]
        public struct l_md_data_t
        {
            [FieldOffset(0)] public int type; // 数据类型: l_md_data_type_e
            [FieldOffset(4)] public long time; // 时间戳
            [FieldOffset(12)] public int w; // 图像宽度
            [FieldOffset(16)] public int h; // 图像高度

            [FieldOffset(20)] public IntPtr p_y; // Y分量指针
            [FieldOffset(28)] public int pitch_y; // Y分量行距
            [FieldOffset(32)] public IntPtr p_u; // U分量指针
            [FieldOffset(40)] public int pitch_u; // U分量行距
            [FieldOffset(44)] public IntPtr p_v; // V分量指针
            [FieldOffset(52)] public int pitch_v; // V分量行距

            // 共享内存：p_rgb 和 p_bgr 使用相同的内存位置
            [FieldOffset(20)] public IntPtr p_rgb; // RGB指针
            [FieldOffset(20)] public IntPtr p_bgr; // BGR指针 (共享内存)

            [FieldOffset(28)] public int pitch_888; // 24位图行距

            // 共享内存：p_argb, p_abgr, p_rgba, p_bgra 使用相同的内存位置
            [FieldOffset(20)] public IntPtr p_argb; // ARGB指针
            [FieldOffset(20)] public IntPtr p_abgr; // ABGR指针
            [FieldOffset(20)] public IntPtr p_rgba; // RGBA指针
            [FieldOffset(20)] public IntPtr p_bgra; // BGRA指针

            [FieldOffset(28)] public int pitch_8888; // 32位图行距

            // 数据缓存
            [FieldOffset(56)] public l_buf_t data; // 数据缓存

            public static l_md_data_t FromIntPtr(IntPtr ptr)
            {
                return (l_md_data_t)Marshal.PtrToStructure(ptr, typeof(l_md_data_t));
            }

            public override string ToString()
            {
                return $"l_md_data_t: " +
                       $"\n  type: {type}" +
                       $"\n  time: {time}" +
                       $"\n  w: {w}, h: {h}" +
                       $"\n  p_y: 0x{p_y.ToInt64():X}, pitch_y: {pitch_y}" +
                       $"\n  p_u: 0x{p_u.ToInt64():X}, pitch_u: {pitch_u}" +
                       $"\n  p_v: 0x{p_v.ToInt64():X}, pitch_v: {pitch_v}" +
                       $"\n  p_rgb: 0x{p_rgb.ToInt64():X}, p_bgr: 0x{p_bgr.ToInt64():X}" +
                       $"\n  pitch_888: {pitch_888}" +
                       $"\n  p_argb: 0x{p_argb.ToInt64():X}, p_abgr: 0x{p_abgr.ToInt64():X}, p_rgba: 0x{p_rgba.ToInt64():X}, p_bgra: 0x{p_bgra.ToInt64():X}" +
                       $"\n  pitch_8888: {pitch_8888}" +
                       $"\n  data 64: {data}";
            }
        }
# else
        [StructLayout(LayoutKind.Explicit, Pack = 4)]
        public struct l_md_data_t
        {
            [FieldOffset(0)] public int type; // 数据类型: l_md_data_type_e
            [FieldOffset(4)] public long time; // 时间戳
            [FieldOffset(12)] public int w; // 图像宽度
            [FieldOffset(16)] public int h; // 图像高度

            [FieldOffset(20)] public IntPtr p_y; // Y分量指针
            [FieldOffset(24)] public int pitch_y; // Y分量行距
            [FieldOffset(28)] public IntPtr p_u; // U分量指针
            [FieldOffset(32)] public int pitch_u; // U分量行距
            [FieldOffset(36)] public IntPtr p_v; // V分量指针
            [FieldOffset(40)] public int pitch_v; // V分量行距

            // 共享内存：p_rgb 和 p_bgr 使用相同的内存位置
            [FieldOffset(20)] public IntPtr p_rgb; // RGB指针
            [FieldOffset(20)] public IntPtr p_bgr; // BGR指针 (共享内存)

            [FieldOffset(24)] public int pitch_888; // 24位图行距

            // 共享内存：p_argb, p_abgr, p_rgba, p_bgra 使用相同的内存位置
            [FieldOffset(20)] public IntPtr p_argb; // ARGB指针
            [FieldOffset(20)] public IntPtr p_abgr; // ABGR指针
            [FieldOffset(20)] public IntPtr p_rgba; // RGBA指针
            [FieldOffset(20)] public IntPtr p_bgra; // BGRA指针

            [FieldOffset(24)] public int pitch_8888; // 32位图行距

            // 数据缓存
            [FieldOffset(44)] public l_buf_t data; // 数据缓存

            // 工具方法，供创建使用
            public static l_md_data_t FromIntPtr(IntPtr ptr)
            {
                return (l_md_data_t)Marshal.PtrToStructure(ptr, typeof(l_md_data_t));
            }

            public override string ToString()
            {
                return $"l_md_data_t: " +
                       $"\n  type: {type}" +
                       $"\n  time: {time}" +
                       $"\n  w: {w}, h: {h}" +
                       $"\n  p_y: 0x{p_y.ToInt64():X}, pitch_y: {pitch_y}" +
                       $"\n  p_u: 0x{p_u.ToInt64():X}, pitch_u: {pitch_u}" +
                       $"\n  p_v: 0x{p_v.ToInt64():X}, pitch_v: {pitch_v}" +
                       $"\n  p_rgb: 0x{p_rgb.ToInt64():X}, p_bgr: 0x{p_bgr.ToInt64():X}" +
                       $"\n  pitch_888: {pitch_888}" +
                       $"\n  p_argb: 0x{p_argb.ToInt64():X}, p_abgr: 0x{p_abgr.ToInt64():X}, p_rgba: 0x{p_rgba.ToInt64():X}, p_bgra: 0x{p_bgra.ToInt64():X}" +
                       $"\n  pitch_8888: {pitch_8888}" +
                       $"\n  data32: {data}";
            }
        }
#endif

        public class NetCamera
        {
            private int kStreamDecID = 100;
            protected int m_id;
            protected int m_chnn;
            protected int m_idx;
            private CameraSettings m_cameraSettings;
            protected CameraInfo m_cameraInfo = new CameraInfo();

            public void SetStreamDecID(int value)
            {
                kStreamDecID = value;
            }

            public bool GetVideoFrameData(ref l_md_data_t frameData)
            {
                IntPtr frameDataPtr = IntPtr.Zero;
                int ret = l_sdk_dec_get_md_data(kStreamDecID, ref frameDataPtr);
                if (ret != 0 || frameDataPtr == IntPtr.Zero)
                {
                    frameData = default;
                    return false;
                }

                frameData = l_md_data_t.FromIntPtr(frameDataPtr);
                l_sdk_dec_free_md_data(frameDataPtr);
                return true;
            }

            public static NetCamera Create(string cameraModel, int version)
            {
                if (cameraModel == "M112_IPC" ||
                    cameraModel == "M114_IPC" ||
                    cameraModel == "M202_IPC")
                {
                    return new NetCamera();
                }
                else if (cameraModel == "F801W_IPC")
                {
                    return new F801WNetCamera();
                }
                else if ((cameraModel == "C312_IPC" ||
                          cameraModel == "C313_IPC" ||
                          cameraModel == "C314_IPC" ||
                          cameraModel == "L313_IPC" ||
                          cameraModel == "L314_IPC") && version < 180)
                {
                    return new NetCamera();
                }
                else if (cameraModel == "L116_IPC" && version < 170)
                {
                    return new NetCamera();
                }

                return new NewNetCamera();
            }

            public CameraSettings GetCameraSettings() => m_cameraSettings;

            public void SetCameraSettings(CameraSettings settings) => m_cameraSettings = settings;

            public static void InitNet()
            {
                l_sdk_init("");
                l_sdk_discover_open("");
                l_sdk_discover_run(true);
            }

            public static void DestroyNet()
            {
                l_sdk_discover_run(false);
                l_sdk_discover_close();
                l_sdk_quit();
            }

            public static List<NetInfo> SearchAvailableCameras()
            {
                List<NetInfo> infos = new List<NetInfo>();
                SLcamNet.l_sdk_discover_get_devs(out IntPtr p_res);
                if (p_res == IntPtr.Zero) return infos;

                string res = Marshal.PtrToStringAnsi(p_res);
                //SLcamNet.NetCamera.LOG("res = " + res);
                SLcamNet.l_sdk_free(p_res);
                JArray resJson = null;
                SLcamNet.NetCamera.LOG("res length() = " + res.Length);

                if (string.IsNullOrEmpty(res) || res.Equals("{}") || res.Length < 10)
                    return infos;

                resJson = JArray.Parse(res);

                foreach (var item in resJson)
                {
                    try
                    {
                        var info = new NetInfo
                        {
                            Ip = item["ip"].ToString(),
                            Model = item["discover"]["model"].ToString(),
                            SoftwareVersion = item["discover"]["sw_ver"].ToString(),
                            HardwareVersion = item["discover"]["hw_ver"].ToString(),
                            Port = item["discover"]["port"].ToObject<int>()
                        };
                        info.Version = VersionFromSoftwareVer(info.SoftwareVersion);
                        infos.Add(info);
                    }
                    catch (Exception e)
                    {
                        SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                    }
                }

                return infos;
            }

            public JObject GetRequest(JObject requestJson)
            {
                string request = requestJson.ToString(Formatting.None);
                SLcamNet.l_sdk_request(m_id, request, out IntPtr resultStr);

                if (resultStr == IntPtr.Zero)
                {
                    SLcamNet.NetCamera.LOG("Get request failed resultStr");
                    return null;
                }

                string result = Marshal.PtrToStringAnsi(resultStr);
                SLcamNet.l_sdk_free(resultStr);

                JObject resJson = JObject.Parse(result);
                //SLcamNet.NetCamera.LOG(resJson.ToString());
                int code = resJson[requestJson["cmd"].ToString()]["code"].ToObject<int>();

                if (code != 0)
                {
                    SLcamNet.NetCamera.LOG("Get request failed code != 0");
                    return null;
                }

                return resJson;
            }

            public bool SetRequest(JObject requestJson)
            {
                string request = requestJson.ToString(Formatting.None);
                int ret = SLcamNet.l_sdk_request(m_id, request, out IntPtr resultStr);

                if (resultStr == IntPtr.Zero)
                {
                    SLcamNet.NetCamera.LOG("Set request failed");
                    return false;
                }

                string result = Marshal.PtrToStringAnsi(resultStr);
                SLcamNet.l_sdk_free(resultStr);

                JObject resJson = JObject.Parse(result);
                int code = resJson[requestJson["cmd"].ToString()]["code"].ToObject<int>();

                return ret == 0 && code == 0;
            }

            public bool Login()
            {
                JObject request = new JObject
                {
                    ["ip"] = m_cameraSettings.netInfo.Ip,
                    ["port"] = m_cameraSettings.netInfo.Port,
                    ["cmd"] = "login",
                    ["login"] = new JObject
                    {
                        ["username"] = m_cameraSettings.netInfo.Username,
                        ["passwd"] = m_cameraSettings.netInfo.Password
                    }
                };

                if (SLcamNet.l_sdk_login(out m_id, request.ToString(Formatting.None)) != 0)
                    return false;

                SLcamNet.NetCamera.LOG($"login success! id={m_id}");
                return true;
            }

            public bool Logout() => SLcamNet.l_sdk_logout(m_id) == 0;

            public bool Open()
            {
                JObject decJson = new JObject();
                switch (m_cameraSettings.StreamFormat)
                {
                    case StreamFormat.kMJPEG:
                        decJson["md_fmt"] = "MJPEG";
                        break;
                    case StreamFormat.kH264:
                        decJson["md_fmt"] = "H264";
                        break;
                    default:
                        break;
                }

                switch (m_cameraSettings.PixFormat)
                {
                    case PixFormat.kYUV420P:
                        decJson["pix_fmt"] = "YUV420P_SEPARATE";
                        break;
                    case PixFormat.kRGB888:
                        decJson["pix_fmt"] = "RGB888";
                        break;
                    case PixFormat.kRGBA8888:
                        decJson["pix_fmt"] = "RGBA8888";
                        break;
                    case PixFormat.kBGR888:
                        decJson["pix_fmt"] = "BGR888";
                        break;
                    case PixFormat.kBGRA8888:
                        decJson["pix_fmt"] = "BGRA8888";
                        break;
                    case PixFormat.kABGR8888:
                        decJson["pix_fmt"] = "ABGR8888";
                        break;
                    case PixFormat.kARGB8888:
                        decJson["pix_fmt"] = "ARGB8888";
                        break;
                    default:
                        break;
                }

                int ret = l_sdk_dec_open(kStreamDecID, decJson.ToString(Formatting.None));
                if (ret != 0)
                {
                    SLcamNet.NetCamera.LOG("Create decoder failed!");
                    return false;
                }

                ret = SLcamNet.l_sdk_dec_bind(kStreamDecID, m_id, m_chnn, m_idx, 0);
                if (ret != 0)
                {
                    SLcamNet.l_sdk_dec_close(kStreamDecID);
                    SLcamNet.NetCamera.LOG("Bind decoder failed!");
                    return false;
                }

                // GetNetCameraAllInfo();

                if (m_cameraSettings.StreamFormat != StreamFormat.kMJPEG &&
                    m_cameraSettings.Bitrate > 30000)
                    m_cameraSettings.Bitrate = 30000;

                SetStream(
                    decJson["md_fmt"].ToString(),
                    m_cameraSettings.RcMode,
                    m_cameraSettings.Resolution.Size,
                    m_cameraSettings.Quality,
                    m_cameraSettings.FrameRate,
                    m_cameraSettings.Bitrate,
                    m_cameraSettings.KeyFrameInterval);

                return RequestStream();
            }

            public bool Close()
            {
                CloseStream();
                l_sdk_dec_unbind(kStreamDecID);
                l_sdk_dec_close(kStreamDecID);
                Logout();
                return true;
            }

            public bool ChangeStreamSettings(CameraSettings settings)
            {
                bool streamFormatChanged = (m_cameraSettings.StreamFormat != settings.StreamFormat);
                m_cameraSettings = settings;

                JObject decJson = new JObject();
                string strStreamFormat = "h264";
                if (settings.StreamFormat == StreamFormat.kMJPEG)
                {
                    decJson["md_fmt"] = "MJPEG";
                    strStreamFormat = "mjpeg";
                }
                else if (settings.StreamFormat == StreamFormat.kH264)
                {
                    decJson["md_fmt"] = "H264";
                    strStreamFormat = "h264";
                }

                if (streamFormatChanged)
                {
                    SLcamNet.NetCamera.LOG("Stream format changed!");
                    SLcamNet.l_sdk_dec_unbind(kStreamDecID);
                    SLcamNet.l_sdk_dec_close(kStreamDecID);
                }

                if (settings.StreamFormat != StreamFormat.kMJPEG && settings.Bitrate > 30000)
                {
                    settings.Bitrate = 30000;
                }

                SetStream(
                    strStreamFormat,
                    settings.RcMode,
                    settings.Resolution.Size,
                    settings.Quality,
                    settings.FrameRate,
                    settings.Bitrate,
                    settings.KeyFrameInterval
                );

                if (streamFormatChanged)
                {
                    switch (settings.PixFormat)
                    {
                        case PixFormat.kYUV420P:
                            decJson["pix_fmt"] = "YUV420P_SEPARATE";
                            break;
                        case PixFormat.kRGB888:
                            decJson["pix_fmt"] = "RGB888";
                            break;
                        case PixFormat.kRGBA8888:
                            decJson["pix_fmt"] = "RGBA8888";
                            break;
                        case PixFormat.kBGR888:
                            decJson["pix_fmt"] = "BGR888";
                            break;
                        case PixFormat.kBGRA8888:
                            decJson["pix_fmt"] = "BGRA8888";
                            break;
                        case PixFormat.kABGR8888:
                            decJson["pix_fmt"] = "ABGR8888";
                            break;
                        case PixFormat.kARGB8888:
                            decJson["pix_fmt"] = "ARGB8888";
                            break;
                        default:
                            break;
                    }

                    l_sdk_dec_open(kStreamDecID, decJson.ToString(Formatting.None));
                    l_sdk_dec_bind(kStreamDecID, m_id, m_chnn, m_idx, 0);
                }

                return true;
            }

            public virtual CameraInfo GetNetCameraAllInfo()
            {
                GetSupportFuncInfo();
                GetRangeInfo();
                getStreamInfo();
                GetControlInfo();
                GetFocusRegionInfo();
                GetExposureInfo();
                GetWbInfo();
                GetFlipMirrorStatus();
                GetGammaValue();
                GetDefogStatus();
                GetFlickStatus();
                GetSystemBaseInfo();
                GetIpv4Info();
                // 是否具有WIFI功能
                if (m_cameraInfo.ipInfo.EnWireless)
                {
                    GetWirelessIpv4Info();
                    GetWirelessInfo();
                }

                GetCACInfo();
                GetDRCInfo();
                GetSceneInfo();
                GetWbTemperatureInfo();
                return m_cameraInfo;
            }

            private void getStreamInfo()
            {
                JObject streamInfoJson = getStreamInfoJson();

                if (streamInfoJson == null)
                    return;

                SLcamNet.NetCamera.LOG($"getStreamInfo, {streamInfoJson.ToString()}");
            }

            public bool GetVideoFrameData(ref IntPtr frameData)
            {
                // frameData = IntPtr.Zero;
                if (l_sdk_dec_get_md_data(kStreamDecID, ref frameData) != 0 || frameData == IntPtr.Zero)
                    return false;

                l_md_data_t frameDatas = l_md_data_t.FromIntPtr(frameData);

                // Your conditions, for example checking resolution
                if (frameDatas.w != m_cameraSettings.Resolution.Width ||
                    frameDatas.h != m_cameraSettings.Resolution.Height)
                {
                    SLcamNet.l_sdk_dec_free_md_data(frameData);
                    frameData = IntPtr.Zero;
                    return false;
                }

                return true;
            }

            public virtual bool SetFocus(FocusMode mode, int value, int speed)
            {
                m_cameraInfo.Focus.Mode = mode;
                // C31X相机默认范围为-25-375
                // M系列value值无效, 置0即可
                if (m_cameraSettings.netInfo.Model == "M202_IPC" ||
                    m_cameraSettings.netInfo.Model == "M114_IPC" ||
                    m_cameraSettings.netInfo.Model == "M112_IPC")
                {
                    m_cameraInfo.Focus.Value.Cur = 0;
                }
                else
                {
                    m_cameraInfo.Focus.Value.Cur = value;
                }

                m_cameraInfo.Focus.Speed.Cur = speed;
                int focusMode = ConvertNetFocusMode(mode);

                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    m_cameraInfo.Saturation.Cur, m_cameraInfo.Hue.Cur, focusMode,
                    m_cameraInfo.Focus.Value.Cur,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public virtual bool SetFocusRegion(bool enable, int x, int y, int mode)
            {
                m_cameraInfo.Focus.Region.Enable = enable;
                m_cameraInfo.Focus.Region.X.Cur = x;
                m_cameraInfo.Focus.Region.Y.Cur = y;
                m_cameraInfo.Focus.Region.Mode = mode;

                JObject request = new JObject
                {
                    ["cmd"] = "set_img_roi",
                    ["set_img_roi"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["roi"] = enable,
                        ["roi_x"] = x,
                        ["roi_y"] = y,
                        ["roi_mode"] = mode
                    }
                };

                return SetRequest(request);
            }

            public bool FocusAdd()
            {
                if (m_cameraInfo.Focus.Mode != FocusMode.kFocusManual)
                    return false;

                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    m_cameraInfo.Saturation.Cur, m_cameraInfo.Hue.Cur,
                    ConvertNetFocusMode(m_cameraInfo.Focus.Mode), 1,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public bool FocusSub()
            {
                if (m_cameraInfo.Focus.Mode != FocusMode.kFocusManual)
                    return false;

                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    m_cameraInfo.Saturation.Cur, m_cameraInfo.Hue.Cur,
                    ConvertNetFocusMode(m_cameraInfo.Focus.Mode), -1,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public virtual bool SetExposure(ExposureMode mode, int compensation, int gain, int shutterTime)
            {
                m_cameraInfo.Exposure.Mode = mode;
                m_cameraInfo.Exposure.Bright.Cur = compensation;
                m_cameraInfo.Exposure.Gain.Cur = gain;
                m_cameraInfo.Exposure.Time.Cur = shutterTime;

                SetMEExposure(
                    m_cameraInfo.Exposure.Mode, m_cameraInfo.Exposure.Gain.Cur,
                    m_cameraInfo.Exposure.Time.Cur);

                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    m_cameraInfo.Saturation.Cur, m_cameraInfo.Hue.Cur,
                    ConvertNetFocusMode(m_cameraInfo.Focus.Mode),
                    m_cameraInfo.Focus.Value.Cur,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public virtual bool SetWb(WbMode mode, int red, int green, int blue)
            {
                string wbMode = ConvertNetWhiteBalanceMode(mode);
                m_cameraInfo.WhiteBalance.Red.Cur = red;
                m_cameraInfo.WhiteBalance.Green.Cur = green;
                m_cameraInfo.WhiteBalance.Blue.Cur = blue;

                JObject request = new JObject
                {
                    ["cmd"] = "set_img_awb",
                    ["set_img_awb"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["awb"] = wbMode,
                        ["b"] = m_cameraInfo.WhiteBalance.Blue.Cur,
                        ["gb"] = m_cameraInfo.WhiteBalance.Green.Cur,
                        ["gr"] = m_cameraInfo.WhiteBalance.Green.Cur,
                        ["r"] = m_cameraInfo.WhiteBalance.Red.Cur
                    }
                };

                return SetRequest(request);
            }

            public virtual bool SetZoom(int ezoomValue, int dzoomValue, int speed,
                ZoomMode mode = ZoomMode.ZoomAbsolute)
            {
                if (ezoomValue > m_cameraInfo.Zoom.Max)
                    ezoomValue = m_cameraInfo.Zoom.Max;

                if (dzoomValue > m_cameraInfo.Dzoom.Max)
                    dzoomValue = m_cameraInfo.Dzoom.Max;

                m_cameraInfo.Zoom.Cur = ezoomValue;
                m_cameraInfo.Dzoom.Cur = ezoomValue == m_cameraInfo.Zoom.Max ? dzoomValue : 0;
                m_cameraInfo.ZoomSpeed.Cur = speed;

                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    m_cameraInfo.Saturation.Cur, m_cameraInfo.Hue.Cur,
                    ConvertNetFocusMode(m_cameraInfo.Focus.Mode),
                    m_cameraInfo.Focus.Value.Cur,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public virtual void GetZoomInfo()
            {
                GetControlInfo();
            }

            public virtual bool SetFlip(bool enable)
            {
                m_cameraInfo.Flip = enable;
                return SetFlipMirror(m_cameraInfo.Flip, m_cameraInfo.Mirror);
            }

            public virtual bool SetMirror(bool enable)
            {
                m_cameraInfo.Mirror = enable;
                return SetFlipMirror(m_cameraInfo.Flip, m_cameraInfo.Mirror);
            }

            public virtual bool SetCAC(bool enable)
            {
                m_cameraInfo.Cac = enable;
                var request = new JObject
                {
                    ["cmd"] = "set_img_cac",
                    ["set_img_cac"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["cac"] = m_cameraInfo.Cac
                    }
                };

                return SetRequest(request);
            }

            public virtual bool SetFlick(bool enable)
            {
                m_cameraInfo.PowerlineFrequency = enable;
                JObject request = new JObject
                {
                    ["cmd"] = "set_img_flick_hz",
                    ["set_img_flick_hz"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["flick"] = m_cameraInfo.PowerlineFrequency
                    }
                };

                return SetRequest(request);
            }

            public virtual bool SetMonochrome(bool enable)
            {
                m_cameraInfo.Monochrome = enable;
                int tempSaturation = enable ? 0 : m_cameraInfo.Saturation.Cur;

                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    tempSaturation, m_cameraInfo.Hue.Cur,
                    ConvertNetFocusMode(m_cameraInfo.Focus.Mode),
                    m_cameraInfo.Focus.Value.Cur,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public virtual bool SetDefog(bool enable)
            {
                m_cameraInfo.Defog = enable;
                JObject request = new JObject
                {
                    ["cmd"] = "set_motor_demist",
                    ["set_motor_demist"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["demist"] = m_cameraInfo.Defog
                    }
                };

                return SetRequest(request);
            }

            public bool SetLed(int index, int level)
            {
                SLcamNet.NetCamera.LOG("Led not support!");
                return false;
            }

            public bool SetContrastValue(int value)
            {
                m_cameraInfo.Contrast.Cur = value;
                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    m_cameraInfo.Saturation.Cur, m_cameraInfo.Hue.Cur,
                    ConvertNetFocusMode(m_cameraInfo.Focus.Mode),
                    m_cameraInfo.Focus.Value.Cur,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public bool SetSaturationValue(int value)
            {
                m_cameraInfo.Saturation.Cur = value;
                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    m_cameraInfo.Saturation.Cur, m_cameraInfo.Hue.Cur,
                    ConvertNetFocusMode(m_cameraInfo.Focus.Mode),
                    m_cameraInfo.Focus.Value.Cur,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public bool SetHueValue(int value)
            {
                m_cameraInfo.Hue.Cur = value;
                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    m_cameraInfo.Saturation.Cur, m_cameraInfo.Hue.Cur,
                    ConvertNetFocusMode(m_cameraInfo.Focus.Mode),
                    m_cameraInfo.Focus.Value.Cur,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public bool SetSharpnessValue(int value)
            {
                m_cameraInfo.Sharpness.Cur = value;
                return Control(
                    m_cameraInfo.Exposure.Bright.Cur, m_cameraInfo.Contrast.Cur,
                    m_cameraInfo.Saturation.Cur, m_cameraInfo.Hue.Cur,
                    ConvertNetFocusMode(m_cameraInfo.Focus.Mode),
                    m_cameraInfo.Focus.Value.Cur,
                    m_cameraInfo.Zoom.Cur + m_cameraInfo.Dzoom.Cur,
                    m_cameraInfo.Sharpness.Cur);
            }

            public bool SetGammaValue(int value)
            {
                m_cameraInfo.Gamma.Cur = value;
                JObject request = new JObject
                {
                    ["cmd"] = "set_img_hdr_gamma",
                    ["set_img_hdr_gamma"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["level"] = m_cameraInfo.Gamma.Cur
                    }
                };

                return SetRequest(request);
            }

            public bool SetDRCValue(int value)
            {
                m_cameraInfo.Drc.Cur = value;

                var request = new JObject
                {
                    ["cmd"] = "set_img_drc",
                    ["set_img_drc"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["drc"] = m_cameraInfo.Drc.Cur
                    }
                };

                return SetRequest(request);
            }

            // 场景切换, 0-生物, 1-工业, 2-金相, 3-单筒 (仅L116支持，支持工业/金相/单筒)
            public virtual bool SetScene(int mode)
            {
                m_cameraInfo.SceneMode = mode;
                var request = new JObject
                {
                    ["cmd"] = "set_img_mode",
                    ["set_img_mode"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["mode"] = m_cameraInfo.SceneMode
                    }
                };

                return SetRequest(request);
            }

            public virtual bool SetWbTemperature(int temperature)
            {
                Console.WriteLine("Current camera does not support setting wb temperature");
                return false;
            }

            public bool CloseStream()
            {
                var request = new JObject
                {
                    ["cmd"] = "close_stream",
                    ["close_stream"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };

                return SetRequest(request);
            }

            // 设置相机有线IP相关信息
            // dhcp     是否启用DHCP, 有线DHCP暂不可用，默认为false
            // ip       相机新IP
            // gateway  网关
            // netmask  子网掩码
            // dns      dns地址
            // mac      MAC地址
            // 如在拉流状态, 修改IP后视频流将会断开

            public bool SetIpv4(bool dhcp, string ip, string gateway, string netmask, string dns1, string dns2,
                string mac)
            {
                JObject request = new JObject
                {
                    ["cmd"] = "set_ipv4",
                    ["set_ipv4"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["dhcp"] = dhcp,
                        ["ip"] = ip,
                        ["gateway"] = gateway,
                        ["netmask"] = netmask,
                        ["dns1"] = dns1,
                        ["dns2"] = dns2,
                        ["mac"] = mac
                    }
                };

                return SetRequest(request);
            }

            // gateway 网关
            // netmask 子网掩码
            // dns dns地址
            // mac MAC地址, 仅支持单播MAC, 即第一个字节最后一位不可为1
            public bool SetWirelessIPv4(bool dhcp, string ip, string gateway, string netmask, string dns1, string dns2,
                string mac)
            {
                JObject request = new JObject
                {
                    ["cmd"] = "set_wireless_ipv4",
                    ["set_wireless_ipv4"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["dhcp"] = dhcp,
                        ["ip"] = ip,
                        ["gateway"] = gateway,
                        ["netmask"] = netmask,
                        ["dns1"] = dns1,
                        ["dns2"] = dns2,
                        ["mac"] = mac
                    }
                };

                return SetRequest(request);
            }

            // 设置相机无线相关信息
            // type        ap/sta 无线模式
            // net         2.4g/5g 无线频段, 目前仅支持5g
            // apSsid      AP模式下，相机AP名称
            // apPasswd    相机AP密码
            // staSsid     STA模式下，路由器无线名称
            // staPasswd   STA模式下，路由器无线密码
            // enc         wpa-wpa2 psk/wpa2 psk/psk/none
            //             AP模式下，为相机AP的加密方式;
            //             STA模式下，为路由器WIFI的加密方式
            public bool SetWireless(string type, string net, string apSsid, string apPasswd, string staSsid,
                string staPasswd, string enc)
            {
                // 兼容老型号处理
                bool apMode = type.Equals("AP", StringComparison.OrdinalIgnoreCase);
                string ssid = apMode ? apSsid : staSsid;
                string passwd = apMode ? apPasswd : staPasswd;

                var request = new JObject
                {
                    ["cmd"] = "set_wireless",
                    ["set_wireless"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["type"] = type,
                        ["net"] = net,
                        ["ssid"] = ssid,
                        ["passwd"] = passwd,
                        ["ap_ssid"] = apSsid,
                        ["ap_passwd"] = apPasswd,
                        ["sta_ssid"] = staSsid,
                        ["sta_passwd"] = staPasswd,
                        ["enc"] = enc
                    }
                };

                return SetRequest(request);
            }

            public bool Upgrade(string upgradeFilePath)
            {
                JObject requestJson = new JObject
                {
                    ["cmd"] = "upgrade",
                    ["ip"] = m_cameraSettings.netInfo.Ip,
                    ["port"] = m_cameraSettings.netInfo.Port,
                    ["path"] = upgradeFilePath,
                    ["upgrade"] = new JObject
                    {
                        ["username"] = m_cameraSettings.netInfo.Username,
                        ["passwd"] = m_cameraSettings.netInfo.Password
                    }
                };

                string request = requestJson.ToString(Formatting.None);
                int ret = SLcamNet.l_sdk_request(0, request, out IntPtr resultStr);

                if (resultStr == IntPtr.Zero)
                    return false;

                string result = Marshal.PtrToStringAnsi(resultStr);
                SLcamNet.l_sdk_free(resultStr);

                JObject resJson = JObject.Parse(result);
                int code = resJson[requestJson["cmd"].ToString()]["code"].ToObject<int>();

                return ret == 0 && code == 0;
            }

            private bool Control(int bright, int contrast, int saturation, int hue, int focusMode, int focusValue,
                int zoomValue, int sharpness)
            {
                JObject request = new JObject
                {
                    ["cmd"] = "set_image",
                    ["set_image"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["bright"] = bright,
                        ["contrast"] = contrast,
                        ["saturation"] = saturation,
                        ["hue"] = hue,
                        ["focus_mode"] = focusMode,
                        ["focus_value"] = focusValue,
                        ["ezoom_value"] = zoomValue,
                        ["sharpness"] = sharpness
                    }
                };

                return SetRequest(request);
            }

            private bool SetStream(string streamFormat, string rcMode, string resolution, string quality, int frameRate,
                int bitrate, int keyFrameInterval)
            {
                JObject request = new JObject
                {
                    ["cmd"] = "set_stream",
                    ["set_stream"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx,
                        ["fmt"] = streamFormat,
                        ["rc_mode"] = rcMode,
                        ["wh"] = resolution,
                        ["quality"] = quality,
                        ["frame_rate"] = frameRate,
                        ["bitrate"] = bitrate,
                        ["i_interval"] = keyFrameInterval
                    }
                };

                SLcamNet.NetCamera.LOG("setStream: " + request.ToString(Formatting.None));
                return SetRequest(request);
            }

            private bool RequestStream()
            {
                JObject request = new JObject
                {
                    ["cmd"] = "open_stream",
                    ["open_stream"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };

                return SetRequest(request);
            }

            private bool SetMEExposure(ExposureMode mode, int gain, int shutterTime)
            {
                int exposureMode = ConvertNetExposureMode(mode);
                JObject request = new JObject
                {
                    ["cmd"] = "set_img_meexposure",
                    ["set_img_meexposure"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["expo_type"] = exposureMode,
                        ["shuttertime"] = shutterTime,
                        ["a_gain"] = gain
                    }
                };

                return SetRequest(request);
            }

            public bool SetFlipMirror(bool flip, bool mirror)
            {
                JObject request = new JObject
                {
                    ["cmd"] = "set_img_mirror_flip",
                    ["set_img_mirror_flip"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["flip"] = flip,
                        ["mirror"] = mirror
                    }
                };

                return SetRequest(request);
            }

            public virtual void GetRangeInfo()
            {
                // 目前仅C313 1.5.0版本及L116 1.6.0版本后支持获取范围及默认值接口
                GetControlRangeInfo();
                GetFocusRegionRangeInfo();
                GetMExposureRangeInfo();
                GetWbRangeInfo();
                GetGammaRangeInfo();
                GetDRCRangeInfo();
                GetWbTemperatureRangeInfo();

                m_cameraInfo.Focus.Value.Def = 175;
                m_cameraInfo.Focus.Value.Max = 375;
                m_cameraInfo.Focus.Value.Min = -25;
                m_cameraInfo.Focus.Value.Res = 1;

                m_cameraInfo.Exposure.Bright.Def = 56;
                m_cameraInfo.Exposure.Bright.Max = 250;
                m_cameraInfo.Exposure.Bright.Min = 0;
                m_cameraInfo.Exposure.Bright.Res = 1;

                m_cameraInfo.Exposure.Time.Def = 150;
                m_cameraInfo.Exposure.Time.Max = 333;
                m_cameraInfo.Exposure.Time.Min = 1;
                m_cameraInfo.Exposure.Time.Res = 1;

                m_cameraInfo.Exposure.Gain.Def = 0;
                m_cameraInfo.Exposure.Gain.Max = 33;
                m_cameraInfo.Exposure.Gain.Min = 0;
                m_cameraInfo.Exposure.Gain.Res = 1;

                m_cameraInfo.WhiteBalance.Red.Def = 628;
                m_cameraInfo.WhiteBalance.Red.Max = 4095;
                m_cameraInfo.WhiteBalance.Red.Min = 1;
                m_cameraInfo.WhiteBalance.Red.Res = 1;

                m_cameraInfo.WhiteBalance.Green.Def = 256;
                m_cameraInfo.WhiteBalance.Green.Max = 4095;
                m_cameraInfo.WhiteBalance.Green.Min = 1;
                m_cameraInfo.WhiteBalance.Green.Res = 1;

                m_cameraInfo.WhiteBalance.Blue.Def = 421;
                m_cameraInfo.WhiteBalance.Blue.Max = 4095;
                m_cameraInfo.WhiteBalance.Blue.Min = 1;
                m_cameraInfo.WhiteBalance.Blue.Res = 1;

                m_cameraInfo.Hue.Def = 50;
                m_cameraInfo.Hue.Max = 100;
                m_cameraInfo.Hue.Min = 0;
                m_cameraInfo.Hue.Res = 1;

                m_cameraInfo.Saturation.Def = 128;
                m_cameraInfo.Saturation.Max = 255;
                m_cameraInfo.Saturation.Min = 0;
                m_cameraInfo.Saturation.Res = 1;

                m_cameraInfo.Contrast.Def = 50;
                m_cameraInfo.Contrast.Max = 100;
                m_cameraInfo.Contrast.Min = 0;
                m_cameraInfo.Contrast.Res = 1;

                m_cameraInfo.Sharpness.Def = 8;
                m_cameraInfo.Sharpness.Max = 15;
                m_cameraInfo.Sharpness.Min = 0;
                m_cameraInfo.Sharpness.Res = 1;

                m_cameraInfo.Gamma.Def = 1;
                m_cameraInfo.Gamma.Min = 0;
                m_cameraInfo.Gamma.Max = 3;
                m_cameraInfo.Gamma.Res = 1;

                m_cameraInfo.Drc.Def = 100;
                m_cameraInfo.Drc.Max = 255;
                m_cameraInfo.Drc.Min = 0;
                m_cameraInfo.Drc.Res = 1;

                m_cameraInfo.Zoom.Def = 0;
                m_cameraInfo.Zoom.Min = 0;

                // M202: zoom(0-144), dzoom(0-30)
                // M112/M114: zoom(0-130), dzoom(0-50)
                if (m_cameraSettings.netInfo.Model == "M202_IPC")
                    m_cameraInfo.Zoom.Max = 144;
                else
                    m_cameraInfo.Zoom.Max = 130;

                m_cameraInfo.Zoom.Res = 1;

                m_cameraInfo.Dzoom.Def = 0;
                m_cameraInfo.Dzoom.Min = 0;

                if (m_cameraSettings.netInfo.Model == "M202_IPC")
                    m_cameraInfo.Dzoom.Max = 30;
                else
                    m_cameraInfo.Dzoom.Max = 50;

                m_cameraInfo.Dzoom.Res = 1;
            }

            private void GetControlInfo()
            {
                JObject imgRecJson = GetControlInfoJson();
                if (imgRecJson == null)
                    return;

                try
                {
                    int focusMode = imgRecJson["get_image"]["focus_mode"].ToObject<int>();
                    m_cameraInfo.Focus.Mode = ConvertFromNetFocusMode(focusMode);

                    m_cameraInfo.Focus.Value.Cur =
                        imgRecJson["get_image"]["focus_value"].ToObject<int>();
                    m_cameraInfo.Exposure.Bright.Cur =
                        imgRecJson["get_image"]["bright"].ToObject<int>();
                    m_cameraInfo.Contrast.Cur =
                        imgRecJson["get_image"]["contrast"].ToObject<int>();
                    m_cameraInfo.Saturation.Cur =
                        imgRecJson["get_image"]["saturation"].ToObject<int>();
                    m_cameraInfo.Monochrome = m_cameraInfo.Saturation.Cur == 0;
                    m_cameraInfo.Hue.Cur =
                        imgRecJson["get_image"]["hue"].ToObject<int>();
                    m_cameraInfo.Sharpness.Cur =
                        imgRecJson["get_image"]["sharpness"].ToObject<int>();
                    m_cameraInfo.Zoom.Cur =
                        imgRecJson["get_image"]["ezoom_value"].ToObject<int>();

                    if (m_cameraInfo.Zoom.Cur > m_cameraInfo.Zoom.Max)
                    {
                        m_cameraInfo.Dzoom.Cur =
                            m_cameraInfo.Zoom.Cur - m_cameraInfo.Zoom.Max;
                        m_cameraInfo.Zoom.Cur = m_cameraInfo.Zoom.Max;
                    }
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                }
            }

            public virtual void GetWirelessInfo()
            {
                JObject wirelessInfoJson = GetWirelessInfoJson();
                if (wirelessInfoJson == null)
                    return;

                try
                {
                    var wireless = wirelessInfoJson["wireless"];
                    if (wireless == null)
                        throw new Exception("Wireless data is missing in JSON.");

                    m_cameraInfo.ipInfo.WirelessType = wireless.Value<string>("type");
                    m_cameraInfo.ipInfo.WirelessNet = wireless.Value<string>("net");

                    bool apMode = m_cameraInfo.ipInfo.WirelessType == "AP";

                    if (wireless["ap_ssid"] != null)
                    {
                        m_cameraInfo.ipInfo.WirelessApSsid = wireless.Value<string>("ap_ssid");
                        m_cameraInfo.ipInfo.WirelessApPasswd = wireless.Value<string>("ap_passwd");
                        m_cameraInfo.ipInfo.WirelessStaSsid = wireless.Value<string>("sta_ssid");
                        m_cameraInfo.ipInfo.WirelessStaPasswd = wireless.Value<string>("sta_passwd");

                        if (apMode)
                        {
                            m_cameraInfo.ipInfo.WirelessSsid = m_cameraInfo.ipInfo.WirelessApSsid;
                            m_cameraInfo.ipInfo.WirelessPasswd = m_cameraInfo.ipInfo.WirelessApPasswd;
                        }
                        else
                        {
                            m_cameraInfo.ipInfo.WirelessSsid = m_cameraInfo.ipInfo.WirelessStaSsid;
                            m_cameraInfo.ipInfo.WirelessPasswd = m_cameraInfo.ipInfo.WirelessStaPasswd;
                        }
                    }
                    else
                    {
                        m_cameraInfo.ipInfo.WirelessSsid = wireless.Value<string>("ssid");
                        m_cameraInfo.ipInfo.WirelessPasswd = wireless.Value<string>("passwd");

                        if (apMode)
                        {
                            m_cameraInfo.ipInfo.WirelessApSsid = m_cameraInfo.ipInfo.WirelessSsid;
                            m_cameraInfo.ipInfo.WirelessApPasswd = m_cameraInfo.ipInfo.WirelessPasswd;
                        }
                        else
                        {
                            m_cameraInfo.ipInfo.WirelessStaSsid = m_cameraInfo.ipInfo.WirelessSsid;
                            m_cameraInfo.ipInfo.WirelessStaPasswd = m_cameraInfo.ipInfo.WirelessPasswd;
                        }
                    }

                    string wirelessEnc = wireless.Value<string>("enc");
                    SLcamNet.NetCamera.LOG($"Wireless Encryption: {wirelessEnc}");
                }
                catch (Exception e)
                {
                    Console.Error.WriteLine($"GetWirelessInfo failed, JSON error: {e.Message}");
                }

                SLcamNet.NetCamera.LOG($"getWirelessInfo: {wirelessInfoJson.ToString()}");
            }

            public virtual void GetWirelessIpv4Info()
            {
                JObject wirelessIpv4InfoJson = GetWirelessIpv4InfoJson();
                if (wirelessIpv4InfoJson == null)
                    return;

                try
                {
                    var wirelessIpv4 = wirelessIpv4InfoJson["wireless_ipv4"];
                    if (wirelessIpv4 == null)
                        throw new Exception("Wireless IPv4 data is missing in JSON.");

                    m_cameraInfo.ipInfo.WirelessDhcp = wirelessIpv4["dhcp"]?.ToObject<bool>() ?? false;
                    m_cameraInfo.ipInfo.WirelessIp = wirelessIpv4["ip"]?.ToString();
                    m_cameraInfo.ipInfo.WirelessGateway = wirelessIpv4["gateway"]?.ToString();
                    m_cameraInfo.ipInfo.WirelessNetmask = wirelessIpv4["netmask"]?.ToString();
                    m_cameraInfo.ipInfo.WirelessDns1 = wirelessIpv4["dns1"]?.ToString();
                    m_cameraInfo.ipInfo.WirelessDns2 = wirelessIpv4["dns2"]?.ToString();
                    m_cameraInfo.ipInfo.WirelessMac = wirelessIpv4["mac"]?.ToString();
                }
                catch (Exception e)
                {
                    Console.Error.WriteLine($"GetWirelessIpv4Info failed, JSON error: {e.Message}");
                }

                SLcamNet.NetCamera.LOG($"GetWirelessIpv4Info: {wirelessIpv4InfoJson.ToString()}");
            }

            public virtual void GetFocusInfo()
            {
                JObject imgRecJson = GetControlInfoJson();
                // focus_value,bright,contrast,saturation,hue,sharpness,ezoom_value  M系列，C系列 这几个值均通过GetControlInfoJson接口获取
                int focusMode = imgRecJson["get_image"]["focus_mode"].ToObject<int>();
                m_cameraInfo.Focus.Mode = ConvertFromNetFocusMode(focusMode);

                m_cameraInfo.Focus.Value.Cur =
                    imgRecJson["get_image"]["focus_value"].ToObject<int>();
            }

            public virtual void GetExposureInfo()
            {
                var exposureRecJson = GetExposureInfoJson();
                if (exposureRecJson == null)
                    return;
                var imgRecJson = GetControlInfoJson();
                if (imgRecJson == null)
                    return;
                try
                {
                    m_cameraInfo.Exposure.Bright.Cur =
                        imgRecJson["get_image"]["bright"].ToObject<int>();
                    int mode = exposureRecJson["img_meexposure"]["expo_type"].ToObject<int>();
                    m_cameraInfo.Exposure.Mode = ConvertFromNetExposureMode(mode);
                    m_cameraInfo.Exposure.Time.Cur =
                        exposureRecJson["img_meexposure"]["shuttertime"].ToObject<int>();
                    m_cameraInfo.Exposure.Gain.Cur =
                        exposureRecJson["img_meexposure"]["a_gain"].ToObject<int>();
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                }
            }

            public virtual void GetWbMode()
            {
                JObject awbRecJson = GetWbModeJson();
                if (awbRecJson == null)
                    return;

                try
                {
                    string wbMode = awbRecJson["img_awb"]["awb"].ToString();
                    m_cameraInfo.WhiteBalance.Mode = ConvertFromNetWhiteBalanceMode(wbMode);
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                }
            }

            public virtual void GetWbInfo()
            {
                GetWbMode();
                JObject wbInfoJson = GetWbInfoJson();
                if (wbInfoJson == null)
                    return;

                try
                {
                    m_cameraInfo.WhiteBalance.Red.Cur =
                        wbInfoJson["info_img_awb"]["r"].ToObject<int>();
                    m_cameraInfo.WhiteBalance.Green.Cur =
                        wbInfoJson["info_img_awb"]["gb"].ToObject<int>();
                    m_cameraInfo.WhiteBalance.Blue.Cur =
                        wbInfoJson["info_img_awb"]["b"].ToObject<int>();
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                }
            }

            public void GetFlipMirrorStatus()
            {
                JObject mirrorFlipRecJson = GetFlipMirrorStatusJson();
                if (mirrorFlipRecJson == null)
                    return;

                try
                {
                    m_cameraInfo.Mirror =
                        mirrorFlipRecJson["img_mirror_flip"]["mirror"].ToObject<bool>();
                    m_cameraInfo.Flip =
                        mirrorFlipRecJson["img_mirror_flip"]["flip"].ToObject<bool>();
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                }
            }

            public void GetFlickStatus()
            {
                JObject flickRecJson = GetFlickStatusJson();
                if (flickRecJson == null)
                    return;

                try
                {
                    m_cameraInfo.PowerlineFrequency =
                        flickRecJson["img_flick_hz"]["flick"].ToObject<bool>();
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                }
            }

            public void GetGammaValue()
            {
                JObject hdrRecJson = GetGammaValueJson();
                if (hdrRecJson == null)
                    return;

                try
                {
                    m_cameraInfo.Gamma.Cur =
                        hdrRecJson["img_hdr_gamma"]["level"].ToObject<int>();
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                }
            }

            public virtual void GetDefogStatus()
            {
                JObject defogRecJson = GetDefogStatusJson();
                if (defogRecJson == null)
                    return;

                try
                {
                    m_cameraInfo.Defog =
                        defogRecJson["motor_demist"]["demist"].ToObject<bool>();
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                }
            }

            public void GetDRCInfo()
            {
                JObject drcInfo = GetDRCInfoJson();
                if (drcInfo == null)
                    return;

                try
                {
                    m_cameraInfo.Drc.Cur = (int)drcInfo["img_drc"]["drc"];
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("getDRCInfo failed, throw an error: " + e.Message);
                }
            }

            public JObject GetDRCInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "img_drc",
                    ["img_drc"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };

                return GetRequest(request);
            }

            public void GetCACInfo()
            {
                JObject cacInfo = GetCACInfoJson();
                if (cacInfo == null)
                    return;

                try
                {
                    m_cameraInfo.Cac = (bool)cacInfo["img_cac"]["cac"];
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("getCACInfo failed, throw an error: " + e.Message);
                }
            }

            public JObject GetCACInfoJson()
            {
                // 构造请求 JSON 数据
                var request = new JObject
                {
                    ["cmd"] = "img_cac",
                    ["img_cac"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };

                return GetRequest(request);
            }

            public virtual void GetSceneInfo()
            {
                JObject sceneInfo = GetSceneInfoJson();
                if (sceneInfo == null)
                    return;

                try
                {
                    m_cameraInfo.SceneMode = (int)sceneInfo["img_mode"]["mode"];
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("getSceneInfo failed, throw an error: " + e.Message);
                }
            }

            public JObject GetSceneInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "img_mode",
                    ["img_mode"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };

                return GetRequest(request);
            }

            public virtual void GetIpv4Info()
            {
                JObject ipv4InfoJson = GetIpv4InfoJson();
                if (ipv4InfoJson == null)
                    return;

                try
                {
                    var ipv4 = ipv4InfoJson["ipv4"];
                    if (ipv4 == null)
                        throw new Exception("IPv4 data is missing in JSON.");

                    m_cameraInfo.ipInfo.Dhcp = ipv4["dhcp"]?.ToObject<bool>() ?? false;
                    m_cameraInfo.ipInfo.Ip = ipv4["ip"]?.ToString();
                    m_cameraInfo.ipInfo.Gateway = ipv4["gateway"]?.ToString();
                    m_cameraInfo.ipInfo.Netmask = ipv4["netmask"]?.ToString();
                    m_cameraInfo.ipInfo.Dns1 = ipv4["dns1"]?.ToString();
                    m_cameraInfo.ipInfo.Dns2 = ipv4["dns2"]?.ToString();
                }
                catch (Exception e)
                {
                    Console.Error.WriteLine($"GetIpv4Info failed, JSON error: {e.Message}");
                }

                SLcamNet.NetCamera.LOG($"GetIpv4Info: {ipv4InfoJson.ToString()}");
            }

            public bool GetControlRangeInfo()
            {
                try
                {
                    var info = GetControlRangeInfoJson();

                    m_cameraInfo.Exposure.Bright.Def = info["default_image"]["bright"].Value<int>();
                    m_cameraInfo.Exposure.Bright.Min = info["default_image"]["range"]["bright"]["min"].Value<int>();
                    m_cameraInfo.Exposure.Bright.Max = info["default_image"]["range"]["bright"]["max"].Value<int>();

                    m_cameraInfo.Hue.Def = info["default_image"]["hue"].Value<int>();
                    m_cameraInfo.Hue.Min = info["default_image"]["range"]["hue"]["min"].Value<int>();
                    m_cameraInfo.Hue.Max = info["default_image"]["range"]["hue"]["max"].Value<int>();

                    m_cameraInfo.Contrast.Def = info["default_image"]["contrast"].Value<int>();
                    m_cameraInfo.Contrast.Min = info["default_image"]["range"]["contrast"]["min"].Value<int>();
                    m_cameraInfo.Contrast.Max = info["default_image"]["range"]["contrast"]["max"].Value<int>();

                    m_cameraInfo.Saturation.Def = info["default_image"]["saturation"].Value<int>();
                    m_cameraInfo.Saturation.Min = info["default_image"]["range"]["saturation"]["min"].Value<int>();
                    m_cameraInfo.Saturation.Max = info["default_image"]["range"]["saturation"]["max"].Value<int>();

                    m_cameraInfo.Sharpness.Def = info["default_image"]["sharpness"].Value<int>();
                    m_cameraInfo.Sharpness.Min = info["default_image"]["range"]["sharpness"]["min"].Value<int>();
                    m_cameraInfo.Sharpness.Max = info["default_image"]["range"]["sharpness"]["max"].Value<int>();

                    m_cameraInfo.Focus.Value.Def = info["default_image"]["focus_value"].Value<int>();
                    m_cameraInfo.Focus.Value.Min = info["default_image"]["range"]["focus_value"]["min"].Value<int>();
                    m_cameraInfo.Focus.Value.Max = info["default_image"]["range"]["focus_value"]["max"].Value<int>();

                    return true;
                }
                catch (Exception ex)
                {
                    SLcamNet.NetCamera.LOG("getControlRangeInfo failed, " + ex.Message);
                    return false;
                }
            }

            private JObject GetControlRangeInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_image",
                    ["default_image"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };
                return GetRequest(request);
            }

            public void GetFocusRegionRangeInfo()
            {
                // {
                //     "cmd": "default_img_roi",
                //     "default_img_roi": {
                //         "chnn": 0,
                //         "code": 0,
                //         "range": {
                //             "roi_big_x": {
                //                 "max": 4,
                //                 "min": 0
                //             },
                //             "roi_big_y": {
                //                 "max": 4,
                //                 "min": 0
                //             },
                //             "roi_x": {
                //                 "max": 16,
                //                 "min": 0
                //             },
                //             "roi_y": {
                //                 "max": 14,
                //                 "min": 0
                //             }
                //         },
                //         "roi": true,
                //         "roi_mode": 0,
                //         "roi_x": 8,
                //         "roi_y": 7
                //     }
                // }
                JObject info = GetFocusRegionRangeInfoJson();

                try
                {
                    m_cameraInfo.Focus.Region.X.Def = info["default_img_roi"]["roi_x"].Value<int>();
                    m_cameraInfo.Focus.Region.X.Min = info["default_img_roi"]["range"]["roi_x"]["min"].Value<int>();
                    m_cameraInfo.Focus.Region.X.Max = info["default_img_roi"]["range"]["roi_x"]["max"].Value<int>();

                    m_cameraInfo.Focus.Region.Y.Def = info["default_img_roi"]["roi_y"].Value<int>();
                    m_cameraInfo.Focus.Region.Y.Min = info["default_img_roi"]["range"]["roi_y"]["min"].Value<int>();
                    m_cameraInfo.Focus.Region.Y.Max = info["default_img_roi"]["range"]["roi_y"]["max"].Value<int>();
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("GetFocusRegionRangeInfo failed, " + e.Message);
                }
            }

            public JObject GetFocusRegionRangeInfoJson()
            {
                JObject request = new JObject
                {
                    ["cmd"] = "default_img_roi",
                    ["default_img_roi"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };
                return GetRequest(request);
            }

            public void GetMExposureRangeInfo()
            {
                JObject info = GetMExposureRangeInfoJson();

                try
                {
                    // 提取曝光增益值
                    m_cameraInfo.Exposure.Gain.Def = info["default_img_meexposure"]["a_gain"].Value<int>();
                    m_cameraInfo.Exposure.Gain.Min =
                        info["default_img_meexposure"]["range"]["a_gain"]["min"].Value<int>();
                    m_cameraInfo.Exposure.Gain.Max =
                        info["default_img_meexposure"]["range"]["a_gain"]["max"].Value<int>();

                    // 提取曝光时间值
                    m_cameraInfo.Exposure.Time.Def = info["default_img_meexposure"]["shuttertime"].Value<int>();
                    m_cameraInfo.Exposure.Time.Min =
                        info["default_img_meexposure"]["range"]["shuttertime"]["min"].Value<int>();
                    m_cameraInfo.Exposure.Time.Max =
                        info["default_img_meexposure"]["range"]["shuttertime"]["max"].Value<int>();
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG($"getMExposureRangeInfo failed: {e.Message}");
                }
            }

            public JObject GetMExposureRangeInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_img_meexposure",
                    ["default_img_meexposure"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };
                return GetRequest(request);
            }

            public void GetWbRangeInfo()
            {
                var info = GetWbRangeInfoJson();
                try
                {
                    m_cameraInfo.WhiteBalance.Red.Def = info["default_img_awb"]["r"].Value<int>();
                    m_cameraInfo.WhiteBalance.Red.Min = info["default_img_awb"]["range"]["r"]["min"].Value<int>();
                    m_cameraInfo.WhiteBalance.Red.Max = info["default_img_awb"]["range"]["r"]["max"].Value<int>();

                    m_cameraInfo.WhiteBalance.Green.Def = info["default_img_awb"]["gr"].Value<int>();
                    m_cameraInfo.WhiteBalance.Green.Min = info["default_img_awb"]["range"]["gr"]["min"].Value<int>();
                    m_cameraInfo.WhiteBalance.Green.Max = info["default_img_awb"]["range"]["gr"]["max"].Value<int>();

                    m_cameraInfo.WhiteBalance.Blue.Def = info["default_img_awb"]["b"].Value<int>();
                    m_cameraInfo.WhiteBalance.Blue.Min = info["default_img_awb"]["range"]["b"]["min"].Value<int>();
                    m_cameraInfo.WhiteBalance.Blue.Max = info["default_img_awb"]["range"]["b"]["max"].Value<int>();
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG($"getWbRangeInfo failed: {e.Message}");
                }
            }

            public JObject GetWbRangeInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_img_awb",
                    ["default_img_awb"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };

                return GetRequest(request);
            }

            public void GetGammaRangeInfo()
            {
                JObject info = GetGammaRangeInfoJson();
                try
                {
                    m_cameraInfo.Gamma.Def = (int)info["default_img_hdr_gamma"]["level"];
                    m_cameraInfo.Gamma.Min = (int)info["default_img_hdr_gamma"]["range"]["level"]["min"];
                    m_cameraInfo.Gamma.Max = (int)info["default_img_hdr_gamma"]["range"]["level"]["max"];
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("getGammaRangeInfoJson failed, " + e.Message);
                }
            }

            public JObject GetGammaRangeInfoJson()
            {
                // 构造请求的 JSON 数据
                var request = new JObject
                {
                    ["cmd"] = "default_img_hdr_gamma",
                    ["default_img_hdr_gamma"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };

                return GetRequest(request);
            }

            public void GetDRCRangeInfo()
            {
                JObject info = GetDRCRangeInfoJson();
                try
                {
                    m_cameraInfo.Drc.Def = (int)info["default_img_drc"]["drc"];
                    m_cameraInfo.Drc.Min = (int)info["default_img_drc"]["range"]["drc"]["min"];
                    m_cameraInfo.Drc.Max = (int)info["default_img_drc"]["range"]["drc"]["max"];
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("getDRCRangeInfoJson failed, " + e.Message);
                }
            }

            public JObject GetDRCRangeInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_img_drc",
                    ["default_img_drc"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };

                return GetRequest(request);
            }

            public void GetWbTemperatureRangeInfo()
            {
                JObject info = GetWbTemperatureRangeInfoJson();
                try
                {
                    m_cameraInfo.WhiteBalance.Temperature.Def = (int)info["default_img_color_temp"]["color_temp"];
                    m_cameraInfo.WhiteBalance.Temperature.Min =
                        (int)info["default_img_color_temp"]["range"]["color_temp"]["min"];
                    m_cameraInfo.WhiteBalance.Temperature.Max =
                        (int)info["default_img_color_temp"]["range"]["color_temp"]["max"];
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("getWbTemperatureRangeInfo failed, " + e.Message);
                }
            }

            public JObject GetWbTemperatureRangeInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_img_color_temp",
                    ["default_img_color_temp"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };

                return GetRequest(request);
            }

            public void GetWbTemperatureInfo()
            {
                JObject info = GetWbTemperatureInfoJson();
                if (info == null)
                    return;

                try
                {
                    m_cameraInfo.WhiteBalance.Temperature.Cur = (int)info["img_color_temp"]["color_temp"];
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("getWbTemperatureInfo failed, throw an error: " + e.Message);
                }
            }

            public JObject GetWbTemperatureInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "img_color_temp",
                    ["img_color_temp"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };

                return GetRequest(request);
            }

            public JObject getStreamInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "stream",
                    ["stream"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };
                return GetRequest(request);
            }

            public void GetSystemBaseInfo()
            {
                JObject systemBaseJson = GetSystemBaseJson();
                if (systemBaseJson == null)
                    return;

                try
                {
                    var system = systemBaseJson["system"];
                    if (system == null)
                        throw new Exception("System data is missing in JSON.");

                    m_cameraInfo.Model = system["model"]?.ToString();

                    if (system["model_suffix"] != null)
                    {
                        m_cameraInfo.ModelSuffix = system["model_suffix"]?.ToString();
                    }

                    if (system["wifi"] != null)
                    {
                        m_cameraInfo.ipInfo.EnWireless = system["wifi"]?.ToObject<bool>() ?? false;
                    }
                }
                catch (Exception e)
                {
                    Console.Error.WriteLine($"GetSystemBaseInfo failed, JSON error: {e.Message}");
                }

                SLcamNet.NetCamera.LOG($"GetSystemBaseInfo: {systemBaseJson.ToString()}");
            }

            public bool GetUpgradeStatus(out string status, out double percentage)
            {
                status = string.Empty;
                percentage = 0.0;

                JObject upgradeStatusJson = GetUpgradeStatusJson();
                if (upgradeStatusJson == null)
                    return false;

                try
                {
                    status = upgradeStatusJson["status_upgrade"]["status"].ToString();
                    percentage = upgradeStatusJson["status_upgrade"]["percentage"].ToObject<double>();
                    return true;
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG("json throw an error:" + e.Message);
                    return false;
                }
            }

            public void GetSupportFuncInfo()
            {
                var request = new JObject
                {
                    ["cmd"] = "support"
                };

                var response = GetRequest(request);

                SLcamNet.NetCamera.LOG("getSupportFuncInfo, " + response.ToString());
            }

            public JObject GetStreamJson()
            {
                JObject request = new JObject
                {
                    { "cmd", "get_stream" },
                    {
                        "get_stream", new JObject
                        {
                            { "chnn", m_chnn },
                            { "idx", m_idx }
                        }
                    }
                };

                return GetRequest(request);
            }

            public virtual bool SetSharpnessEnhancement(bool enable)
            {
                Console.WriteLine("Current camera does not support setting sharpness enhancement");
                return false;
            }

            public virtual bool SetContrastEnhancementValue(int value)
            {
                Console.WriteLine("Current camera does not support setting contrast enhancement");
                return false;
            }

            public static int ConvertNetFocusMode(FocusMode mode)
            {
                switch (mode)
                {
                    case FocusMode.kFocusAuto:
                        return 0;
                    case FocusMode.kFocusManual:
                        return 1;
                    case FocusMode.kFocusOnce:
                        return 2;
                    default:
                        return -1;
                }
            }

            public static int ConvertNetExposureMode(ExposureMode mode)
            {
                switch (mode)
                {
                    case ExposureMode.kExpAuto:
                        return 0;
                    case ExposureMode.kExpManual:
                        return 1;
                    default:
                        return -1;
                }
            }

            public string ConvertNetWhiteBalanceMode(WbMode mode)
            {
                switch (mode)
                {
                    case WbMode.kWBManual:
                        return "manual";
                    case WbMode.kWBAuto:
                        return "auto";
                    default:
                        return "auto";
                }
            }

            public static int ConvertNetWhiteBalanceMode2(WbMode mode)
            {
                switch (mode)
                {
                    case WbMode.kWBManual:
                        return 1;
                    case WbMode.kWBAuto:
                        return 0;
                    default:
                        return -1;
                }
            }

            public static int VersionFromSoftwareVer(string softwareVer)
            {
                int version = 0;
                var matches = Regex.Matches(softwareVer, @"\d+");

                foreach (Match match in matches)
                {
                    version = version * 10 + int.Parse(match.Value);
                }

                return version;
            }

            public FocusMode ConvertFromNetFocusMode(int mode)
            {
                switch (mode)
                {
                    case 0:
                        return FocusMode.kFocusAuto;
                    case 1:
                        return FocusMode.kFocusManual;
                    case 2:
                        return FocusMode.kFocusOnce;
                    default:
                        return FocusMode.kFocusAuto;
                }
            }

            public ExposureMode ConvertFromNetExposureMode(int mode)
            {
                switch (mode)
                {
                    case 0:
                        return ExposureMode.kExpAuto;
                    case 1:
                        return ExposureMode.kExpManual;
                    default:
                        return ExposureMode.kExpAuto;
                }
            }

            public WbMode ConvertFromNetWhiteBalanceMode(string mode)
            {
                switch (mode)
                {
                    case "manual":
                        return WbMode.kWBManual;
                    case "auto":
                        return WbMode.kWBAuto;
                    default:
                        return WbMode.kWBAuto;
                }
            }

            public WbMode ConvertFromNetWhiteBalanceMode(int mode)
            {
                switch (mode)
                {
                    case 0:
                        return WbMode.kWBAuto;
                    case 1:
                        return WbMode.kWBManual;
                    default:
                        return WbMode.kWBAuto;
                }
            }

            public void GetFocusRegionInfo()
            {
                JObject focusRegionJson = GetFocusRegionInfoJson();
                if (focusRegionJson == null)
                    return;

                try
                {
                    m_cameraInfo.Focus.Region.Enable = (bool)focusRegionJson["img_roi"]["roi"];
                    m_cameraInfo.Focus.Region.Mode = (int)focusRegionJson["img_roi"]["roi_mode"];
                    m_cameraInfo.Focus.Region.X.Cur = (int)focusRegionJson["img_roi"]["roi_x"];
                    m_cameraInfo.Focus.Region.Y.Cur = (int)focusRegionJson["img_roi"]["roi_y"];
                }
                catch (Exception e)
                {
                    SLcamNet.NetCamera.LOG($"JSON parsing error: {e.Message}");
                }
            }

            public JObject GetControlInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "get_image",
                    ["get_image"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };
                return GetRequest(request);
            }

            public JObject GetOldControlInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "image",
                    ["image"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };
                return GetRequest(request);
            }

            public virtual JObject GetWirelessInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "wireless"
                };
                return GetRequest(request);
            }

            public virtual JObject GetWirelessIpv4InfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "wireless_ipv4"
                };
                return GetRequest(request);
            }

            public JObject GetExposureInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "img_meexposure",
                    ["img_meexposure"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };
                return GetRequest(request);
            }

            public JObject GetWbModeJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "img_awb",
                    ["img_awb"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };
                return GetRequest(request);
            }

            public JObject GetWbInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "info_img_awb",
                    ["info_img_awb"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };
                return GetRequest(request);
            }

            public JObject GetFlipMirrorStatusJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "img_mirror_flip",
                    ["img_mirror_flip"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };
                return GetRequest(request);
            }

            public JObject GetFlickStatusJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "img_flick_hz",
                    ["img_flick_hz"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };
                return GetRequest(request);
            }

            public JObject GetGammaValueJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "img_hdr_gamma",
                    ["img_hdr_gamma"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };
                return GetRequest(request);
            }

            public virtual JObject GetDefogStatusJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "motor_demist",
                    ["motor_demist"] = new JObject
                    {
                        ["chnn"] = m_chnn
                    }
                };
                return GetRequest(request);
            }

            public virtual JObject GetIpv4InfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "ipv4"
                };
                return GetRequest(request);
            }

            public JObject GetSystemBaseJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "system"
                };
                return GetRequest(request);
            }

            public JObject GetUpgradeStatusJson()
            {
                var requestJson = new JObject
                {
                    ["cmd"] = "status_upgrade",
                    ["ip"] = m_cameraSettings.netInfo.Ip,
                    ["port"] = m_cameraSettings.netInfo.Port
                };

                string request = requestJson.ToString(Formatting.None);
                IntPtr resultPtr = IntPtr.Zero;

                // 网络升级时，登录Id必须设置为0
                int ret = SLcamNet.l_sdk_request(0, request, out resultPtr);

                if (ret != 0 || resultPtr == IntPtr.Zero)
                {
                    SLcamNet.NetCamera.LOG("getUpgradeStatusJson failed");
                    return null;
                }

                string resultStr = Marshal.PtrToStringAnsi(resultPtr);
                SLcamNet.l_sdk_free(resultPtr);

                JObject resJson = null;
                int code = -1;
                if (!string.IsNullOrEmpty(resultStr))
                {
                    try
                    {
                        resJson = JObject.Parse(resultStr);
                        string cmd = requestJson["cmd"].ToString();
                        code = (int)resJson[cmd]["code"];
                    }
                    catch (Exception e)
                    {
                        SLcamNet.NetCamera.LOG("JSON parsing error: " + e.Message);
                        return null;
                    }
                }

                if (code != 0)
                {
                    SLcamNet.NetCamera.LOG("getUpgradeStatusJson failed");
                    return null;
                }

                return resJson;
            }

            // GetFocusRegionInfoJson 方法实际上是上面方法的一部分，但为了完整性，我们也单独实现
            public JObject GetFocusRegionInfoJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "img_roi",
                    ["img_roi"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };
                return GetRequest(request);
            }

            public JObject GetNameJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "name"
                };
                return GetRequest(request);
            }

            public JObject GetNtpJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "ntp"
                };
                return GetRequest(request);
            }

            public JObject GetDefaultStreamJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_stream",
                    ["default_stream"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };
                return GetRequest(request);
            }

            public JObject GetDefaultIpv4Json()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_ipv4",
                    ["default_ipv4"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };
                return GetRequest(request);
            }

            public JObject GetDefaultWirelessJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_wireless",
                    ["default_wireless"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };
                return GetRequest(request);
            }

            public JObject GetDefaultWirelessIpv4Json()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_wireless_ipv4",
                    ["default_wireless_ipv4"] = new JObject
                    {
                        ["chnn"] = m_chnn,
                        ["idx"] = m_idx
                    }
                };
                return GetRequest(request);
            }

            public JObject GetDefaultNameJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_name"
                };
                return GetRequest(request);
            }

            public JObject GetDefaultNtpJson()
            {
                var request = new JObject
                {
                    ["cmd"] = "default_ntp"
                };
                return GetRequest(request);
            }

            public static void LOG(string s)
            {
                Console.WriteLine(s);
            }
        }
    }
}