import ctypes
import filecmp
import numpy as np
import os
import platform
import sys
import shutil
from OpenGL import GL
from PyQt5.QtCore import Qt, QTimer, QSignalBlocker, QElapsedTimer
from PyQt5.QtWidgets import QApplication, QWidget, QVBoxLayout, QGridLayout, QPushButton, QComboBox, \
    QGroupBox, QCheckBox, QSlider, QLabel, QFileDialog, QMessageBox, QHBoxLayout, QSizePolicy, QOpenGLWidget
from PyQt5.QtGui import QImage

import slncam

def copy_files(source_folders, dest_folder):
    os.makedirs(dest_folder, exist_ok=True)

    for sources_folder in source_folders:
        for root, dirs, files in os.walk(sources_folder):
            for file in files:
                source_path = os.path.join(root, file)
                relative_path = os.path.relpath(root, sources_folder)

                if relative_path == '.':
                    relative_path = ''

                dest_dir = os.path.join(dest_folder, relative_path)
                os.makedirs(dest_dir, exist_ok=True)

                dest_path = os.path.join(dest_dir, file)

                if not os.path.exists(dest_path) or not filecmp.cmp(source_path, dest_path, shallow=False):
                    try:
                        shutil.copy2(source_path, dest_path)
                        # print(f'Copy succeed {source_path} -> {dest_path}')
                    except Exception as e:
                        print(f'Copy failed {source_path}: {e}')

def i420_to_rgb(frame: slncam.SLNcamVideoFrame):
    rgb_data = np.zeros((frame.resolution.h, frame.resolution.w, 3), dtype=np.uint8)
    y_plane = np.ctypeslib.as_array(
        ctypes.cast(frame.data.yuv_data.p_y, ctypes.POINTER(ctypes.c_ubyte)),
        shape=(frame.resolution.h, frame.data.yuv_data.pitch_y)
    )
    u_plane = np.ctypeslib.as_array(
        ctypes.cast(frame.data.yuv_data.p_u, ctypes.POINTER(ctypes.c_ubyte)),
        shape=(frame.resolution.h // 2, frame.data.yuv_data.pitch_u)
    )
    v_plane = np.ctypeslib.as_array(
        ctypes.cast(frame.data.yuv_data.p_v, ctypes.POINTER(ctypes.c_ubyte)),
        shape=(frame.resolution.h // 2, frame.data.yuv_data.pitch_v)
    )

    u_plane = np.repeat(u_plane, 2, 0)
    u_plane = np.repeat(u_plane, 2, 1)
    v_plane = np.repeat(v_plane, 2, 0)
    v_plane = np.repeat(v_plane, 2, 1)

    c = (y_plane - np.array([16])) * 298
    d = u_plane - np.array([128])
    e = v_plane - np.array([128])

    r = (c + 409 * e + 128) // 256
    g = (c - 100 * d - 208 * e + 128) // 256
    b = (c + 516 * d + 128) // 256

    r = np.where(r < 0, 0, r)
    r = np.where(r > 255,255,r)

    g = np.where(g < 0, 0, g)
    g = np.where(g > 255,255,g)

    b = np.where(b < 0, 0, b)
    b = np.where(b > 255,255,b)

    rgb_data[:, :, 0] = r
    rgb_data[:, :, 1] = g
    rgb_data[:, :, 2] = b

    return rgb_data
class OpenGLVideoWidget(QOpenGLWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.program = None
        self.textures = [0, 0, 0]
        self.video_size = (0, 0)
        self.aspect_ratio = 16 / 9

        self.vertices = np.array([
            -1.0, -1.0,
            1.0, -1.0,
            -1.0, 1.0,
            1.0, 1.0
        ], dtype=np.float32)

        self.tex_coords = np.array([
            0.0, 1.0,
            1.0, 1.0,
            0.0, 0.0,
            1.0, 0.0
        ], dtype=np.float32)

    def initializeGL(self):
        self.program = GL.glCreateProgram()
        vertex_shader = GL.glCreateShader(GL.GL_VERTEX_SHADER)
        vs_source = '''
           #version 330 core
           layout (location = 0) in vec4 vertexIn;
           layout (location = 1) in vec2 textureIn;
           out vec2 textureOut;
           void main(void)
           {
               gl_Position = vertexIn;
               textureOut = textureIn;
           }
           '''
        GL.glShaderSource(vertex_shader, vs_source)
        GL.glCompileShader(vertex_shader)
        GL.glAttachShader(self.program, vertex_shader)

        fragment_shader = GL.glCreateShader(GL.GL_FRAGMENT_SHADER)
        fs_source = '''
           #version 330 core
           in vec2 textureOut;
           uniform sampler2D tex_y;
           uniform sampler2D tex_u;
           uniform sampler2D tex_v;
           out vec4 FragColor;
           void main(void)
           {
               vec3 yuv;
               vec3 rgb;
               yuv.x = texture(tex_y, textureOut).r - 0.063;
               yuv.y = texture(tex_u, textureOut).r - 0.5;
               yuv.z = texture(tex_v, textureOut).r - 0.5;
               rgb = mat3(1.164,  1.164,  1.164,
                          0.0, -0.392,  2.017,
                          1.596, -0.813,  0.0) * yuv;
               FragColor = vec4(rgb, 1.0);
           }
           '''
        GL.glShaderSource(fragment_shader, fs_source)
        GL.glCompileShader(fragment_shader)
        GL.glAttachShader(self.program, fragment_shader)

        GL.glLinkProgram(self.program)

        self.textures = GL.glGenTextures(3)
        for i in range(3):
            GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[i])
            GL.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_MAG_FILTER, GL.GL_LINEAR)
            GL.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_MIN_FILTER, GL.GL_LINEAR)
            GL.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_WRAP_S, GL.GL_CLAMP_TO_EDGE)
            GL.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_WRAP_T, GL.GL_CLAMP_TO_EDGE)

    def paintGL(self):
        if self.video_size == (0, 0):
            return

        GL.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT)
        GL.glUseProgram(self.program)

        view_w, view_h = self.width(), self.height()
        video_ratio = self.aspect_ratio

        if view_w / view_h > video_ratio:
            render_h = view_h
            render_w = int(render_h * video_ratio)
        else:
            render_w = view_w
            render_h = int(render_w / video_ratio)

        x = (view_w - render_w) // 2
        y = (view_h - render_h) // 2
        GL.glViewport(x, y, render_w, render_h)

        vertex_attr = GL.glGetAttribLocation(self.program, 'vertexIn')
        GL.glEnableVertexAttribArray(vertex_attr)
        GL.glVertexAttribPointer(vertex_attr, 2, GL.GL_FLOAT, GL.GL_FALSE, 0, self.vertices)

        tex_attr = GL.glGetAttribLocation(self.program, 'textureIn')
        GL.glEnableVertexAttribArray(tex_attr)
        GL.glVertexAttribPointer(tex_attr, 2, GL.GL_FLOAT, GL.GL_FALSE, 0, self.tex_coords)

        for i, name in enumerate(['tex_y', 'tex_u', 'tex_v']):
            GL.glActiveTexture(GL.GL_TEXTURE0 + i)
            GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[i])
            GL.glUniform1i(GL.glGetUniformLocation(self.program, name), i)

        GL.glDrawArrays(GL.GL_TRIANGLE_STRIP, 0, 4)
        GL.glFlush()

    def resizeGL(self, w, h):
        GL.glViewport(0, 0, w, h)

    def update_yuv(self, y_data, u_data, v_data, width, height):
        if (width, height) != self.video_size:
            self.init_textures(width, height)
            self.video_size = (width, height)
            self.aspect_ratio = width / height

        self.update_texture_data(y_data, u_data, v_data)
        self.update()

    def init_textures(self, width, height):
        GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[0])
        GL.glTexImage2D(GL.GL_TEXTURE_2D, 0, GL.GL_RED,
                        width, height, 0, GL.GL_RED,
                        GL.GL_UNSIGNED_BYTE, None)
        GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[1])
        GL.glTexImage2D(GL.GL_TEXTURE_2D, 0, GL.GL_RED,
                        width // 2, height // 2, 0, GL.GL_RED,
                        GL.GL_UNSIGNED_BYTE, None)
        GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[2])
        GL.glTexImage2D(GL.GL_TEXTURE_2D, 0, GL.GL_RED,
                        width // 2, height // 2, 0, GL.GL_RED,
                        GL.GL_UNSIGNED_BYTE, None)

    def update_texture_data(self, y_data, u_data, v_data):
        self.makeCurrent()

        y_ptr = y_data.ctypes.data_as(ctypes.POINTER(ctypes.c_ubyte))
        u_ptr = u_data.ctypes.data_as(ctypes.POINTER(ctypes.c_ubyte))
        v_ptr = v_data.ctypes.data_as(ctypes.POINTER(ctypes.c_ubyte))

        for i, data_ptr in enumerate([y_ptr, u_ptr, v_ptr]):
            GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[i])
            GL.glTexSubImage2D(GL.GL_TEXTURE_2D, 0, 0, 0,
                               self.video_size[0] // (2 if i else 1),
                               self.video_size[1] // (2 if i else 1),
                               GL.GL_RED, GL.GL_UNSIGNED_BYTE, data_ptr)

        self.doneCurrent()

class MainWidget(QWidget):
    def __init__(self):
        super().__init__()
        # Set the library file load path
        slncam.NetDLL.set_load_dll_path(self.load_dll_path())
        slncam.NetCamera.init_net()
        self.camera = None
        self.dev_infos = None
        self.is_capture = False
        self.capture_file_index = 0

        self.frame_count = 0
        self.fps = 0
        self.fps_label = QLabel('FPS: 0.00')

        self.fps_timer = QTimer(self)
        self.fps_timer.timeout.connect(self.update_fps)
        self.fps_timer.start(1000)

        self.update_frame_timer = QTimer(self)
        self.update_frame_timer.timeout.connect(self.read_frame)
        self.setWindowTitle('SLNCamera')
        self.setMinimumSize(1280, 850)
        self.init_ui()

    @staticmethod
    def load_dll_path():
        script_dir = os.path.dirname(os.path.abspath(__file__))
        arch = platform.architecture()[0]
        arch_dir = 'x64' if '64' in arch else 'x86'

        if sys.platform == 'win32':
            platform_dir = 'win'
            dll_dir = 'bin'

        sdk_path = os.path.abspath(os.path.join(script_dir, '../../sdk', platform_dir, 'slncam', dll_dir, arch_dir))
        ffmpeg_path = os.path.abspath(os.path.join(script_dir, '../../sdk', platform_dir, 'FFmpeg', dll_dir, arch_dir))
        run_path = os.path.abspath(os.path.join(script_dir, 'build', arch_dir))
        copy_files([sdk_path, ffmpeg_path], run_path)
        return run_path

    def init_ui(self):
        main_grid = QGridLayout(self)
        main_grid.setColumnStretch(0, 1)
        main_grid.setColumnStretch(1, 4)
        main_grid.setSpacing(0)

        ctrl_panel = QWidget()
        ctrl_panel.setFixedWidth(300)
        ctrl_layout = QVBoxLayout(ctrl_panel)

        self.scan_btn = QPushButton('Scan')
        self.camera_cmb = QComboBox()
        ctrl_layout.addWidget(self.scan_btn)
        ctrl_layout.addWidget(self.camera_cmb)

        # Resolution Setting
        res_group = QGroupBox('Resolution Settings')
        self.res_cmb = QComboBox()
        res_group.setLayout(QVBoxLayout())
        res_group.layout().addWidget(self.res_cmb)

        # Exposure
        exp_group = QGroupBox('Exposure Control')
        self.auto_exp_cbox = QCheckBox('Auto Exposure')
        self.exp_bright_lab = QLabel('0')
        self.exp_time_lab = QLabel('0')
        self.exp_gain_lab = QLabel('0')
        self.exp_bright_slider = QSlider(Qt.Orientation.Horizontal)
        self.exp_time_slider = QSlider(Qt.Orientation.Horizontal)
        self.exp_gain_slider = QSlider(Qt.Orientation.Horizontal)
        self.exp_bright_slider.setTracking(False)
        self.exp_time_slider.setTracking(False)
        self.exp_gain_slider.setTracking(False)
        exp_group.setLayout(QVBoxLayout())
        exp_group.layout().addLayout(self.create_param_layout('Bright', self.exp_bright_slider, self.exp_bright_lab))
        exp_group.layout().addLayout(self.create_param_layout('Time', self.exp_time_slider, self.exp_time_lab))
        exp_group.layout().addLayout(self.create_param_layout('Gain', self.exp_gain_slider, self.exp_gain_lab))
        exp_group.layout().addWidget(self.auto_exp_cbox)

        # White Balance
        wb_group = QGroupBox('White Balance')
        self.auto_wb_cbox = QCheckBox('Auto WB')
        self.red_lab = QLabel('0')
        self.green_lab = QLabel('0')
        self.blue_lab = QLabel('0')
        self.red_slider = QSlider(Qt.Orientation.Horizontal)
        self.green_slider = QSlider(Qt.Orientation.Horizontal)
        self.blue_slider = QSlider(Qt.Orientation.Horizontal)
        self.red_slider.setTracking(False)
        self.green_slider.setTracking(False)
        self.blue_slider.setTracking(False)
        wb_group.setLayout(QVBoxLayout())
        wb_group.layout().addLayout(self.create_param_layout('Red', self.red_slider, self.red_lab))
        wb_group.layout().addLayout(self.create_param_layout('Green', self.green_slider, self.green_lab))
        wb_group.layout().addLayout(self.create_param_layout('Blue', self.blue_slider, self.blue_lab))
        wb_group.layout().addWidget(self.auto_wb_cbox)

        func_group = QGroupBox('Function')
        self.hue_lab = QLabel('0')
        self.saturation_lab = QLabel('0')
        self.contrast_lab = QLabel('0')
        self.sharpness_lab = QLabel('0')
        self.gamma_lab = QLabel('0')
        self.hue_slider = QSlider(Qt.Orientation.Horizontal)
        self.saturation_slider = QSlider(Qt.Orientation.Horizontal)
        self.contrast_slider = QSlider(Qt.Orientation.Horizontal)
        self.sharpness_slider = QSlider(Qt.Orientation.Horizontal)
        self.gamma_slider = QSlider(Qt.Orientation.Horizontal)
        self.hue_slider.setTracking(False)
        self.saturation_slider.setTracking(False)
        self.contrast_slider.setTracking(False)
        self.sharpness_slider.setTracking(False)
        self.gamma_slider.setTracking(False)
        self.power_line_frequency_cbox = QCheckBox('Power Line Frequency')
        self.flip_cbox = QCheckBox('Flip')
        self.mirror_cbox = QCheckBox('Mirror')
        func_vbox_layout = QVBoxLayout()
        func_vbox_layout.addLayout(self.create_param_layout('Hue', self.hue_slider, self.hue_lab))
        func_vbox_layout.addLayout(self.create_param_layout('Saturation', self.saturation_slider, self.saturation_lab))
        func_vbox_layout.addLayout(self.create_param_layout('Contrast', self.contrast_slider, self.contrast_lab))
        func_vbox_layout.addLayout(self.create_param_layout('Sharpness', self.sharpness_slider, self.sharpness_lab))
        func_vbox_layout.addLayout(self.create_param_layout('Gamma', self.gamma_slider, self.gamma_lab))
        func_vbox_layout.addWidget(self.power_line_frequency_cbox)
        func_vbox_layout.addWidget(self.flip_cbox)
        func_vbox_layout.addWidget(self.mirror_cbox)
        func_group.setLayout(func_vbox_layout)

        self.open_btn = QPushButton('Open Camera')
        self.close_btn = QPushButton('Close Camera')
        self.snap_btn = QPushButton('Capture')

        ctrl_layout.addWidget(self.open_btn)
        ctrl_layout.addWidget(self.close_btn)
        ctrl_layout.addWidget(res_group)
        ctrl_layout.addWidget(exp_group)
        ctrl_layout.addWidget(wb_group)
        ctrl_layout.addWidget(func_group)
        ctrl_layout.addWidget(self.snap_btn)
        ctrl_layout.addStretch()

        self.fps_label = QLabel('FPS: 0.00')
        self.fps_label.setStyleSheet('font-size: 14px; color: black;')
        ctrl_layout.addWidget(self.fps_label)
        self.fps_label.setAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignBottom)

        self.video_wiget = OpenGLVideoWidget()
        self.video_wiget.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)

        main_grid.addWidget(ctrl_panel, 0, 0)
        main_grid.addWidget(self.video_wiget, 0, 1)

        self.scan_btn.clicked.connect(self.on_scan_btn_clicked)
        self.open_btn.clicked.connect(self.on_open_btn_clicked)
        self.close_btn.clicked.connect(self.on_close_btn_clicked)
        self.auto_exp_cbox.toggled.connect(self.on_auto_exp_cbox_toggled)
        self.auto_wb_cbox.toggled.connect(self.on_auto_wb_cbox_toggled)
        self.exp_bright_slider.valueChanged.connect(self.on_exp_bright_slider_changed)
        self.exp_time_slider.valueChanged.connect(self.on_exp_time_slider_changed)
        self.exp_gain_slider.valueChanged.connect(self.on_exp_gain_slider_changed)
        self.red_slider.valueChanged.connect(self.on_red_slider_changed)
        self.green_slider.valueChanged.connect(self.on_green_slider_changed)
        self.blue_slider.valueChanged.connect(self.on_blue_slider_changed)
        self.hue_slider.valueChanged.connect(self.on_hue_slider_changed)
        self.saturation_slider.valueChanged.connect(self.on_saturation_slider_changed)
        self.contrast_slider.valueChanged.connect(self.on_contrast_slider_changed)
        self.sharpness_slider.valueChanged.connect(self.on_sharpness_slider_changed)
        self.gamma_slider.valueChanged.connect(self.on_gamma_slider_changed)
        self.power_line_frequency_cbox.toggled.connect(self.on_power_line_frequency_cbox_toggled)
        self.flip_cbox.toggled.connect(self.on_flip_cbox_toggled)
        self.mirror_cbox.toggled.connect(self.on_mirror_cbox_toggled)
        self.camera_cmb.currentIndexChanged.connect(self.on_camera_cmb_index_changed)
        self.res_cmb.currentIndexChanged.connect(self.on_res_cmb_index_changed)
        self.snap_btn.clicked.connect(self.on_snap_btn_clicked)

        self.exp_bright_slider.valueChanged.connect(lambda v: self.exp_bright_lab.setText(str(v)))
        self.exp_time_slider.valueChanged.connect(lambda v: self.exp_time_lab.setText(str(v)))
        self.exp_gain_slider.valueChanged.connect(lambda v: self.exp_gain_lab.setText(str(v)))
        self.red_slider.valueChanged.connect(lambda v: self.red_lab.setText(str(v)))
        self.green_slider.valueChanged.connect(lambda v: self.green_lab.setText(str(v)))
        self.blue_slider.valueChanged.connect(lambda v: self.blue_lab.setText(str(v)))
        self.hue_slider.valueChanged.connect(lambda v: self.hue_lab.setText(str(v)))
        self.saturation_slider.valueChanged.connect(lambda v: self.saturation_lab.setText(str(v)))
        self.contrast_slider.valueChanged.connect(lambda v: self.contrast_lab.setText(str(v)))
        self.sharpness_slider.valueChanged.connect(lambda v: self.sharpness_lab.setText(str(v)))
        self.gamma_slider.valueChanged.connect(lambda v: self.gamma_lab.setText(str(v)))

        self.set_controls_enable(False)

    def create_param_layout(self, label, slider, value_label):
        hbox = QHBoxLayout()
        hbox.addWidget(QLabel(label))
        hbox.addWidget(slider)
        hbox.addSpacing(10)
        hbox.addWidget(value_label)
        return hbox

    def on_scan_btn_clicked(self):
        self.dev_infos = slncam.NetCamera.search_available_cameras()
        self.camera_cmb.clear()
        if len(self.dev_infos) <= 0:
            QMessageBox.information(self, 'No Devices', 'No cameras detected')
            return

        for devInfo in self.dev_infos:
            self.camera_cmb.addItem('%s  %s' % (devInfo.ip, devInfo.model))

    def on_open_btn_clicked(self):
        if self.camera:
            self.on_close_btn_clicked()
            return

        if self.camera_cmb.currentIndex() == -1:
            QMessageBox.warning(self, 'Selection Error', 'No camera selected')
            return

        try:
            dev_index = self.camera_cmb.currentIndex()
            dev_info = self.dev_infos[dev_index]
            
            settings = slncam.CameraStreamSettings(dev_info)
            self.camera = slncam.NetCamera.create(dev_info.model, dev_info.version)
            if not self.camera:
                raise RuntimeError('Camera open failed')

            self.camera.set_stream_settings(settings)
            self.camera.login()
            self.camera_info = self.camera.get_net_camera_all_info()
            capabilities = self.camera_info.video_stream_capabilities

            self.camera_cmb.blockSignals(True)
            self.res_cmb.clear()
            for capability in capabilities:
                if capability.format == slncam.SLNCAM_VIDEO_FORMAT_MJPEG:
                    format_str = 'MJPEG'
                elif capability.format == slncam.SLNCAM_VIDEO_FORMAT_H264:
                    format_str = 'H264'
                capability_str = '%s %s %d' % (format_str, capability.resolution.size, capability.fps)

                self.res_cmb.addItem(capability_str)
            self.camera_cmb.blockSignals(False)

            if self.res_cmb.count() > 0:
                self.res_cmb.setCurrentIndex(0)
                self.on_res_cmb_index_changed(0)
            else:
                return

            self.init_controls()
            self.update_frame_timer.start(10)

        except Exception as e:
            QMessageBox.critical(self, 'Error', str(e))
            self.on_close_btn_clicked()

    def update_fps(self):
        self.fps_label.setText(f'FPS: {self.frame_count:.2f}')
        self.frame_count = 0

    def read_frame(self):
        try:
            frame = self.camera.get_video_frame_data()

            if frame is None:
                return

            if frame.type != slncam.SLNCAM_PIX_FORMAT_YUV420P_SEPARATE:
                slncam.NetCamera.free_video_frame_data(frame)
                return

            self.frame_count += 1

            if self.is_capture:
                self.save_captured_image(frame)
                self.is_capture = False

            y_plane = np.ctypeslib.as_array(
                ctypes.cast(frame.data.yuv_data.p_y, ctypes.POINTER(ctypes.c_ubyte)),
                shape=(frame.resolution.h, frame.data.yuv_data.pitch_y)
            )
            u_plane = np.ctypeslib.as_array(
                ctypes.cast(frame.data.yuv_data.p_u, ctypes.POINTER(ctypes.c_ubyte)),
                shape=(frame.resolution.h // 2, frame.data.yuv_data.pitch_u)
            )
            v_plane = np.ctypeslib.as_array(
                ctypes.cast(frame.data.yuv_data.p_v, ctypes.POINTER(ctypes.c_ubyte)),
                shape=(frame.resolution.h // 2, frame.data.yuv_data.pitch_v)
            )

            self.video_wiget.update_yuv(y_plane, u_plane, v_plane, frame.resolution.w, frame.resolution.h)
            slncam.NetCamera.free_video_frame_data(frame)

        except Exception as e:
            print(f'Frame processing error: {str(e)}')
            # pass

    def closeEvent(self, event):
        self.on_close_btn_clicked()
        slncam.NetCamera.destroy_net()
        super().closeEvent(event)

    def on_close_btn_clicked(self):
        self.update_frame_timer.stop()
        if self.camera:
            self.camera.close_stream()
            self.camera.logout()
            self.camera = None
            self.video_wiget.video_size = (0, 0)
            self.video_wiget.update()
            self.set_controls_enable(False)

    def on_snap_btn_clicked(self):
        self.is_capture = True

    def set_controls_enable(self, enable):
        self.res_cmb.setEnabled(enable)
        self.auto_wb_cbox.setEnabled(enable)
        self.auto_exp_cbox.setEnabled(enable)
        self.exp_bright_slider.setEnabled(enable)
        self.exp_time_slider.setEnabled(enable)
        self.exp_gain_slider.setEnabled(enable)
        self.red_slider.setEnabled(enable)
        self.green_slider.setEnabled(enable)
        self.blue_slider.setEnabled(enable)
        self.hue_slider.setEnabled(enable)
        self.saturation_slider.setEnabled(enable)
        self.contrast_slider.setEnabled(enable)
        self.sharpness_slider.setEnabled(enable)
        self.gamma_slider.setEnabled(enable)
        self.power_line_frequency_cbox.setEnabled(enable)
        self.flip_cbox.setEnabled(enable)
        self.mirror_cbox.setEnabled(enable)

    def init_controls(self):
        self.set_controls_enable(True)
        with QSignalBlocker(self.exp_bright_slider), \
                QSignalBlocker(self.exp_time_slider), \
                QSignalBlocker(self.exp_gain_slider), \
                QSignalBlocker(self.red_slider), \
                QSignalBlocker(self.green_slider), \
                QSignalBlocker(self.blue_slider), \
                QSignalBlocker(self.hue_slider), \
                QSignalBlocker(self.saturation_slider), \
                QSignalBlocker(self.contrast_slider), \
                QSignalBlocker(self.sharpness_slider), \
                QSignalBlocker(self.gamma_slider):
            self.exp_bright_slider.setRange(self.camera_info.exposure.bright.min, self.camera_info.exposure.bright.max)
            self.exp_time_slider.setRange(self.camera_info.exposure.time.min, self.camera_info.exposure.time.max)
            self.exp_gain_slider.setRange(self.camera_info.exposure.gain.min, self.camera_info.exposure.gain.max)
            self.red_slider.setRange(self.camera_info.white_balance.red.min, self.camera_info.white_balance.red.max)
            self.green_slider.setRange(self.camera_info.white_balance.green.min, self.camera_info.white_balance.green.max)
            self.blue_slider.setRange(self.camera_info.white_balance.blue.min, self.camera_info.white_balance.blue.max)
            self.hue_slider.setRange(self.camera_info.hue.min, self.camera_info.hue.max)
            self.saturation_slider.setRange(self.camera_info.saturation.min, self.camera_info.saturation.max)
            self.contrast_slider.setRange(self.camera_info.contrast.min, self.camera_info.contrast.max)
            self.sharpness_slider.setRange(self.camera_info.sharpness.min, self.camera_info.sharpness.max)
            self.gamma_slider.setRange(self.camera_info.gamma.min, self.camera_info.gamma.max)

        self.exp_bright_slider.setValue(self.camera_info.exposure.bright.cur)
        self.exp_time_slider.setValue(self.camera_info.exposure.time.cur)
        self.exp_gain_slider.setValue(self.camera_info.exposure.gain.cur)
        self.red_slider.setValue(self.camera_info.white_balance.red.cur)
        self.green_slider.setValue(self.camera_info.white_balance.green.cur)
        self.blue_slider.setValue(self.camera_info.white_balance.blue.cur)
        self.hue_slider.setValue(self.camera_info.hue.cur)
        self.saturation_slider.setValue(self.camera_info.saturation.cur)
        self.contrast_slider.setValue(self.camera_info.contrast.cur)
        self.sharpness_slider.setValue(self.camera_info.sharpness.cur)
        self.gamma_slider.setValue(self.camera_info.gamma.cur)

        self.auto_exp_cbox.setChecked(self.camera_info.exposure.mode == slncam.SLNCAM_EXPOSURE_MODE_AUTO)
        # self.on_auto_exp_cbox_clicked()
        self.auto_wb_cbox.setChecked(self.camera_info.white_balance.mode == slncam.SLNCAM_WB_MODE_AUTO)
        # self.on_auto_wb_cbox_clicked()
        self.power_line_frequency_cbox.setChecked(self.camera_info.power_line_frequency)
        self.flip_cbox.setChecked(self.camera_info.flip)
        self.mirror_cbox.setChecked(self.camera_info.mirror)

        self.exp_bright_slider.setEnabled(self.auto_exp_cbox.isChecked())
        self.exp_time_slider.setEnabled(not self.auto_exp_cbox.isChecked())
        self.exp_gain_slider.setEnabled(not self.auto_exp_cbox.isChecked())

        self.red_slider.setEnabled(not self.auto_wb_cbox.isChecked())
        self.green_slider.setEnabled(not self.auto_wb_cbox.isChecked())
        self.blue_slider.setEnabled(not self.auto_wb_cbox.isChecked())

    def on_auto_exp_cbox_toggled(self, checked):
        mode = slncam.SLNCAM_EXPOSURE_MODE_AUTO if checked else slncam.SLNCAM_EXPOSURE_MODE_MANUAL
        if self.camera_info.exposure.mode == mode:
            return

        # 此处获取当前状态下的曝光相关值以更新UI
        self.camera_info.exposure = self.camera.get_exposure_info()
        self.camera_info.exposure.mode = mode
        self.camera.set_exposure(self.camera_info.exposure.mode, self.camera_info.exposure.bright.cur,
                                 self.camera_info.exposure.gain.cur, self.camera_info.exposure.time.cur)

        self.exp_bright_slider.setValue(self.camera_info.exposure.bright.cur)
        self.exp_time_slider.setValue(self.camera_info.exposure.time.cur)
        self.exp_gain_slider.setValue(self.camera_info.exposure.gain.cur)

        self.exp_bright_slider.setEnabled(checked)
        self.exp_time_slider.setEnabled(not checked)
        self.exp_gain_slider.setEnabled(not checked)

    def on_exp_bright_slider_changed(self, value):
        if self.camera_info.exposure.bright.cur == value:
            return
        
        self.camera_info.exposure.bright.cur = value
        self.camera.set_exposure(self.camera_info.exposure.mode, self.camera_info.exposure.bright.cur,
                                 self.camera_info.exposure.gain.cur, self.camera_info.exposure.time.cur)

    def on_exp_time_slider_changed(self, value):
        if self.camera_info.exposure.time.cur == value:
            return
        
        self.camera_info.exposure.time.cur = value
        self.camera.set_exposure(self.camera_info.exposure.mode, self.camera_info.exposure.bright.cur,
                                 self.camera_info.exposure.gain.cur, self.camera_info.exposure.time.cur)

    def on_exp_gain_slider_changed(self, value):
        if self.camera_info.exposure.gain.cur == value:
            return
        
        self.camera_info.exposure.gain.cur = value
        self.camera.set_exposure(self.camera_info.exposure.mode, self.camera_info.exposure.bright.cur,
                                 self.camera_info.exposure.gain.cur, self.camera_info.exposure.time.cur)

    def on_auto_wb_cbox_toggled(self, checked):
        mode = slncam.SLNCAM_WB_MODE_AUTO if checked else slncam.SLNCAM_WB_MODE_MANUAL
        if mode == self.camera_info.white_balance.mode:
            return

        # 此处获取当前状态下的白平衡相关值以更新UI
        self.camera_info.white_balance = self.camera.get_wb_info()
        self.camera_info.white_balance.mode = mode

        self.red_slider.setEnabled(not checked)
        self.green_slider.setEnabled(not checked)
        self.blue_slider.setEnabled(not checked)
        self.red_slider.setValue(self.camera_info.white_balance.red.cur)
        self.green_slider.setValue(self.camera_info.white_balance.green.cur)
        self.blue_slider.setValue(self.camera_info.white_balance.blue.cur)
        self.camera.set_wb(self.camera_info.white_balance.mode, self.camera_info.white_balance.red.cur,
                           self.camera_info.white_balance.green.cur, self.camera_info.white_balance.blue.cur)

    def on_red_slider_changed(self, value):
        if self.camera_info.white_balance.red.cur == value:
            return
        
        self.camera_info.white_balance.red.cur = value
        self.camera.set_wb(self.camera_info.white_balance.mode, self.camera_info.white_balance.red.cur,
                           self.camera_info.white_balance.green.cur, self.camera_info.white_balance.blue.cur)

    def on_green_slider_changed(self, value):
        if self.camera_info.white_balance.green.cur == value:
            return

        self.camera_info.white_balance.green.cur = value
        self.camera.set_wb(self.camera_info.white_balance.mode, self.camera_info.white_balance.red.cur,
                           self.camera_info.white_balance.green.cur, self.camera_info.white_balance.blue.cur)

    def on_blue_slider_changed(self, value):
        if self.camera_info.white_balance.blue.cur == value:
            return

        self.camera_info.white_balance.blue.cur = value
        self.camera.set_wb(self.camera_info.white_balance.mode, self.camera_info.white_balance.red.cur,
                           self.camera_info.white_balance.green.cur, self.camera_info.white_balance.blue.cur)

    def on_hue_slider_changed(self, value):
        if self.camera_info.hue.cur == value:
            return

        self.camera_info.hue.cur = value
        self.camera.set_hue(value)

    def on_saturation_slider_changed(self, value):
        if self.camera_info.saturation.cur == value:
            return

        self.camera_info.saturation.cur = value
        self.camera.set_saturation(value)

    def on_contrast_slider_changed(self, value):
        if self.camera_info.contrast.cur == value:
            return

        self.camera_info.contrast.cur = value
        self.camera.set_contrast(value)

    def on_sharpness_slider_changed(self, value):
        if self.camera_info.sharpness.cur == value:
            return

        self.camera_info.sharpness.cur = value
        self.camera.set_sharpness(value)

    def on_gamma_slider_changed(self, value):
        if self.camera_info.gamma.cur == value:
            return

        self.camera_info.gamma.cur = value
        self.camera.set_gamma(value)

    def on_power_line_frequency_cbox_toggled(self, checked):
        if self.camera_info.power_line_frequency == checked:
            return

        self.camera_info.power_line_frequency = checked
        self.camera.set_power_line_frequency(checked)

    def on_flip_cbox_toggled(self, checked):
        if self.camera_info.flip == checked:
            return

        self.camera_info.flip = checked
        self.camera.set_flip(checked)

    def on_mirror_cbox_toggled(self, checked):
        if self.camera_info.mirror == checked:
            return

        self.camera_info.mirror = checked
        self.camera.set_mirror(checked)

    def on_camera_cmb_index_changed(self, index):
        pass

    def on_res_cmb_index_changed(self, index):
        if self.camera is None or index == -1:
            return

        capability = self.camera_info.video_stream_capabilities[index]
        net_info = self.camera.get_stream_settings().net_info
        settings = slncam.CameraStreamSettings(net_info, capability.resolution, capability.format, 
                                         slncam.SLNCAM_PIX_FORMAT_YUV420P_SEPARATE,
                                         'cbr', 'high', capability.fps, 86000, 120)
        if self.camera.is_streaming():
            self.camera.change_stream_settings(settings)
        else:
            self.camera.set_stream_settings(settings)
            self.camera.open_stream()

    def save_captured_image(self, frame):
        options = QFileDialog.Option.DontUseNativeDialog
        file_name, selected_filter = QFileDialog.getSaveFileName(
            self,
            'Save Image',
            os.path.expanduser('~/Pictures/capture%d.jpg' % (self.capture_file_index)),
            'JPEG Files (*.jpg);;PNG Files (*.png)',
            options = options
        )

        if not file_name:
            return

        if selected_filter.startswith('JPEG'):
            if not file_name.lower().endswith('.jpg'):
                file_name += '.jpg'
        else:
            if not file_name.lower().endswith('.png'):
                file_name += '.png'

        try:
            rgb = i420_to_rgb(frame)
            img = QImage(rgb.data, rgb.shape[1], rgb.shape[0], rgb.shape[1] * 3, QImage.Format.Format_RGB888)
            if img.save(file_name):
                self.capture_file_index = self.capture_file_index + 1
                QMessageBox.information(self, 'Success', f'Image saved to:\n{file_name}')
            else:
                QMessageBox.critical(self, 'Error', f'Save operation failed')
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Save operation failed, {e}')

if __name__ == '__main__':
    app = QApplication(sys.argv)
    window = MainWidget()
    window.show()
    sys.exit(app.exec())
