import json
from typing import List, Optional, Tuple
import logging
import ctypes
import os.path
import sys
import re

# SDK错误码
# 成功
SLNCAM_SDK_SUCCESS = 0
# SDK未初始化, 或初始化错误
SLNCAM_SDK_UNINITIALIZED = 1
# 等待超时
SLNCAM_SDK_TIMEOUT = 2
# 参数错误
SLNCAM_SDK_INVALIDPARAMETER = 3
# 打开失败或没有打开对应的对象
SLNCAM_SDK_OPEN_FAILED = 4
# 无数据,或没有新数据刷新
SLNCAM_SDK_NO_DATA = 5

# 网络协议错误码
# 不支持的协议命令
SLNCAM_PROTOCOL_UNSUPPORT = 4097
# 未找到; 没找到数据,配置项,网页等
SLNCAM_PROTOCOL_NOT_FOUND = 4098
# 协议参数错误
SLNCAM_PROTOCOL_INVALID_PARAMETER = 4099

# 视频流格式
# MJPEG所有相机均支持(除F801W), H264目前仅A3xx/F801W支持
SLNCAM_VIDEO_FORMAT_MJPEG = 0
SLNCAM_VIDEO_FORMAT_H264 = 1

# 解码后数据格式
# MJPEG, H264解码后原生数据格式为YUV420P_SEPARATE
SLNCAM_PIX_FORMAT_NULL                 = 0
SLNCAM_PIX_FORMAT_ARGB8888             = 1    # [ARGB][ARGB][ARGB]
SLNCAM_PIX_FORMAT_ARGB8888_SEPARATE    = 2    # [AAAA][RRRR][GGGG][BBBB]
SLNCAM_PIX_FORMAT_RGB888               = 3    # [RGB][RGB][RGB]
SLNCAM_PIX_FORMAT_RGB888_SEPARATE      = 4    # [RRR][GGG][BBB]
SLNCAM_PIX_FORMAT_YUV420P              = 5    # [YUV][YUV][YUV]...[Y00][Y00][Y00]
SLNCAM_PIX_FORMAT_YUV420P_SEPARATE     = 6    # [YYY][UUU][VVV]
SLNCAM_PIX_FORMAT_ARGB1555             = 7    # [ARGB][ARGB][ARGB]
SLNCAM_PIX_FORMAT_ABGR8888             = 41   # [ABGR][ABGR][ABGR]
SLNCAM_PIX_FORMAT_BGR888               = 43   # [BGR][BGR[BGR]
SLNCAM_PIX_FORMAT_RGBA8888             = 45   # [RGBA][RGBA][RGBA]
SLNCAM_PIX_FORMAT_BGRA8888             = 47   # [BGRA][BGRA][BGRA]

# 图片流格式
SLNCAM_PIC_STREAM_FORMAT_JPEG = 0

# 变倍模式, 绝对/相对
SLNCAM_ZOOM_MODE_ABSOLUTE = 1
SLNCAM_ZOOM_MODE_RELATIVE = 2

# 聚焦模式, 自动/手动/一键
SLNCAM_FOCUS_MODE_AUTO = 0
SLNCAM_FOCUS_MODE_MANUAL = 1
# 一键聚焦, 聚焦一次, 清晰后停止聚焦
SLNCAM_FOCUS_MODE_ONCE = 2

# 白平衡模式
SLNCAM_WB_MODE_AUTO = 0
SLNCAM_WB_MODE_MANUAL = 1

# 曝光模式
SLNCAM_EXPOSURE_MODE_AUTO = 0
SLNCAM_EXPOSURE_MODE_MANUAL = 1

class SLNcamMediaFrame(ctypes.Structure):
    '''媒体帧结构, 包含视频音频, 对应l_frame_t'''
    _pack_ = 4
    _fields_ = [
        ('size', ctypes.c_uint8),
        ('ver', ctypes.c_uint8),
        ('fmt', ctypes.c_uint8),
        ('v_type', ctypes.c_uint8),
        ('len', ctypes.c_uint32),
        ('track', ctypes.c_uint16),
        ('bit_wide', ctypes.c_uint16),
        ('sample', ctypes.c_uint32),
        ('time', ctypes.c_int64),
        ('chnn', ctypes.c_uint16),
        ('idx', ctypes.c_uint16),
        ('resv2', ctypes.c_uint32)
    ]

class SLNcamMediaBuffer(ctypes.Structure):
    '''媒体缓存结构, 对应l_md_buf_t'''
    _pack_ = 4
    _fields_ = [
        ('tc', ctypes.c_uint32),
        ('md_id', ctypes.c_uint32),
        ('ver', ctypes.c_uint8),
        ('resv1', ctypes.c_uint8),
        ('resv2', ctypes.c_uint16),
        ('f_v1', SLNcamMediaFrame),
        ('p_buf', ctypes.c_void_p),
        ('buf_len', ctypes.c_int),
        ('start', ctypes.c_int),
        ('end', ctypes.c_int),
        ('p_count', ctypes.c_void_p)
    ]

class SLNcamResolution(ctypes.Structure):
    _pack_ = 4
    _fields_ = [
        ('w', ctypes.c_int),
        ('h', ctypes.c_int)
    ]

    def __init__(self, w=0, h=0):
        super(SLNcamResolution, self).__init__()
        self.w = w
        self.h = h

    def __repr__(self):
        return 'SLNcamResolution(w=%d, h=%d)' % (self.w, self.h)

    def __eq__(self, rt):
        return self.w == rt.w and self.h == rt.h

    def __ne__(self, rt):
        return not (self.w == rt.w and self.h == rt.h)

    @property
    def size(self) -> str:
        return '%d*%d' % (self.w, self.h)

class SLNcamYUVData(ctypes.Structure):
    _pack_ = 4
    _fields_ = [
        ('p_y', ctypes.c_void_p),  # Y分量数据地址
        ('pitch_y', ctypes.c_int), # Y分量步长
        ('p_u', ctypes.c_void_p),  # U分量数据地址
        ('pitch_u', ctypes.c_int), # U分量步长
        ('p_v', ctypes.c_void_p),  # V分量数据地址
        ('pitch_v', ctypes.c_int)  # V分量步长
    ]

class SLNcamRGBData(ctypes.Structure):
    _pack_ = 4
    _fields_ = [
        ('p_rgb', ctypes.c_void_p),
        ('pitch_888', ctypes.c_int) # 步长
    ]

class SLNcamRGBAData(ctypes.Structure):
    _pack_ = 4
    _fields_ = [
        ('p_rgba', ctypes.c_void_p),
        ('pitch_8888', ctypes.c_int)
    ]

class SLNcamData(ctypes.Union):
    _pack_ = 4
    _fields_ = [
        ('yuv_data', SLNcamYUVData),
        ('rgb_data', SLNcamRGBData),
        ('rgba_data', SLNcamRGBAData)
    ]

class SLNcamBuffer(ctypes.Structure):
    _pack_ = 4
    _fields_ = [
        ('p_buf', ctypes.c_void_p),
        ('buf_len', ctypes.c_int),
        ('start', ctypes.c_int),
        ('end', ctypes.c_int)
    ]

class SLNcamVideoFrame(ctypes.Structure):
    '''视频帧数据结构 - 对应l_md_data_t'''
    _pack_ = 4
    _fields_ = [
        ('type', ctypes.c_int), # pix_format
        ('time', ctypes.c_int64),
        ('resolution', SLNcamResolution),
        ('data', SLNcamData),
        ('buffer', SLNcamBuffer)
    ]

class NetInfo:
    def __init__(self,
                 ip: str = '',
                 model: str = '',
                 hardware_version: str = '',
                 software_version: str = '',
                 username: str = '',
                 password:str = '',
                 port: int = 0,
                 version: int = 0):
        self.ip = ip
        self.model = model
        self.hardware_version = hardware_version
        self.software_version = software_version
        self.username = username
        self.password = password
        self.port = port
        self.version = version

    def __repr__(self):
        return 'NetInfo(ip=%s, model=%s, hardware_version=%s, ' \
                'software_version=%s, username=%s, password=%s, port=%d, version=%d)' \
                % (self.ip, self.model, self.hardware_version,
                   self.software_version, self.username,
                   self.password, self.port, self.version)

class CommonAttribute:
    def __init__(self,
                 cur: int = -1, 
                 min: int = 0,
                 max: int = 0,
                 default: int = 0,
                 res: int = 1):
        self.cur = cur
        self.min = min
        self.max = max
        self.default = default
        self.res = res

class CameraPicStreamSettings:
    def __init__(self,
                 resolution: SLNcamResolution = SLNcamResolution(),
                 pic_stream_format : int = SLNCAM_PIC_STREAM_FORMAT_JPEG,
                 quality: str = '',
                 interval: int = 0):
        self.resolution = resolution
        self.pic_stream_format = pic_stream_format
        self.quality = quality
        self.interval = interval

class CameraStreamSettings:
    def __init__(self,
                 net_info: NetInfo = NetInfo(),
                 resolution: SLNcamResolution = SLNcamResolution(),
                 video_format : int = SLNCAM_VIDEO_FORMAT_MJPEG,
                 pix_format: int = SLNCAM_PIX_FORMAT_YUV420P_SEPARATE,
                 rc_mode: str = '',
                 quality: str = '',
                 frame_rate: int = 0,
                 bitrate: int = 0,
                 key_frame_interval: int = 0):
        self.net_info = net_info
        self.resolution = resolution
        # 视频流格式
        self.video_format = video_format
        # 解码后视频帧格式
        self.pix_format = pix_format
        # cbr(定码率) / vbr(可变码率)
        self.rc_mode = rc_mode
        # highest/higher/high/middle/low/lower/lowest
        self.quality = quality
        self.frame_rate = frame_rate
        self.bitrate = bitrate
        self.key_frame_interval = key_frame_interval

class FocusRegion:
    def __init__(self,
                 enable: bool = False,
                 mode: int = 0,
                 x: CommonAttribute = CommonAttribute(),
                 y: CommonAttribute = CommonAttribute()):
        self.enable = enable
        # 大小框
        # 0小框 (x范围：0-16, y范围：0-14)
        # 1大框 (x范围：0-4, y范围：0-4)
        self.mode = mode
        self.x = x
        self.y = y

class FocusRegionV2:
    def __init__(self,
                 enable: bool = False,
                 mode: int = 0,
                 x: int = 0,
                 y: int = 0,
                 width: int = 0,
                 height: int = 0):
        self.enable = enable
        self.mode = mode
        self.x = x
        self.y = y
        self.width = width
        self.height = height

class FocusInfo:
    def __init__(self,
                 mode: int = SLNCAM_FOCUS_MODE_AUTO,
                 value: CommonAttribute = CommonAttribute(),
                 speed: CommonAttribute = CommonAttribute(),
                 region_version: int = 1,
                 region: FocusRegion = FocusRegion(),
                 region_v2: FocusRegionV2 = FocusRegionV2()):
        self.mode = mode
        self.value = value
        self.speed = speed
        self.region_version = region_version
        self.region = region
        self.region_v2 = region_v2

class ExposureInfo:
    def __init__(self,
                 mode: int = SLNCAM_EXPOSURE_MODE_AUTO,
                 bright: CommonAttribute = CommonAttribute(),
                 gain: CommonAttribute = CommonAttribute(),
                 time: CommonAttribute = CommonAttribute()):
        self.mode = mode
        self.bright = bright
        self.gain = gain
        self.time = time

class WhiteBalanceInfo:
    def __init__(self,
                 mode: int = SLNCAM_WB_MODE_AUTO,
                 temperature: CommonAttribute = CommonAttribute(),
                 red: CommonAttribute = CommonAttribute(),
                 green: CommonAttribute = CommonAttribute(),
                 blue: CommonAttribute = CommonAttribute()):
        self.mode = mode
        self.temperature = temperature
        self.red = red
        self.green = green
        self.blue = blue

class LedPartInfo:
    def __init__(self,
                 index: int = 0,
                 enable: bool = False,
                 level: int = 0):
        self.index = index
        self.enable = enable
        self.level = level

class LedInfo:
    def __init__(self,
                 part_number: int = 0,
                 levelRange: CommonAttribute = CommonAttribute(),
                 led_parts: List[LedPartInfo] = []):
        self.part_number = part_number
        self.levelRange = levelRange
        self.parts = led_parts

class IpInfo:
    def __init__(self,
                 dhcp: bool = False,
                 ip: str = '',
                 netmask: str = '',
                 gateway: str = '',
                 mac: str = '',
                 dns1: str = '',
                 dns2: str = '',
                 enable_wireless: bool = False,
                 wireless_dhcp: bool = False,
                 wireless_ip: str = '',
                 wireless_gateway: str = '',
                 wireless_netmask: str = '',
                 wireless_mac: str = '',
                 wireless_dns1: str = '',
                 wireless_dns2: str = '',
                 wireless_type: str = '',
                 wireless_net: str = '',
                 wireless_ap_ssid: str = '',
                 wireless_ap_password: str = '',
                 wireless_sta_ssid: str = '',
                 wireless_sta_password: str = '',
                 wireless_encryption: str = '',
                 wireless_ssid: str = '',
                 wireless_password: str = ''):
        self.dhcp = dhcp
        self.ip = ip
        self.netmask = netmask
        self.gateway = gateway
        self.mac = mac
        self.dns1 = dns1
        self.dns2 = dns2
        # 是否具有无线功能
        self.enable_wireless = enable_wireless
        self.wireless_dhcp = wireless_dhcp
        self.wireless_ip = wireless_ip
        self.wireless_gateway = wireless_gateway
        self.wireless_netmask = wireless_netmask
        self.wireless_mac = wireless_mac
        self.wireless_dns1 = wireless_dns1
        self.wireless_dns2 = wireless_dns2
        # 无线模式, ap/sta
        self.wireless_type = wireless_type
        # 无线频段, 2.4g/5g
        self.wireless_net = wireless_net
        # AP名称
        self.wireless_ap_ssid = wireless_ap_ssid
        # AP密码
        self.wireless_ap_password = wireless_ap_password
        # STA名称
        self.wireless_sta_ssid = wireless_sta_ssid
        # STA密码
        self.wireless_sta_password = wireless_sta_password
        # 无线加密方式, wpa-wpa2 psk/wpa2 psk/psk/none
        self.wireless_encryption = wireless_encryption
        # 下面两个兼容F801W
        # 当wirelessType为AP时, wirelessSsid为AP名称, wirelessPasswd为AP密码
        # 当wirelessType为STA时, wirelessSsid为路由器WIFI名称, wirelessPasswd为路由器WIFI密码
        self.wireless_ssid = wireless_ssid
        self.wireless_password = wireless_password

class VideoStreamCapability:
    def __init__(self, format: int = SLNCAM_VIDEO_FORMAT_MJPEG, resolution: SLNcamResolution = SLNcamResolution(), fps: int = 0):
        self.format = format
        self.resolution = resolution
        self.fps = fps

class CameraInfo:
    def __init__(self,
                 model: str = '',
                 model_suffix: str = '',
                 video_stream_capabilities: List[VideoStreamCapability] = [VideoStreamCapability(SLNCAM_VIDEO_FORMAT_MJPEG, SLNcamResolution(1920, 1080), 30)],
                 focus: FocusInfo = FocusInfo(),
                 exposure: ExposureInfo = ExposureInfo(),
                 white_balance: WhiteBalanceInfo = WhiteBalanceInfo(),
                 led: LedInfo = LedInfo(),
                 ip_info: IpInfo = IpInfo(),
                 hue: CommonAttribute = CommonAttribute(),
                 saturation: CommonAttribute = CommonAttribute(),
                 contrast: CommonAttribute = CommonAttribute(),
                 sharpness: CommonAttribute = CommonAttribute(),
                 gamma: CommonAttribute = CommonAttribute(),
                 drc: CommonAttribute = CommonAttribute(),
                 zoom_speed: CommonAttribute = CommonAttribute(),
                 zoom: CommonAttribute = CommonAttribute(),
                 dzoom: CommonAttribute = CommonAttribute(),
                 contrast_enhancement: CommonAttribute = CommonAttribute(),
                 defog: bool = False,
                 power_line_frequency: bool = False,
                 monochrome: bool = False,
                 flip: bool = False,
                 mirror: bool = False,
                 cac: bool = False,
                 sharpness_enhancement: bool = False,
                 scene_mode: int = 0):
        self.model = model
        self.model_suffix = model_suffix
        # 相机支持的分辨率/格式/帧率列表
        # 老版协议相机无法获取, 需根据相机实际自行调整此变量
        # 新版协议相机可获取得到, 无需写死
        self.video_stream_capabilities = video_stream_capabilities
        self.focus = focus
        self.exposure = exposure
        self.white_balance = white_balance
        self.led = led
        self.ip_info = ip_info
        self.hue = hue
        self.saturation = saturation
        self.contrast = contrast
        self.sharpness = sharpness
        self.gamma = gamma
        self.drc = drc
        self.zoom_speed = zoom_speed
        self.zoom = zoom
        self.dzoom = dzoom
        self.contrast_enhancement = contrast_enhancement
        self.defog = defog
        self.power_line_frequency = power_line_frequency
        self.monochrome = monochrome
        self.flip = flip
        self.mirror = mirror
        # 去紫边
        self.cac = cac
        self.sharpness_enhancement = sharpness_enhancement
        # 场景 0-生物，1-工业 2-金相 3-单筒
        self.scene_mode = scene_mode

class NetDLL:
    __lib = None
    __load_dll_path = None

    def __init__(self):
        pass

    @classmethod
    def set_load_dll_path(cls, path: str):
        cls.__load_dll_path = path

    @classmethod
    def init_sdk(cls, config: str = '') -> bool:
        '''初始化SDK'''
        cls.__init_lib()
        result = cls.__lib.l_sdk_init(config.encode('utf-8'))
        return result == SLNCAM_SDK_SUCCESS

    @classmethod
    def quit_sdk(cls) -> bool:
        '''销毁SDK'''
        cls.__init_lib()
        result = cls.__lib.l_sdk_quit()
        return result == SLNCAM_SDK_SUCCESS

    @classmethod
    def discover_open(cls, config: str = '') -> bool:
        '''打开设备发现'''
        cls.__init_lib()
        result = cls.__lib.l_sdk_discover_open(config.encode('utf-8'))
        return result == SLNCAM_SDK_SUCCESS

    @classmethod
    def discover_close(cls) -> bool:
        '''关闭设备发现'''
        cls.__init_lib()
        result = cls.__lib.l_sdk_discover_close()
        return result == SLNCAM_SDK_SUCCESS

    @classmethod
    def discover_run(cls, enable: bool) -> bool:
        '''运行/停止设备发现'''
        cls.__init_lib()
        result = cls.__lib.l_sdk_discover_run(enable)
        return result == SLNCAM_SDK_SUCCESS

    @classmethod
    def discover_get_devices(cls) -> Optional[str]:
        '''获取发现的设备JSON字符串'''
        cls.__init_lib()

        result_ptr = ctypes.c_char_p()
        ret = cls.__lib.l_sdk_discover_get_devs(ctypes.byref(result_ptr))
        
        if ret != SLNCAM_SDK_SUCCESS or not result_ptr:
            return None
        
        try:
            result = result_ptr.value.decode('utf-8')
            cls.__lib.l_sdk_free(result_ptr)
            return result
        except:
            return None

    @classmethod
    def login(cls, login_json: str) -> Tuple[bool, int]:
        '''登录设备'''
        cls.__init_lib()

        device_id = ctypes.c_int()
        ret = cls.__lib.l_sdk_login(ctypes.byref(device_id), login_json.encode('utf-8'))

        if ret == SLNCAM_SDK_SUCCESS:
            return True, device_id.value
        return False, -1

    @classmethod
    def logout(cls, device_id: int) -> bool:
        '''登出设备'''
        cls.__init_lib()
        ret = cls.__lib.l_sdk_logout(device_id)
        return ret == SLNCAM_SDK_SUCCESS

    @classmethod
    def request(cls, device_id: int, request: str, encoding: str= 'utf-8') -> Optional[str]:
        '''发送请求'''
        cls.__init_lib()

        result_ptr = ctypes.c_char_p()
        ret = cls.__lib.l_sdk_request(device_id, request.encode(encoding), ctypes.byref(result_ptr))

        if result_ptr and ret == SLNCAM_SDK_SUCCESS:
            try:
                result = result_ptr.value.decode(encoding)
                cls.__lib.l_sdk_free(result_ptr)
                return result
            except Exception as e:
                logging.warning('requset error', e)
        return None

    @classmethod
    def dec_open(cls, dec_id: int, config_json: str) -> bool:
        '''打开解码器'''
        cls.__init_lib()
        ret = cls.__lib.l_sdk_dec_open(dec_id, config_json.encode('utf-8'))
        return ret == SLNCAM_SDK_SUCCESS

    @classmethod
    def dec_close(cls, dec_id: int) -> bool:
        '''关闭解码器'''
        cls.__init_lib()
        ret = cls.__lib.l_sdk_dec_close(dec_id)
        return ret == SLNCAM_SDK_SUCCESS

    @classmethod
    def dec_bind(cls, dec_id: int, device_id: int, chnn: int, idx: int, stream_type: int) -> bool:
        '''绑定解码器'''
        cls.__init_lib()
        ret = cls.__lib.l_sdk_dec_bind(dec_id, device_id, chnn, idx, stream_type)
        return ret == SLNCAM_SDK_SUCCESS

    @classmethod
    def dec_unbind(cls, dec_id: int) -> bool:
        '''解绑解码器'''
        cls.__init_lib()
        ret = cls.__lib.l_sdk_dec_unbind(dec_id)
        return ret == SLNCAM_SDK_SUCCESS

    @classmethod
    def dec_get_frame_data(cls, dec_id: int) -> Optional[SLNcamVideoFrame]:
        '''获取解码后的帧数据'''
        cls.__init_lib()

        frame_ptr = ctypes.POINTER(SLNcamVideoFrame)()
        ret = cls.__lib.l_sdk_dec_get_md_data(dec_id, ctypes.byref(frame_ptr))

        if ret == SLNCAM_SDK_SUCCESS and frame_ptr:
            return frame_ptr.contents
        return None

    @classmethod
    def dec_free_frame_data(cls, frame_data: SLNcamVideoFrame):
        '''释放帧数据'''
        cls.__init_lib()
        cls.__lib.l_sdk_dec_free_md_data(ctypes.byref(frame_data))

    @classmethod
    def get_media_buffer(cls, id: int, chnn: int, idx: int, md_id: int) -> Optional[SLNcamMediaBuffer]:
        '''
        从内存缓存中获取最新的媒体数据
        id        登录id
        chnn      通道
        idx       流序号
        md_id     媒体id
        '''
        cls.__init_lib()
        media_buffer = ctypes.POINTER(SLNcamMediaBuffer)()
        ret = cls.__lib.l_sdk_md_get(id, chnn, idx, md_id, ctypes.byref(media_buffer))

        if ret == SLNCAM_SDK_SUCCESS and media_buffer:
            return media_buffer.contents
        return None

    @classmethod
    def media_buffer_sub(cls, buffer: SLNcamMediaBuffer):
        '''媒体数据块引用计数减1'''
        cls.__init_lib()
        cls.__lib.l_sdk_md_buf_sub(ctypes.byref(buffer))

    @classmethod
    def media_buffer_add(cls, buffer: SLNcamMediaBuffer):
        ''''媒体数据块引用计数加1'''
        cls.__init_lib()
        cls.__lib.l_sdk_md_buf_add(ctypes.byref(buffer))

    @classmethod
    def __init_lib(cls):
        if cls.__lib is None:
            try:  # Firstly try to load the library in the directory where this file is located
                if cls.__load_dll_path is None:
                    dir = os.path.dirname(os.path.realpath(__file__))
                else:
                    dir = cls.__load_dll_path

                if sys.platform == 'win32':
                    cls.__lib = ctypes.cdll.LoadLibrary(os.path.join(dir, 'l_sdk.dll'))
            except Exception:
                pass

            if cls.__lib is None:
                if sys.platform == 'win32':
                    cls.__lib = ctypes.cdll.LoadLibrary('l_sdk.dll')

            # 基础SDK函数
            cls.__lib.l_sdk_init.argtypes = [ctypes.c_char_p]
            cls.__lib.l_sdk_init.restype = ctypes.c_int

            cls.__lib.l_sdk_quit.argtypes = []
            cls.__lib.l_sdk_quit.restype = None

            # 设备发现相关
            cls.__lib.l_sdk_discover_open.argtypes = [ctypes.c_char_p]
            cls.__lib.l_sdk_discover_open.restype = ctypes.c_int

            cls.__lib.l_sdk_discover_close.argtypes = []
            cls.__lib.l_sdk_discover_close.restype = ctypes.c_int

            cls.__lib.l_sdk_discover_run.argtypes = [ctypes.c_bool]
            cls.__lib.l_sdk_discover_run.restype = ctypes.c_int

            cls.__lib.l_sdk_discover_get_devs.argtypes = [ctypes.POINTER(ctypes.c_char_p)]
            cls.__lib.l_sdk_discover_get_devs.restype = ctypes.c_int

            # 内存释放
            cls.__lib.l_sdk_free.argtypes = [ctypes.c_char_p]
            cls.__lib.l_sdk_free.restype = None

            # 登录/登出
            cls.__lib.l_sdk_login.argtypes = [ctypes.POINTER(ctypes.c_int), ctypes.c_char_p]
            cls.__lib.l_sdk_login.restype = ctypes.c_int

            cls.__lib.l_sdk_logout.argtypes = [ctypes.c_int]
            cls.__lib.l_sdk_logout.restype = ctypes.c_int

            # 请求接口
            cls.__lib.l_sdk_request.argtypes = [ctypes.c_int, ctypes.c_char_p, ctypes.POINTER(ctypes.c_char_p)]
            cls.__lib.l_sdk_request.restype = ctypes.c_int

            # 解码器相关
            cls.__lib.l_sdk_dec_open.argtypes = [ctypes.c_int, ctypes.c_char_p]
            cls.__lib.l_sdk_dec_open.restype = ctypes.c_int

            cls.__lib.l_sdk_dec_close.argtypes = [ctypes.c_int]
            cls.__lib.l_sdk_dec_close.restype = ctypes.c_int

            cls.__lib.l_sdk_dec_bind.argtypes = [ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int]
            cls.__lib.l_sdk_dec_bind.restype = ctypes.c_int

            cls.__lib.l_sdk_dec_unbind.argtypes = [ctypes.c_int]
            cls.__lib.l_sdk_dec_unbind.restype = ctypes.c_int

            cls.__lib.l_sdk_dec_get_md_data.argtypes = [ctypes.c_int, ctypes.POINTER(ctypes.POINTER(SLNcamVideoFrame))]
            cls.__lib.l_sdk_dec_get_md_data.restype = ctypes.c_int

            cls.__lib.l_sdk_dec_free_md_data.argtypes = [ctypes.POINTER(SLNcamVideoFrame)]
            cls.__lib.l_sdk_dec_free_md_data.restype = ctypes.c_int

            cls.__lib.l_sdk_md_get.argtypes = [ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.POINTER(ctypes.POINTER(SLNcamMediaBuffer))]
            cls.__lib.l_sdk_md_get.restype = ctypes.c_int

            cls.__lib.l_sdk_md_buf_add.argtypes = [ctypes.POINTER(SLNcamMediaBuffer)]
            cls.__lib.l_sdk_md_buf_add.restype = ctypes.c_int

            cls.__lib.l_sdk_md_buf_sub.argtypes = [ctypes.POINTER(SLNcamMediaBuffer)]
            cls.__lib.l_sdk_md_buf_sub.restype = ctypes.c_int

def _convert_wb_mode2str(mode: int) -> str:
    if mode == SLNCAM_WB_MODE_AUTO:
        mode_str = 'auto'
    else:
        mode_str = 'manual'
    return mode_str

def _convert_pic_strem_format2str(format: int) -> str:
    if format == SLNCAM_PIC_STREAM_FORMAT_JPEG:
        return 'jpeg'
    return None

def _version_from_software_ver(software_ver: str) -> int:
    version = 0
    matches = re.findall(r'(\d+)', software_ver)

    for match in matches:
        version = version * 10 + int(match)

    return version

class NetCamera:
    '''网络相机类'''
    _dec_id = 1
    def __init__(self, camera_model: str = ''):
        self.camera_model = camera_model
        self._is_logged_in = False
        self._is_streaming = False
        self._device_id = 0
        self._stream_dec_id = NetCamera._dec_id
        NetCamera._dec_id == NetCamera._dec_id + 1
        self._idx = 0   # 索引
        self._pic_idx = 64 # 图片流
        self._chnn = 0  # 通道号
        self._camera_info = CameraInfo()
        self._camera_settings = None
        self._camera_pic_settings = None

    @staticmethod
    def create(camera_model: str, version: int) -> 'NetCamera':
        '''根据相机型号创建相机实例'''
        if camera_model == 'F801W_IPC':
            return F801WNetCamera()
        # 老版协议相机型号
        elif camera_model in ['M112_IPC', 'M114_IPC', 'M202_IPC']:
            return NetCamera(camera_model)
        # C312/C313/C314/L313/L314 1.8.0版本，L116 1.7.0版本开始仅支持新版协议
        elif camera_model in ['C312_IPC', 'C313_IPC', 'C314_IPC', 'L313_IPC', 'L314_IPC'] and version < 180:
            return NetCamera(camera_model)
        elif camera_model == 'L116_IPC' and version < 170:
            return NetCamera(camera_model)
        else:
            return NewNetCamera(camera_model)

    def get_stream_settings(self) -> Optional[CameraStreamSettings]:
        '''获取相机取流参数'''
        return self._camera_settings

    def set_stream_settings(self, settings: CameraStreamSettings):
        '''设置相机取流参数'''
        self._camera_settings = settings

    @staticmethod
    def init_net() -> bool:
        '''初始化SDK环境'''
        if not NetDLL.init_sdk(''):
            return False
        if not NetDLL.discover_open(''):
            return False
        return NetDLL.discover_run(True)

    @staticmethod
    def destroy_net() -> bool:
        '''销毁SDK环境'''
        NetDLL.discover_run(False)
        NetDLL.discover_close()
        return NetDLL.quit_sdk()

    @staticmethod
    def search_available_cameras() -> List[NetInfo]:
        '''搜索同一网段下的所有网络相机'''
        devices_json = NetDLL.discover_get_devices()
        if not devices_json:
            return []

        try:
            devices = json.loads(devices_json)
            cameras = []

            for device in devices:
                net_info = NetInfo(
                    ip=device.get('ip', ''),
                    model=device.get('discover', {}).get('model', ''),
                    hardware_version=device.get('discover', {}).get('hw_ver', '0.0.0'),
                    software_version=device.get('discover', {}).get('sw_ver', '0.0.0'),
                    port=device.get('discover', {}).get('port', 80),
                    version=_version_from_software_ver(device.get('discover', {}).get('sw_ver', '0.0.0'))
                )
                cameras.append(net_info)

            return cameras
        except json.JSONDecodeError:
            return []

    def login(self) -> bool:
        '''登录至指定IP相机'''
        if not self._camera_settings:
            return False

        login_request = {
            'ip': self._camera_settings.net_info.ip,
            'port': self._camera_settings.net_info.port,
            'cmd': 'login',
            'login': {
                'username': self._camera_settings.net_info.username,
                'passwd': self._camera_settings.net_info.password
            }
        }

        success, device_id = NetDLL.login(json.dumps(login_request))
        if success:
            self._device_id = device_id
            self._is_logged_in = True
            logging.debug(f'登录成功! 设备ID={self._device_id}')
            return True

        logging.error('登录失败')
        return False

    def logout(self) -> bool:
        '''退出登录'''
        success = NetDLL.logout(self._device_id)
        if success:
            self._is_logged_in = False
            logging.debug('退出登录成功')
            return True
        return False

    def open_stream(self) -> bool:
        '''开始取流'''
        if not self._is_logged_in:
            return False

        # 创建解码器配置
        dec_config = {
            'md_fmt': 'MJPEG' if self._camera_settings.video_format == SLNCAM_VIDEO_FORMAT_MJPEG else 'H264',
            'pix_fmt': self._get_pix_format_string(self._camera_settings.pix_format)
        }

        # 打开解码器
        if not NetDLL.dec_open(self._stream_dec_id, json.dumps(dec_config)):
            logging.error('创建解码器失败')
            return False

        # 绑定解码器
        if not NetDLL.dec_bind(self._stream_dec_id, self._device_id, self._chnn, self._idx, 0):
            NetDLL.dec_close(self._stream_dec_id)
            logging.error('绑定解码器失败')
            return False

        # 设置视频流参数
        video_format = 'mjpeg' if self._camera_settings.video_format == SLNCAM_VIDEO_FORMAT_MJPEG else 'h264'

        # H264码率限制
        if self._camera_settings.video_format != SLNCAM_VIDEO_FORMAT_MJPEG and self._camera_settings.bitrate > 30000:
            self._camera_settings.bitrate = 30000

        self._set_stream(
            video_format,
            self._camera_settings.rc_mode,
            self._camera_settings.resolution.size,
            self._camera_settings.quality,
            self._camera_settings.frame_rate,
            self._camera_settings.bitrate,
            self._camera_settings.key_frame_interval
        )

        # 请求视频流
        if self._request_stream():
            self._is_streaming = True
            logging.debug('视频流开启成功')
            return True

        return False

    def close_stream(self) -> bool:
        '''停止取流'''
        NetDLL.dec_unbind(self._stream_dec_id)
        NetDLL.dec_close(self._stream_dec_id)

        request = {
            'cmd': 'close_stream',
            'close_stream': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }

        success = self._set_request(request)
        if success:
            self._is_streaming = False
            logging.debug('视频流关闭成功')
        return success

    def is_streaming(self) -> bool:
        return self._is_streaming

    def change_stream_settings(self, settings: CameraStreamSettings) -> bool:
        '''变更视频流参数'''
        video_format_changed = (self._camera_settings.video_format != settings.video_format)
        self._camera_settings = settings

        if video_format_changed:
            logging.debug('视频流格式已改变, 重新创建解码器')
            NetDLL.dec_unbind(self._stream_dec_id)
            NetDLL.dec_close(self._stream_dec_id)

        # H264码率限制
        if self._camera_settings.video_format != SLNCAM_VIDEO_FORMAT_MJPEG and self._camera_settings.bitrate > 30000:
            self._camera_settings.bitrate = 30000

        video_format = 'mjpeg' if self._camera_settings.video_format == SLNCAM_VIDEO_FORMAT_MJPEG else 'h264'
        self._set_stream(
            video_format,
            self._camera_settings.rc_mode,
            self._camera_settings.resolution.size,
            self._camera_settings.quality,
            self._camera_settings.frame_rate,
            self._camera_settings.bitrate,
            self._camera_settings.key_frame_interval
        )

        if video_format_changed:
            dec_config = {
                'md_fmt': 'MJPEG' if self._camera_settings.video_format == SLNCAM_VIDEO_FORMAT_MJPEG else 'H264',
                'pix_fmt': self._get_pix_format_string(self._camera_settings.pix_format)
            }
            NetDLL.dec_open(self._stream_dec_id, json.dumps(dec_config))
            NetDLL.dec_bind(self._stream_dec_id, self._device_id, self._chnn, self._idx, 0)

        return True

    def get_video_frame_data(self) -> Optional[SLNcamVideoFrame]:
        '''获取一帧视频帧'''
        if not self._is_streaming:
            return None

        frame = NetDLL.dec_get_frame_data(self._stream_dec_id)
        if not frame:
            return None

        # 检查帧数据有效性
        if (frame.resolution.w != self._camera_settings.resolution.w or 
            frame.resolution.h != self._camera_settings.resolution.h):
            NetDLL.dec_free_frame_data(frame)
            return None

        return frame

    @staticmethod
    def free_video_frame_data(frame: SLNcamVideoFrame):
        '''释放视频帧数据'''
        NetDLL.dec_free_frame_data(frame)

    def get_net_camera_all_info(self) -> CameraInfo:
        '''获取当前已登录相机的各参数'''
        self._get_support_func_info()
        self._get_range_info()
        self._get_stream_info()
        self.get_pic_stream()
        self._get_control_info()
        self._get_focus_region_info()
        self._get_exposure_info()
        self._get_wb_info()
        self._get_flip_mirror_status()
        self._get_gamma_value()
        self._get_defog_status()
        self._get_power_line_frequency_status()
        self._get_system_base_info()
        self._get_ipv4_info()

        # 检查是否有WIFI功能
        if self._camera_info.ip_info.enable_wireless:
            self._get_wireless_ipv4_info()
            self._get_wireless_info()

        self._get_cac_info()
        self._get_drc_info()
        self._get_scene_info()
        self._get_wb_temperature_info()

        return self._camera_info

    def _get_request(self, request_json: dict) -> Optional[dict]:
        '''发送GET请求'''
        request_str = json.dumps(request_json)
        result_str = NetDLL.request(self._device_id, request_str)

        if result_str is None:
            logging.warning('GET请求失败')
            return None

        try:
            result = json.loads(result_str)
            cmd = request_json.get('cmd')
            code = result.get(cmd, {}).get('code', -1)

            if code != 0:
                logging.warning(f'GET请求失败: {request_str}, code={code}')
                return None

            return result
        except json.JSONDecodeError as e:
            logging.warning('解析响应JSON失败', e)
            return None

    def _set_request(self, request_json: dict) -> bool:
        '''发送SET请求'''
        request_str = json.dumps(request_json)
        result_str = NetDLL.request(self._device_id, request_str)

        if not result_str:
            return False

        try:
            result = json.loads(result_str)
            cmd = request_json.get('cmd')
            code = result.get(cmd, {}).get('code', -1)
            
            if code == 0:
                return True
            
            logging.warning(f'SET请求失败: {request_str}, code={code}')
            return False
        except json.JSONDecodeError as e:
            logging.warning('解析响应JSON失败', e)
            return False

    def _get_pix_format_string(self, pix_format: int) -> str:
        '''获取像素格式字符串'''
        format_map = {
            SLNCAM_PIX_FORMAT_YUV420P: 'YUV420P_SEPARATE',
            SLNCAM_PIX_FORMAT_RGB888: 'RGB888',
            SLNCAM_PIX_FORMAT_RGBA8888: 'RGBA8888',
            SLNCAM_PIX_FORMAT_BGR888: 'BGR888',
            SLNCAM_PIX_FORMAT_BGRA8888: 'BGRA8888',
            SLNCAM_PIX_FORMAT_ABGR8888: 'ABGR8888',
            SLNCAM_PIX_FORMAT_ARGB8888: 'ARGB8888'
        }
        return format_map.get(pix_format, 'YUV420P_SEPARATE')

    def _set_stream(self, video_format: str, rc_mode: str, resolution: str, 
                   quality: str, frame_rate: int, bitrate: int, key_frame_interval: int) -> bool:
        '''
        调节视频流接口
        streamFormat     视频流格式
        rcMode           码率控制模式  vbr-可变码率 cbr-恒定码率
        resolution       分辨率
        quality          视频质量  highest/higher/high/meddle/low/lower/lowest 
        对应新版协议能力集中的数字     6      5      4    3     2    1     0
        但仍使用字符串发送, 以便兼容老型号
        frame_rate        帧率
        bitrate          码率
        key_frame_interval 关键帧间隔  对MJPEG格式无效
        '''
        request = {
            'cmd': 'set_stream',
            'set_stream': {
                'chnn': self._chnn,
                'idx': self._idx,
                'fmt': video_format,
                'rc_mode': rc_mode,
                'wh': resolution,
                'quality': quality,
                'frame_rate': frame_rate,
                'bitrate': bitrate,
                'i_interval': key_frame_interval
            }
        }
        return self._set_request(request)

    def _request_stream(self) -> bool:
        '''请求视频流'''
        request = {
            'cmd': 'open_stream',
            'open_stream': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        return self._set_request(request)

    def _get_support_func_info(self):
        request = {
            'cmd': 'support'
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_name_info(self):
        request = {
            'cmd': 'name'
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_ntp_info(self):
        request = {
            'cmd': 'ntp'
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_default_stream_info(self):
        request = {
            'cmd': 'default_stream',
            'default_stream': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_default_ipv4_info(self):
        request = {
            'cmd': 'default_ipv4',
            'default_ipv4': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_default_wireless_info(self):
        request = {
            'cmd': 'default_wireless',
            'default_wireless': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_default_wirelessIpv4_info(self):
        request = {
            'cmd': 'default_wireless_ipv4',
            'default_wireless_ipv4': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_default_name_info(self):
        request = {
            'cmd': 'default_name'
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_default_ntp_info(self):
        request = {
            'cmd': 'default_ntp'
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_control_range_info(self):
        request = {
            'cmd': 'default_image',
            'default_image': {
                'chnn': self._chnn
            }
        }
        result = self._get_request(request)
        if result is None:
            return

        try:
            self._camera_info.exposure.bright.default = result['default_image']['bright']
            self._camera_info.exposure.bright.min = result['default_image']['range']['bright']['min']
            self._camera_info.exposure.bright.max = result['default_image']['range']['bright']['max']

            self._camera_info.hue.default = result['default_image']['hue']
            self._camera_info.hue.min = result['default_image']['range']['hue']['min']
            self._camera_info.hue.max = result['default_image']['range']['hue']['max']

            self._camera_info.contrast.default = result['default_image']['contrast']
            self._camera_info.contrast.min = result['default_image']['range']['contrast']['min']
            self._camera_info.contrast.max = result['default_image']['range']['contrast']['max']

            self._camera_info.saturation.default = result['default_image']['saturation']
            self._camera_info.saturation.min = result['default_image']['range']['saturation']['min']
            self._camera_info.saturation.max = result['default_image']['range']['saturation']['max']

            self._camera_info.sharpness.default = result['default_image']['sharpness']
            self._camera_info.sharpness.min = result['default_image']['range']['sharpness']['min']
            self._camera_info.sharpness.max = result['default_image']['range']['sharpness']['max']

            self._camera_info.focus.value.default = result['default_image']['focus_value']
            self._camera_info.focus.value.min = result['default_image']['range']['focus_value']['min']
            self._camera_info.focus.value.max = result['default_image']['range']['focus_value']['max']
        except Exception as e:
            logging.warning(e)


    def _get_focus_region_range_info(self):
        request = {
            'cmd': 'default_img_roi',
            'default_img_roi': {
                'chnn': self._chnn
            }
        }
        result = self._get_request(request)
        if result is None:
            return

        try:
            self._camera_info.focus.region.x.default = result['default_img_roi']['roi_x']
            self._camera_info.focus.region.x.min = result['default_img_roi']['range']['roi_x']['min']
            self._camera_info.focus.region.x.max = result['default_img_roi']['range']['roi_x']['max']

            self._camera_info.focus.region.y.default = result['default_img_roi']['roi_y']
            self._camera_info.focus.region.y.min = result['default_img_roi']['range']['roi_y']['min']
            self._camera_info.focus.region.y.max = result['default_img_roi']['range']['roi_y']['max']
        except Exception as e:
            logging.warning(e)

    def _get_mexposure_range_info(self):
        request = {
            'cmd': 'default_img_meexposure',
            'default_img_meexposure': {
                'chnn': self._chnn
            }
        }
        result = self._get_request(request)
        if result is None:
            return

        try:
            self._camera_info.exposure.gain.default = result['default_img_meexposure']['a_gain']
            self._camera_info.exposure.gain.min = result['default_img_meexposure']['range']['a_gain']['min']
            self._camera_info.exposure.gain.max = result['default_img_meexposure']['range']['a_gain']['max']

            self._camera_info.exposure.time.default = result['default_img_meexposure']['shuttertime']
            self._camera_info.exposure.time.min = result['default_img_meexposure']['range']['shuttertime']['min']
            self._camera_info.exposure.time.max = result['default_img_meexposure']['range']['shuttertime']['max']
        except Exception as e:
            logging.warning(e)

    def _get_wb_range_info(self):
        request = {
            'cmd': 'default_img_awb',
            'default_img_awb': {
                'chnn': self._chnn
            }
        }
        result = self._get_request(request)
        if result is None:
            return

        try:
            self._camera_info.white_balance.red.default = result['default_img_awb']['r']
            self._camera_info.white_balance.red.min = result['default_img_awb']['range']['r']['min']
            self._camera_info.white_balance.red.max = result['default_img_awb']['range']['r']['max']

            self._camera_info.white_balance.green.default = result['default_img_awb']['gr']
            self._camera_info.white_balance.green.min = result['default_img_awb']['range']['gr']['min']
            self._camera_info.white_balance.green.max = result['default_img_awb']['range']['gr']['max']

            self._camera_info.white_balance.blue.default = result['default_img_awb']['b']
            self._camera_info.white_balance.blue.min = result['default_img_awb']['range']['b']['min']
            self._camera_info.white_balance.blue.max = result['default_img_awb']['range']['b']['max']
        except Exception as e:
            logging.warning(e)

    def _get_gamma_range_info(self):
        request = {
            'cmd': 'default_img_hdr_gamma',
            'default_img_hdr_gamma': {
                'chnn': self._chnn
            }
        }
        result = self._get_request(request)
        if result is None:
            return

        try:
            self._camera_info.gamma.default = result['default_img_hdr_gamma']['level']
            self._camera_info.gamma.min = result['default_img_hdr_gamma']['range']['level']['min']
            self._camera_info.gamma.max = result['default_img_hdr_gamma']['range']['level']['max']

        except Exception as e:
            logging.warning(e)

    def _get_drc_range_info(self):
        request = {
            'cmd': 'default_img_drc',
            'default_img_drc': {
                'chnn': self._chnn
            }
        }
        result = self._get_request(request)
        if result is None:
            return

        try:
            self._camera_info.drc.default = result['default_img_drc']['drc']
            self._camera_info.drc.min = result['default_img_drc']['range']['drc']['min']
            self._camera_info.drc.max = result['default_img_drc']['range']['drc']['max']

        except Exception as e:
            logging.warning(e)

    def _get_wb_temperature_range_info(self):
        request = {
            'cmd': 'default_img_color_temp',
            'default_img_color_temp': {
                'chnn': self._chnn
            }
        }
        result = self._get_request(request)
        if result is None:
            return

        try:
            self._camera_info.white_balance.temperature.default = result['default_img_color_temp']['color_temp']
            self._camera_info.white_balance.temperature.min = result['default_img_color_temp']['range']['color_temp']['min']
            self._camera_info.white_balance.temperature.max = result['default_img_color_temp']['range']['color_temp']['max']

        except Exception as e:
            logging.warning(e)

    def _get_range_info(self):
        self._camera_info.focus.value.default = 175
        self._camera_info.focus.value.max = 375
        self._camera_info.focus.value.min = -25
        self._camera_info.focus.value.res = 1

        self._camera_info.exposure.bright.default = 56
        self._camera_info.exposure.bright.max = 250
        self._camera_info.exposure.bright.min = 0
        self._camera_info.exposure.bright.res = 1

        self._camera_info.exposure.time.default = 150
        self._camera_info.exposure.time.max = 333
        self._camera_info.exposure.time.min = 1
        self._camera_info.exposure.time.res = 1

        self._camera_info.exposure.gain.default = 0
        self._camera_info.exposure.gain.max = 33
        self._camera_info.exposure.gain.min = 0
        self._camera_info.exposure.gain.res = 1

        # RGB默认值
        # C313/C312/L116  r=628, g=256, b=421
        # C314            r=661, g=256, b=372
        # M114            r=512, g=256, b=478
        # M112/M202       r=490, g=256, b=369
        self._camera_info.white_balance.red.default = 628
        self._camera_info.white_balance.red.max = 4095
        self._camera_info.white_balance.red.min = 1
        self._camera_info.white_balance.red.res = 1

        self._camera_info.white_balance.green.default = 256
        self._camera_info.white_balance.green.max = 4095
        self._camera_info.white_balance.green.min = 1
        self._camera_info.white_balance.green.res = 1

        self._camera_info.white_balance.blue.default = 421
        self._camera_info.white_balance.blue.max = 4095
        self._camera_info.white_balance.blue.min = 1
        self._camera_info.white_balance.blue.res = 1

        self._camera_info.hue.default = 50
        self._camera_info.hue.max = 100
        self._camera_info.hue.min = 0
        self._camera_info.hue.res = 1

        self._camera_info.saturation.default = 128
        self._camera_info.saturation.max = 255
        self._camera_info.saturation.min = 0
        self._camera_info.saturation.res = 1

        self._camera_info.contrast.default = 50
        self._camera_info.contrast.max = 100
        self._camera_info.contrast.min = 0
        self._camera_info.contrast.res = 1

        self._camera_info.sharpness.default = 8
        self._camera_info.sharpness.max = 15
        self._camera_info.sharpness.min = 0
        self._camera_info.sharpness.res = 1

        self._camera_info.gamma.default = 1
        self._camera_info.gamma.min = 0
        self._camera_info.gamma.max = 3
        self._camera_info.gamma.res = 1

        self._camera_info.drc.default = 100
        self._camera_info.drc.max = 255
        self._camera_info.drc.min = 0
        self._camera_info.drc.res = 1

        self._camera_info.zoom.default = 0
        self._camera_info.zoom.min = 0
        # M202: zoom(0-144), dzoom(0-30)
        # M112/M114: zoom(0-130), dzoom(0-50)
        if self.camera_model == 'M202_IPC':
            self._camera_info.zoom.max = 144
            self._camera_info.dzoom.max = 30
        else:
            self._camera_info.zoom.max = 130
            self._camera_info.dzoom.max = 50

        self._camera_info.zoom.res = 1

        self._camera_info.dzoom.default = 0
        self._camera_info.dzoom.min = 0
        self._camera_info.dzoom.res = 1

        self._get_name_info()
        self._get_ntp_info()
        self._get_default_stream_info()
        self._get_default_ipv4_info()
        self._get_default_wireless_info()
        self._get_default_wirelessIpv4_info()
        self._get_default_name_info()
        self._get_default_ntp_info()

        # 目前仅C313 1.5.0版本及L116 1.6.0版本后支持获取范围及默认值接口
        self._get_control_range_info()
        self._get_focus_region_range_info()
        self._get_mexposure_range_info()
        self._get_wb_range_info()
        self._get_gamma_range_info()
        self._get_drc_range_info()
        self._get_wb_temperature_range_info()

    def _get_stream_info(self):
        '''获取视频流信息'''
        request = {
            'cmd': 'stream',
            'stream': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_control_info(self):
        '''获取控制信息'''
        request = {
            'cmd': 'get_image',
            'get_image': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.focus.mode = result['get_image']['focus_mode']
            self._camera_info.focus.value.cur = result['get_image']['focus_value']
            self._camera_info.exposure.bright.cur = result['get_image']['bright']
            self._camera_info.contrast.cur = result['get_image']['contrast']
            self._camera_info.saturation.cur = result['get_image']['saturation']
            self._camera_info.monochrome = (self._camera_info.saturation.cur == 0)
            self._camera_info.hue.cur = result['get_image']['hue']
            self._camera_info.sharpness.cur = result['get_image']['sharpness']
            self._camera_info.zoom.cur = result['get_image']['ezoom_value']

            if self._camera_info.zoom.cur > self._camera_info.zoom.max:
                self._camera_info.dzoom.cur = self._camera_info.zoom.cur - self._camera_info.zoom.max
                self._camera_info.zoom.cur = self._camera_info.zoom.max

        except Exception as e:
            logging.warning(e)

    def _get_focus_region_info(self):
        '''获取聚焦区域信息'''
        request = {
            'cmd': 'img_roi',
            'img_roi': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.focus.region.enable = result['img_roi']['roi']
            self._camera_info.focus.region.mode = result['img_roi']['roi_mode']
            self._camera_info.focus.region.x.cur = result['img_roi']['roi_x']
            self._camera_info.focus.region.y.cur = result['img_roi']['roi_y']

        except Exception as e:
            logging.warning('获取聚焦区域信息失败', e)

    def _get_exposure_info(self):
        '''获取曝光信息'''
        request = {
            'cmd': 'img_meexposure',
            'img_meexposure': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.exposure.mode = result['img_meexposure']['expo_type']
            self._camera_info.exposure.time.cur = result['img_meexposure']['shuttertime']
            self._camera_info.exposure.gain.cur = result['img_meexposure']['a_gain']

        except Exception as e:
            logging.warning(e)

    def _get_wb_info(self):
        '''获取白平衡信息'''
        request = {
            'cmd': 'img_awb',
            'img_awb': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        try:
            mode = result['img_awb']['awb']
            if mode == 'auto':
                self._camera_info.white_balance.mode = SLNCAM_WB_MODE_AUTO
            else:
                self._camera_info.white_balance.mode = SLNCAM_WB_MODE_MANUAL

        except Exception as e:
            logging.warning(e)

        request = {
            'cmd': 'info_img_awb',
            'info_img_awb': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.white_balance.red.cur = result['info_img_awb']['r']
            self._camera_info.white_balance.green.cur = result['info_img_awb']['gb']
            self._camera_info.white_balance.blue.cur = result['info_img_awb']['b']

        except Exception as e:
            logging.warning(e)

    def _get_flip_mirror_status(self):
        '''获取翻转镜像状态'''
        request = {
            'cmd': 'img_mirror_flip',
            'img_mirror_flip': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.mirror = result['img_mirror_flip']['mirror']
            self._camera_info.flip = result['img_mirror_flip']['flip']

        except Exception as e:
            logging.warning(e)

    def _get_gamma_value(self):
        '''获取伽马值'''
        request = {
            'cmd': 'img_hdr_gamma',
            'img_hdr_gamma': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.gamma.cur = result['img_hdr_gamma']['level']

        except Exception as e:
            logging.warning(e)

    def _get_defog_status(self):
        '''获取去雾状态'''
        request = {
            'cmd': 'motor_demist',
            'motor_demist': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.power_line_frequency = result['motor_demist']['demist']

        except Exception as e:
            logging.warning(e)

    def _get_power_line_frequency_status(self):
        '''获取抗频闪状态'''
        request = {
            'cmd': 'img_flick_hz',
            'img_flick_hz': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.power_line_frequency = result['img_flick_hz']['flick']

        except Exception as e:
            logging.warning(e)

    def _get_system_base_info(self):
        '''获取系统基础信息'''
        request = {
            'cmd': 'system'
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.model = result['system']['model']
            # 部分相机可能不存在以下两个字段
            self._camera_info.model_suffix = result['system'].get('model_suffix', '')
            self._camera_info.ip_info.enable_wireless = result['system'].get('wifi', False)
            logging.debug(f'model={self._camera_info.model}, model_suffix={self._camera_info.model_suffix}, wireless={self._camera_info.ip_info.enable_wireless}')
        except Exception as e:
            logging.warning(e)

    def _get_ipv4_info(self):
        '''获取IPv4信息'''
        request = {
            'cmd': 'ipv4'
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.ip_info.dhcp = result['ipv4']['dhcp']
            self._camera_info.ip_info.ip = result['ipv4']['ip']
            self._camera_info.ip_info.gateway = result['ipv4']['gateway']
            self._camera_info.ip_info.netmask = result['ipv4']['netmask']
            self._camera_info.ip_info.dns1 = result['ipv4']['dns1']
            self._camera_info.ip_info.dns2 = result['ipv4']['dns2']
            self._camera_info.ip_info.mac = result['ipv4']['mac']

        except Exception as e:
            logging.warning(e)

    def _get_wireless_ipv4_info(self):
        '''获取无线IPv4信息'''
        request = {
            'cmd': 'wireless_ipv4'
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.ip_info.wireless_dhcp = result['wireless_ipv4']['dhcp']
            self._camera_info.ip_info.wireless_ip = result['wireless_ipv4']['ip']
            self._camera_info.ip_info.wireless_gateway = result['wireless_ipv4']['gateway']
            self._camera_info.ip_info.wireless_netmask = result['wireless_ipv4']['netmask']
            self._camera_info.ip_info.wireless_dns1 = result['wireless_ipv4']['dns1']
            self._camera_info.ip_info.wireless_dns2 = result['wireless_ipv4']['dns2']
            self._camera_info.ip_info.wireless_mac = result['wireless_ipv4']['mac']

        except Exception as e:
            logging.warning(e)

    def _get_wireless_info(self):
        '''获取无线信息'''
        request = {
            'cmd': 'wireless'
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.ip_info.wireless_type = result['wireless']['type']
            self._camera_info.ip_info.wireless_net = result['wireless']['net']
            self._camera_info.ip_info.wireless_encryption = result['wireless']['enc']

            ap_mode = (self._camera_info.ip_info.wireless_type == 'ap')
            if 'ap_ssid' in result['wireless']:
                self._camera_info.ip_info.wireless_ap_ssid = result['wireless']['ap_ssid']
                self._camera_info.ip_info.wireless_ap_password = result['wireless']['ap_passwd']
                self._camera_info.ip_info.wireless_sta_ssid = result['wireless']['sta_ssid']
                self._camera_info.ip_info.wireless_sta_password = result['wireless']['sta_passwd']
                if ap_mode:
                    self._camera_info.ip_info.wireless_ssid = self._camera_info.ip_info.wireless_ap_ssid
                    self._camera_info.ip_info.wireless_password = self._camera_info.ip_info.wireless_ap_password
                else:
                    self._camera_info.ip_info.wireless_ssid = self._camera_info.ip_info.wireless_sta_ssid
                    self._camera_info.ip_info.wireless_password = self._camera_info.ip_info.wireless_sta_password
            else:
                self._camera_info.ip_info.wireless_ssid = result['wireless']['ssid']
                self._camera_info.ip_info.wireless_password = result['wireless']['passwd']

                if ap_mode:
                    self._camera_info.ip_info.wireless_ap_ssid = self._camera_info.ip_info.wireless_ssid
                    self._camera_info.ip_info.wireless_ap_password = self._camera_info.ip_info.wireless_password
                else:
                    self._camera_info.ip_info.wireless_sta_ssid = self._camera_info.ip_info.wireless_ssid
                    self._camera_info.ip_info.wireless_sta_password = self._camera_info.ip_info.wireless_password

        except Exception as e:
            logging.warning(e)

    def _get_cac_info(self):
        '''获取去紫边信息'''
        request = {
            'cmd': 'img_cac',
            'img_cac': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.cac = result['img_cac']['cac']

        except Exception as e:
            logging.warning(e)

    def _get_drc_info(self):
        '''获取动态范围压缩信息'''
        request = {
            'cmd': 'img_drc',
            'img_drc': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.scene_mode = result['img_drc']['drc']

        except Exception as e:
            logging.warning(e)

    def _get_scene_info(self):
        '''获取场景信息'''
        request = {
            'cmd': 'img_mode',
            'img_mode': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.scene_mode = result['img_mode']['mode']

        except Exception as e:
            logging.warning(e)

    def _get_wb_temperature_info(self):
        '''获取白平衡色温信息'''
        request = {
            'cmd': 'img_color_temp',
            'img_color_temp': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.white_balance.temperature.cur = result['img_color_temp']['color_temp']

        except Exception as e:
            logging.warning(e)

    # 相机控制功能
    def set_focus(self, mode: int, value: int, speed: int = 3) -> bool:
        '''设置聚焦'''
        self._camera_info.focus.mode = mode

        # M系列相机value值无效, 置0
        if self._camera_settings.net_info.model in ['M202_IPC', 'M114_IPC', 'M112_IPC']:
            self._camera_info.focus.value.cur = 0
        else:
            self._camera_info.focus.value.cur = value

        self._camera_info.focus.speed.cur = speed

        return self._control(
            self._camera_info.exposure.bright.cur,
            self._camera_info.contrast.cur,
            self._camera_info.saturation.cur,
            self._camera_info.hue.cur,
            self._camera_info.focus.mode,
            self._camera_info.focus.value.cur,
            self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
            self._camera_info.sharpness.cur
        )

    def set_focus_region(self, enable: bool, x: int, y: int, mode: int = 0) -> bool:
        '''设置聚焦区域'''
        self._camera_info.focus.region.enable = enable
        self._camera_info.focus.region.x.cur = x
        self._camera_info.focus.region.y.cur = y
        self._camera_info.focus.region.mode = mode

        request = {
            'cmd': 'set_img_roi',
            'set_img_roi': {
                'chnn': self._chnn,
                'roi': enable,
                'roi_x': x,
                'roi_y': y,
                'roi_mode': mode
            }
        }
        return self._set_request(request)

    def set_focus_region_v2(self, enable: bool, x: int, y: int, width: int, height: int) -> bool:
        '''设置聚焦区域V2'''
        logging.warning('当前相机不支持聚焦区域V2设置')
        return False

    def get_focus_region_v2(self) -> Tuple[bool, int, int, int, int]:
        '''获取聚焦区域V2'''
        logging.warning('当前相机不支持获取聚焦区域V2')
        return None

    def get_focus_state(self) -> bool:
        '''获取当前聚焦状态, True代表聚焦中, False代表聚焦完成'''
        logging.warning('当前相机不支持获取聚焦状态')
        return False

    def focus_add(self) -> bool:
        '''M系列相机专用, 增加1点聚焦值'''
        logging.debug('聚焦值+1')
        if self._camera_info.focus.mode != SLNCAM_FOCUS_MODE_MANUAL:
            return False

        return self._control(self._camera_info.exposure.bright.cur, self._camera_info.contrast.cur,
                              self._camera_info.saturation.cur, self._camera_info.hue.cur,
                              self._camera_info.focus.mode, 1, self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
                              self._camera_info.sharpness.cur)

    def focus_sub(self) -> bool:
        '''M系列相机专用, 减少1点聚焦值'''
        logging.debug('聚焦值-1')
        if self._camera_info.focus.mode != SLNCAM_FOCUS_MODE_MANUAL:
            return False

        return self._control(self._camera_info.exposure.bright.cur, self._camera_info.contrast.cur,
                              self._camera_info.saturation.cur, self._camera_info.hue.cur,
                              self._camera_info.focus.mode, -1, self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
                              self._camera_info.sharpness.cur)

    def set_exposure(self, mode: int, bright: int, gain: int, shutter_time: int) -> bool:
        '''设置曝光参数'''
        self._camera_info.exposure.mode = mode
        self._camera_info.exposure.bright.cur = bright
        self._camera_info.exposure.gain.cur = gain
        self._camera_info.exposure.time.cur = shutter_time

        request = {
            'cmd': 'set_img_meexposure',
            'set_img_meexposure': {
                'chnn': self._chnn,
                'expo_type': mode,
                'shuttertime': shutter_time,
                'a_gain': gain
            }
        }
        self._set_request(request)

        return self._control(self._camera_info.exposure.bright.cur, self._camera_info.contrast.cur,
                              self._camera_info.saturation.cur, self._camera_info.hue.cur,
                              self._camera_info.focus.mode, self._camera_info.focus.value.cur,
                              self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
                              self._camera_info.sharpness.cur)

    def set_wb(self, mode: int, red: int, green: int, blue: int) -> bool:
        '''设置白平衡参数'''
        self._camera_info.white_balance.mode = mode
        self._camera_info.white_balance.red.cur = red
        self._camera_info.white_balance.green.cur = green
        self._camera_info.white_balance.blue.cur = blue

        wb_mode = _convert_wb_mode2str(mode)
        request = {
            'cmd': 'set_img_awb',
            'set_img_awb': {
                'chnn': self._chnn,
                'awb': wb_mode,
                'r': red,
                'gb': green,
                'gr': green,
                'b': blue
            }
        }
        return self._set_request(request)

    def set_wb_temperature(self, temperature: int) -> bool:
        '''设置白平衡色温'''
        logging.warning('当前相机不支持设置白平衡色温')
        return False

    def set_zoom(self, ezoom_value: int, dzoom_value: int, speed: int, mode: int = SLNCAM_ZOOM_MODE_ABSOLUTE) -> bool:
        '''设置变倍, 老版协议仅支持绝对值模式, 不支持相对变倍, 不支持变倍速度'''
        if ezoom_value > self._camera_info.zoom.max:
            ezoom_value = self._camera_info.zoom.max

        if dzoom_value > self._camera_info.dzoom.max:
            dzoom_value = self._camera_info.dzoom.max

        if ezoom_value >= self._camera_info.zoom.max:
            self._camera_info.dzoom.cur = dzoom_value
        else:
            self._camera_info.dzoom.cur = 0

        self._camera_info.zoom.cur = ezoom_value
        self._camera_info.dzoom.cur = dzoom_value
        self._camera_info.zoom_speed.cur = speed

        return self._control(self._camera_info.exposure.bright.cur, self._camera_info.contrast.cur,
                              self._camera_info.saturation.cur, self._camera_info.hue.cur,
                              self._camera_info.focus.mode, self._camera_info.focus.value.cur,
                              self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
                              self._camera_info.sharpness.cur)

    def set_flip(self, enable: bool) -> bool:
        '''垂直翻转'''
        logging.debug(f'垂直翻转: {enable}')
        self._camera_info.flip = enable
        request = {
            'cmd': 'set_img_mirror_flip',
            'set_img_mirror_flip': {
                'chnn': self._chnn, 
                'flip': enable, 
                'mirror': self._camera_info.mirror
            }
        }
        return self._set_request(request)

    def set_mirror(self, enable: bool) -> bool:
        '''水平翻转'''
        logging.debug(f'水平翻转: {enable}')
        self._camera_info.mirror = enable
        request = {
            'cmd': 'set_img_mirror_flip',
            'set_img_mirror_flip': {
                'chnn': self._chnn, 
                'flip': self._camera_info.flip, 
                'mirror': enable
            }
        }
        return self._set_request(request)

    def set_cac(self, enable: bool) -> bool:
        ''''去紫边, 色差校正'''
        self._camera_info.cac = enable
        request = {
            'cmd': 'set_img_cac',
            'set_img_cac': {
                'chnn': self._chnn,
                'cac': enable
            }
        }
        return self._set_request(request)

    def set_power_line_frequency(self, enable: bool) -> bool:
        '''抗频闪设置, True 50Hz, False 60Hz'''
        self._camera_info.power_line_frequency = enable
        request = {
            'cmd': 'set_img_flick_hz',
            'set_img_flick_hz': {
                'chnn': self._chnn,
                'flick': enable
            }
        }
        return self._set_request(request)

    def set_monochrome(self, enable: bool) -> bool:
        '''黑白设置, 通过调节饱和度来实现黑白效果'''
        self._camera_info.monochrome = enable
        saturation = self._camera_info.saturation.cur
        if enable:
            saturation = 0

        return self._control(self._camera_info.exposure.bright.cur, self._camera_info.contrast.cur,
                              saturation, self._camera_info.hue.cur,
                              self._camera_info.focus.mode, self._camera_info.focus.value.cur,
                              self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
                              self._camera_info.sharpness.cur)

    def set_defog(self, enable: bool) -> bool:
        '''除雾设置'''
        self._camera_info.defog = enable
        request = {
            'cmd': 'set_motor_demist',
            'set_motor_demist': {
                'chnn': self._chnn,
                'demist': enable
            }
        }
        return self._set_request(request)

    def set_contrast(self, value: int) -> bool:
        '''对比度设置'''
        self._camera_info.contrast.cur = value
        return self._control(self._camera_info.exposure.bright.cur, self._camera_info.contrast.cur,
                              self._camera_info.saturation.cur, self._camera_info.hue.cur,
                              self._camera_info.focus.mode, self._camera_info.focus.value.cur,
                              self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
                              self._camera_info.sharpness.cur)

    def set_saturation(self, value: int) -> bool:
        '''饱和度设置'''
        self._camera_info.saturation.cur = value
        return self._control(self._camera_info.exposure.bright.cur, self._camera_info.contrast.cur,
                              self._camera_info.saturation.cur, self._camera_info.hue.cur,
                              self._camera_info.focus.mode, self._camera_info.focus.value.cur,
                              self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
                              self._camera_info.sharpness.cur)

    def set_hue(self, value: int) -> bool:
        '''色度设置'''
        self._camera_info.hue.cur = value
        return self._control(self._camera_info.exposure.bright.cur, self._camera_info.contrast.cur,
                              self._camera_info.saturation.cur, self._camera_info.hue.cur,
                              self._camera_info.focus.mode, self._camera_info.focus.value.cur,
                              self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
                              self._camera_info.sharpness.cur)

    def set_sharpness(self, value: int) -> bool:
        '''锐度设置'''
        self._camera_info.sharpness.cur = value
        return self._control(self._camera_info.exposure.bright.cur, self._camera_info.contrast.cur,
                              self._camera_info.saturation.cur, self._camera_info.hue.cur,
                              self._camera_info.focus.mode, self._camera_info.focus.value.cur,
                              self._camera_info.zoom.cur + self._camera_info.dzoom.cur,
                              self._camera_info.sharpness.cur)

    def set_gamma(self, value: int) -> bool:
        '''伽马值设置'''
        self._camera_info.gamma.cur = value
        request = {
            'cmd': 'set_img_hdr_gamma',
            'set_img_hdr_gamma': {
                'chnn': self._chnn,
                'level': value
            }
        }
        return self._set_request(request)

    def set_drc(self, value: int) -> bool:
        '''消反光设置'''
        self._camera_info.drc.cur = value
        request = {
            'cmd': 'set_img_drc',
            'set_img_drc': {
                'chnn': self._chnn,
                'drc': value
            }
        }
        return self._set_request(request)

    def set_scene(self, value: int) -> bool:
        '''场景切换, L116专有'''
        self._camera_info.scene_mode = value
        request = {
            'cmd': 'set_img_mode',
            'set_img_mode': {
                'chnn': self._chnn,
                'mode': value
            }
        }
        return self._set_request(request)

    def set_sharpness_enhancement(self, enable: bool) -> bool:
        '''设置锐度增强'''
        logging.warning('当前相机不支持设置锐度增强')
        return False

    def set_contrast_enhancement(self, value: int) -> bool:
        '''设置对比度增强'''
        logging.warning('当前相机不支持设置对比度增强')
        return False

    def set_ipv4(self, dhcp: bool, ip: str, gateway: str, netmask: str,
                 dns1: str, dns2: str, mac: str) -> bool:
        '''    
        设置相机有线IP相关信息
        dhcp     是否启用DHCP, 有线DHCP暂不可用, 默认为false
        ip       相机新IP
        gateway  网关
        netmask  子网掩码
        dns      dns地址
        mac      MAC地址
        如在拉流状态, 修改IP后视频流将会断开
        '''
        self._camera_info.ip_info.dhcp = dhcp
        self._camera_info.ip_info.ip = ip
        self._camera_info.ip_info.gateway = gateway
        self._camera_info.ip_info.netmask = netmask
        self._camera_info.ip_info.dns1 = dns1
        self._camera_info.ip_info.dns2 = dns2
        self._camera_info.ip_info.mac = mac

        request = {
            'cmd': 'set_ipv4',
                'set_ipv4':{
                    'chnn': self._chnn,
                    'dhcp': dhcp,
                    'ip': ip,
                    'gateway': gateway,
                    'netmask': netmask,
                    'dns1': dns1,
                    'dns2': dns2,
                    'mac': mac
                }
            }
        return self._set_request(request)

    def set_wireless_ipv4(self, dhcp: bool, ip: str, gateway: str, netmask: str,
                          dns1: str, dns2: str, mac: str) -> bool:
        '''
        设置相机无线IP相关信息
        dhcp 是否启用DHCP
        ip 相机新IP
        gateway 网关
        netmask 子网掩码
        dns dns地址
        mac MAC地址, 仅支持单播MAC, 即第一个字节最后一位不可为1
        '''
        self._camera_info.ip_info.wireless_dhcp = dhcp
        self._camera_info.ip_info.wireless_ip = ip
        self._camera_info.ip_info.wireless_gateway = gateway
        self._camera_info.ip_info.wireless_netmask = netmask
        self._camera_info.ip_info.wireless_dns1 = dns1
        self._camera_info.ip_info.wireless_dns2 = dns2
        self._camera_info.ip_info.wireless_mac = mac
        request = {
            'cmd': 'set_wireless_ipv4',
                'set_wireless_ipv4':{
                    'chnn': self._chnn,
                    'dhcp': dhcp,
                    'ip': ip,
                    'gateway': gateway,
                    'netmask': netmask,
                    'dns1': dns1,
                    'dns2': dns2,
                    'mac': mac
                }
            }
        return self._set_request(request)

    def set_wireless(self, type: str, net: str, ap_ssid: str, ap_password: str,
                     sta_ssid: str, sta_password: str, encryption: str) -> bool:
        '''
        设置相机无线相关信息
        type           ap/sta 无线模式
        net            2.4g/5g 无线频段, 目前仅支持5g
        ap_ssid        AP模式下, 相机AP名称
        ap_password    相机AP密码
        sta_ssid       STA模式下, 路由器无线名称
        sta_password   STA模式下, 路由器无线密码
        encryption     wpa-wpa2 psk/wpa2 psk/psk/none
                       AP模式下, 为相机AP的加密方式;
                       STA模式下, 为路由器WIFI的加密方式
        '''
        ap_mode = (type == 'ap')
        ssid = ap_ssid if ap_mode else sta_ssid
        password = ap_password if ap_mode else sta_password

        self._camera_info.ip_info.wireless_type = type
        self._camera_info.ip_info.wireless_net = net
        self._camera_info.ip_info.wireless_ap_ssid = ap_ssid
        self._camera_info.ip_info.wireless_ap_password = ap_password
        self._camera_info.ip_info.wireless_sta_ssid = sta_ssid
        self._camera_info.ip_info.wireless_sta_password = sta_password
        self._camera_info.ip_info.wireless_encryption = encryption
        self._camera_info.ip_info.wireless_ssid = ssid
        self._camera_info.ip_info.wireless_password = password

        request = {
            'cmd': 'set_wireless',
                'set_wireless': {
                    'chnn': self._chnn,
                    'type': type,
                    'net': net,
                    'ssid': ssid,
                    'passwd': password,
                    'ap_ssid': ap_ssid,
                    'ap_passwd': ap_password,
                    'sta_ssid': sta_ssid,
                    'sta_passwd': sta_password,
                    'enc': encryption
                }
        }
        return self._set_request(request)

    def upgrade(self, upgrade_file_path: str) -> bool:
        '''
        相机升级功能, 暂不支持中文路径
        upgradeFilePath 相机升级文件绝对路径
        升级文件名称规范 例: C313Net - c313_update.lpk
        '''
        request = {
            'cmd': 'upgrade',
            'ip': self._camera_settings.net_info.ip,
            'port': self._camera_settings.net_info.port,
            'path': upgrade_file_path,
            'upgrade': {
                'username': self._camera_settings.net_info.username,
                'passwd': self._camera_settings.net_info.password
            }
        }
        logging.debug(request)

        request_str = json.dumps(request)
        # 网络升级时, 登录Id必须设置为0
        result_str = NetDLL.request(0, request_str)

        if not result_str:
            return False

        try:
            result = json.loads(result_str)
            cmd = request.get('cmd')
            code = result.get(cmd, {}).get('code', -1)

            if code == 0:
                return True

            logging.warning(f'升级失败: {request_str}, code={code}')
            return False
        except json.JSONDecodeError as e:
            logging.warning('解析响应JSON失败', e)
            return False

    def get_upgrade_status(self) -> Tuple[str, float]:
        '''获取当前升级状态, 返回值分别为 升级状态-str, 升级百分比进度-float'''
        request = {
            'cmd': 'status_upgrade',
            'ip': self._camera_settings.net_info.ip,
            'port': self._camera_settings.net_info.port,
        }

        request_str = json.dumps(request)
        # 网络升级时, 登录Id必须设置为0
        result_str = NetDLL.request(0, request_str)

        if not result_str:
            return None

        try:
            result = json.loads(result_str)
            cmd = request.get('cmd')
            code = result.get(cmd, {}).get('code', -1)

            if code == -1:
                return None

            status = result['status_upgrade']['status']
            percentage = result['status_upgrade']['percentage']
            return status, percentage

        except json.JSONDecodeError as e:
            logging.warning('解析响应JSON失败', e)
            return None

    def set_led(self, index: int, level: int) -> bool:
        '''灯光设置'''
        logging.warning('当前相机不支持灯光设置')
        return False

    def set_pic_stream_settings(self, settings: CameraPicStreamSettings):
        '''设置图片流'''
        self._camera_pic_settings = settings
        request = {
            'cmd': 'set_stream_pic',
            'set_stream_pic': {
                'chnn': self._chnn,
                'idx': self._pic_idx,
                'fmt': _convert_pic_strem_format2str(settings.pic_stream_format),
                'wh': settings.resolution.size,
                'quality': settings.quality,
                'interval': settings.interval
            }
        }
        self._set_request(request)
        logging.debug('设置图片流:', request)

    def request_pic_stream(self) -> bool:
        '''请求图片流'''
        request = {
            'cmd': 'open_stream',
            'open_stream': {
                'chnn': self._chnn,
                'idx': self._pic_idx
            }
        }
        result = self._set_request(request)
        if result:
            logging.debug('图片流请求成功')
        return result

    def get_pic_stream(self):
        '''获取当前图片流参数, 包括范围等信息'''
        request = {
            'cmd': 'stream_pic',
            'stream_pic': {
                'chnn': self._chnn,
                'idx': self._pic_idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

    def close_pic_stream(self) -> bool:
        '''关闭图片流'''
        request = {
            'cmd': 'close_stream',
            'close_stream': {
                'chnn': self._chnn,
                'idx': self._pic_idx
            }
        }
        return self._set_request(request)

    def save_pic(self, file_path: str) -> bool:
        ''''
        图片流开启后才可拍照
        保存图片到指定路径下, 附带文件后缀名, 目前仅支持jpeg
        '''
        buffer = NetDLL.get_media_buffer(self._device_id, self._chnn, self._pic_idx, 0)
        if buffer is None:
            return False

        buffer_len = buffer.end - buffer.start
        buffer_bytes = ctypes.string_at(buffer.p_buf + buffer.start, buffer_len)
        with open(file_path, 'wb') as f:
            f.write(buffer_bytes)

        NetDLL.media_buffer_sub(buffer)
        return True

    def get_wb_info(self) -> Optional[WhiteBalanceInfo]:
        self._get_wb_info()
        return self._camera_info.white_balance

    def get_exposure_info(self) -> Optional[ExposureInfo]:
        self._get_exposure_info()
        return self._camera_info.exposure

    def _control(self, bright: int, contrast: int, saturation: int, hue: int, 
                focus_mode: int, focus_value: int, zoom: int, sharpness: int) -> bool:
        '''
        控制接口
        bright     目标亮度(曝光模式为自动时可调)
        contrast   对比度
        saturation 饱和度
        hue        色度
        gamma      伽马值
        focus_mode  聚焦模式
        focus_value 聚焦值
        zoom_value  变倍值
        sharpness  锐度
        注意点: M系列focus_value无效, 均设置为0即可
        zoom值是光学变倍与数字变倍之和, 光学变倍值为最大后, 才会启用数字变倍
        '''
        request = {
            'cmd': 'set_image',
            'set_image': {
                'chnn': self._chnn,
                'bright': bright,
                'contrast': contrast,
                'saturation': saturation,
                'hue': hue,
                'focus_mode': focus_mode,
                'focus_value': focus_value,
                'ezoom_value': zoom,
                'sharpness': sharpness
            }
        }
        return self._set_request(request)

# F801W相机类
class F801WNetCamera(NetCamera):

    '''F801W相机类'''
    def __init__(self):
        super().__init__('F801W_IPC')

    def get_net_camera_all_info(self) -> CameraInfo:
        '''获取F801W相机信息'''
        self._get_support_func_info()
        self._get_range_info()
        self._get_control_info()
        self._get_exposure_info()
        self._get_wb_info()
        self._get_flip_mirror_status()
        self._get_system_base_info()
        self._get_ipv4_info()
        self._get_wireless_ipv4_info()
        return self._camera_info

    def _get_range_info(self):
        '''获取F801W范围信息'''
        # F801W特定的范围设置
        self._camera_info.exposure.bright.max = 100
        self._camera_info.exposure.bright.min = 0
        self._camera_info.exposure.bright.default = 50

        self._camera_info.exposure.time.default = 0
        self._camera_info.exposure.time.max = 333
        self._camera_info.exposure.time.min = 0

        self._camera_info.exposure.gain.default = 0
        self._camera_info.exposure.gain.max = 33
        self._camera_info.exposure.gain.min = 0

        # 白平衡范围
        self._camera_info.white_balance.red.default = 608
        self._camera_info.white_balance.red.max = 4095
        self._camera_info.white_balance.red.min = 1

        self._camera_info.white_balance.green.default = 269
        self._camera_info.white_balance.green.max = 4095
        self._camera_info.white_balance.green.min = 1

        self._camera_info.white_balance.blue.default = 395
        self._camera_info.white_balance.blue.max = 4095
        self._camera_info.white_balance.blue.min = 1

        # 图像参数范围
        for attr in [self._camera_info.hue, self._camera_info.saturation, self._camera_info.contrast]:
            attr.default = 50
            attr.max = 100
            attr.min = 0
            attr.res = 1

    def _get_control_info(self):
        '''获取F801W控制信息'''
        logging.debug('_get_control_info')
        request = {
            'cmd': 'image',
            'image': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }

        result = self._get_request(request)
        if result is None:
            return

        try:
            self._camera_info.exposure.bright.cur = result['image']['bright']
            self._camera_info.contrast.cur = result['image']['contrast']
            self._camera_info.saturation.cur = result['image']['saturation']
            self._camera_info.monochrome = (self._camera_info.saturation.cur == 0)
            self._camera_info.hue.cur = result['image']['hue']

            if self._camera_info.zoom.cur > self._camera_info.zoom.max:
                self._camera_info.dzoom.cur = self._camera_info.zoom.cur - self._camera_info.zoom.max
                self._camera_info.zoom.cur = self._camera_info.zoom.max

        except Exception as e:
            logging.error(f'获取F801W控制信息失败: {e}')

class ImageBoolSettings:
    def __init__(self):
        self.flip = False
        self.mirror = False
        self.cac = False
        self.power_line_frequency = False
        self.monochrome = False
        self.sharpness_enhancement = False

class ImageValueSettings:
    def __init__(self):
        self.contrast = 0
        self.saturation = 0
        self.hue = 0
        self.sharpness = 0
        self.gamma = 0
        self.drc = 0
        self.contrast_enhancement = 0

# 新版相机类
class NewNetCamera(NetCamera):
    '''新版相机类, 新版网络协议'''
    def __init__(self, camera_model: str = ''):
        super().__init__(camera_model)

    def get_net_camera_all_info(self) -> CameraInfo:
        '''获取新版相机信息'''
        self._get_system_base_info()
        self._get_stream_info()
        self._get_stream_ability()
        self._get_pic_ability()
        self._get_image_ability()
        self._get_motor_ability()
        self._get_led_ability()

        self._get_image_value_info()
        self._get_image_bool_info()
        self._get_focus_info()
        self._get_zoom_info()
        self._get_focus_region_info()
        self._get_focus_region_info_v2()
        self._get_exposure_info()
        self._get_wb_info()
        self._get_led_info()
        self._get_ipv4_info()
        # 仅M系列具有除雾功能
        self._get_defog_status()
        # 仅L116具有场景切换功能
        self._get_scene_info()
        self.get_pic_stream()
        
        # 检查是否有WIFI功能
        if self._camera_info.ip_info.enable_wireless:
            self._get_wireless_ipv4_info()
            self._get_wireless_info()
        
        return self._camera_info

    def set_focus(self, mode: int, value: int, speed: int = 3) -> bool:
        '''设置聚焦 - 新版相机实现'''
        self._camera_info.focus.mode = mode
        self._camera_info.focus.value.cur = value
        self._camera_info.focus.speed.cur = speed

        request = {
            'cmd': 'set_focus',
            'set_focus': {
                'chnn': self._chnn,
                'focus_mode': mode,
                'focus_value': value,
                'focus_speed': speed
            }
        }
        return self._set_request(request)

    def set_focus_region(self, enable: bool, x: int, y: int, mode: int = 0) -> bool:
        '''设置聚焦区域 - 新版相机实现'''
        self._camera_info.focus.region.enable = enable
        self._camera_info.focus.region.x.cur = x
        self._camera_info.focus.region.y.cur = y
        self._camera_info.focus.region.mode = mode

        request = {
            'cmd': 'set_roi',
            'set_roi': {
                'chnn': self._chnn,
                'roi': enable,
                'roi_x': x,
                'roi_y': y,
                'roi_mode': mode
            }
        }
        return self._set_request(request)

    def set_focus_region_v2(self, enable: bool, x: int, y: int, width: int, height: int) -> bool:
        '''设置聚焦区域V2 - 新版相机实现'''
        self._camera_info.focus.region_v2.enable = enable
        self._camera_info.focus.region_v2.x = x
        self._camera_info.focus.region_v2.y = y
        self._camera_info.focus.region_v2.width = width
        self._camera_info.focus.region_v2.height = height

        request = {
            'cmd': 'set_focus_region',
            'set_focus_region': {
                'chnn': self._chnn,
                'roi': enable,
                'roi_x': x,
                'roi_y': y,
                'roi_width': width,
                'roi_height': height
            }
        }
        return self._set_request(request)

    def get_focus_region_v2(self) -> Tuple[bool, int, int, int, int]:
        '''获取聚焦区域V2 - 新版相机实现'''
        self._get_focus_region_info_v2()
        return (
            self._camera_info.focus.region_v2.enable,
            self._camera_info.focus.region_v2.x,
            self._camera_info.focus.region_v2.y,
            self._camera_info.focus.region_v2.width,
            self._camera_info.focus.region_v2.height
        )

    def get_focus_state(self) -> bool:
        '''获取当前聚焦状态 - 新版相机实现'''
        request = {
            'cmd': 'get_focus_state',
            'get_focus_state': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }

        result = self._get_request(request)
        if result is None:
            return False

        try:
            focusing = result.get('get_focus_state', {}).get('focusing', False)
            return focusing
        except Exception as e:
            logging.error(f'获取聚焦状态失败: {e}')
            return False

    def focus_add(self) -> bool:
        '''增加1点聚焦值 - 新版相机实现'''
        return self.__set_focus_relative(1)

    def focus_sub(self) -> bool:
        '''减少1点聚焦值 - 新版相机实现'''
        return self.__set_focus_relative(-1)

    def __set_focus_relative(self, relative: int) -> bool:
        '''设置相对聚焦'''
        request = {
            'cmd': 'set_focus_relative',
            'set_focus_relative': {
                'chnn': self._chnn,
                'relative': relative
            }
        }
        return self._set_request(request)

    def set_exposure(self, mode: int, bright: int, gain: int, shutter_time: int) -> bool:
        '''设置曝光'''
        self._camera_info.exposure.mode = mode
        self._camera_info.exposure.bright.cur = bright
        self._camera_info.exposure.gain.cur = gain
        self._camera_info.exposure.time.cur = shutter_time

        request = {
            'cmd': 'set_exposure',
            'set_exposure': {
                'chnn': self._chnn,
                'expo_type': mode,
                'compensation': bright,
                'shuttertime': shutter_time,
                'a_gain': gain
            }
        }
        return self._set_request(request)

    def set_wb(self, mode: int, red: int, green: int, blue: int) -> bool:
        '''设置白平衡'''
        self._camera_info.white_balance.mode = mode
        self._camera_info.white_balance.red.cur = red
        self._camera_info.white_balance.green.cur = green
        self._camera_info.white_balance.blue.cur = blue

        request = {
            'cmd': 'set_white_balance',
            'set_white_balance': {
                'chnn': self._chnn,
                'awb': mode,
                'b': blue,
                'gb': green,
                'gr': green,
                'r': red
            }
        }
        return self._set_request(request)

    def set_wb_temperature(self, temperature: int) -> bool:
        '''设置白平衡色温'''
        self._camera_info.white_balance.temperature.cur = temperature

        request = {
            'cmd': 'set_white_balance',
            'set_white_balance': {
                'chnn': self._chnn,
                'awb': self._camera_info.white_balance.mode,
                'r': self._camera_info.white_balance.red.cur,
                'gb': self._camera_info.white_balance.green.cur,
                'gr': self._camera_info.white_balance.green.cur,
                'b': self._camera_info.white_balance.blue.cur,
                'use_temperature': True,
                'color_temperature': temperature
            }
        }
        return self._set_request(request)

    def set_zoom(self, ezoom_value: int, dzoom_value: int, speed: int, mode: int = SLNCAM_ZOOM_MODE_ABSOLUTE) -> bool:
        '''
        设置变倍
        ezoom_value 光学变倍值
        dzoom_value 数字变倍值
        speed 变倍速度, 0-低, 1-中, 2-高
        mode 变倍模式, 绝对/相对
             绝对模式下, ezoom_value/dzoom_value均是实际值
             相对模式下, ezoom_value为-1时缩小倍率, 为1时放大倍率, 为0时停止
        '''
        if ezoom_value > self._camera_info.zoom.max:
            ezoom_value = self._camera_info.zoom.max

        if dzoom_value > self._camera_info.dzoom.max:
            dzoom_value = self._camera_info.dzoom.max

        if ezoom_value >= self._camera_info.zoom.max:
            self._camera_info.dzoom.cur = dzoom_value
        else:
            self._camera_info.dzoom.cur = 0

        self._camera_info.zoom.cur = ezoom_value
        self._camera_info.dzoom.cur = dzoom_value
        self._camera_info.zoom_speed.cur = speed

        request = {
            'cmd': 'set_zoom',
            'set_zoom': {
                'chnn': self._chnn,
                'ezoom_value': ezoom_value,
                'dzoom_value': dzoom_value,
                'ezoom_speed': speed,
                'ezoom_mode': mode
            }
        }
        return self._set_request(request)

    def __collect_image_bool_settings(self) -> ImageBoolSettings:
        settings = ImageBoolSettings()
        settings.flip = self._camera_info.flip
        settings.mirror = self._camera_info.mirror
        settings.cac = self._camera_info.cac
        settings.power_line_frequency = self._camera_info.power_line_frequency
        settings.monochrome = self._camera_info.monochrome
        settings.sharpness_enhancement = self._camera_info.sharpness_enhancement
        return settings

    def __collect_image_value_settings(self) -> ImageValueSettings:
        settings = ImageValueSettings()
        settings.contrast = self._camera_info.contrast.cur
        settings.saturation = self._camera_info.saturation.cur
        settings.hue = self._camera_info.hue.cur
        settings.sharpness = self._camera_info.sharpness.cur
        settings.gamma = self._camera_info.gamma.cur
        settings.drc = self._camera_info.drc.cur
        settings.contrast_enhancement = self._camera_info.contrast_enhancement.cur
        return settings

    def __set_image_value(self, settings: ImageValueSettings) -> bool:
        request = {
            'cmd': 'set_image_value',
            'set_image_value': {
                'chnn': self._chnn,
                'contrast': settings.contrast,
                'saturation': settings.saturation,
                'hue': settings.hue,
                'sharpness': settings.sharpness,
                'gamma': settings.gamma,
                'drc': settings.drc,
                # 以下为新增字段
                'contrast_enhance': settings.contrast_enhancement
            }
        }
        return self._set_request(request)

    def __set_image_bool(self, settings: ImageBoolSettings) -> bool:
        request = {
            'cmd': 'set_image_bool',
            'set_image_bool': {
                'chnn': self._chnn,
                'flip': settings.flip,
                'mirror': settings.mirror,
                'cac': settings.cac,
                'flick': settings.power_line_frequency,
                'monochrome': settings.monochrome,
                # 以下为新增字段
                'enhance': settings.sharpness_enhancement
            }
        }
        return self._set_request(request)

    def set_flip(self, enable: bool) -> bool:
        '''设置垂直翻转'''
        self._camera_info.flip = enable
        return self.__set_image_bool(self.__collect_image_bool_settings())

    def set_mirror(self, enable: bool) -> bool:
        '''设置水平翻转'''
        self._camera_info.mirror = enable
        return self.__set_image_bool(self.__collect_image_bool_settings())

    def set_cac(self, enable: bool) -> bool:
        '''去紫边, 色差校正'''
        self._camera_info.cac = enable
        return self.__set_image_bool(self.__collect_image_bool_settings())

    def set_power_line_frequency(self, enable: bool) -> bool:
        '''抗频闪设置, True 50Hz, False 60Hz'''
        self._camera_info.power_line_frequency = enable
        return self.__set_image_bool(self.__collect_image_bool_settings())

    def set_monochrome(self, enable: bool) -> bool:
        '''黑白设置'''
        self._camera_info.monochrome = enable
        return self.__set_image_bool(self.__collect_image_bool_settings())

    def set_sharpness_enhancement(self, enable: bool) -> bool:
        '''锐度增强设置'''
        self._camera_info.sharpness_enhancement = enable
        return self.__set_image_bool(self.__collect_image_bool_settings())

    def set_contrast_enhancement(self, value: int) -> bool:
        '''对比度增强设置'''
        self._camera_info.contrast_enhancement.cur = value
        return self.__set_image_value(self.__collect_image_value_settings())

    def set_scene(self, mode: int) -> bool:
        '''场景切换, L116专有'''
        self._camera_info.scene_mode = mode
        request = {
            'cmd': 'set_scene_mode',
            'set_scene_mode': {
                'chnn': self._chnn,
                'mode': mode
            }
        }
        return self._set_request(request)

    def set_led(self, part_index: int, level: int) -> bool:
        '''灯光设置'''
        for i, led_part in enumerate(self._camera_info.led.parts):
            if led_part.index == part_index:
                self._camera_info.led.parts[i] = level

        # a3xx version < 120
        # request = {
        #     'cmd': 'set_led',
        #     'set_led': {
        #         'chnn': self.__chnn,
        #         'part_index': part_index,
        #         'level': level
        #     }
        # }
        # return self._set_request(request)

        request = {
            'cmd': 'set_leds',
            'set_leds': {
                'chnn': self._chnn,
                'led0': self._camera_info.led.parts[0].level,
                'led1': self._camera_info.led.parts[1].level,
                'led2': self._camera_info.led.parts[2].level,
                'led3': self._camera_info.led.parts[3].level
            }
        }
        return self._set_request(request)

    def set_contrast(self, value: int) -> bool:
        '''设置对比度'''
        self._camera_info.contrast.cur = value
        return self.__set_image_value(self.__collect_image_value_settings())

    def set_saturation(self, value: int) -> bool:
        '''设置饱和度'''
        self._camera_info.saturation.cur = value
        return self.__set_image_value(self.__collect_image_value_settings())

    def set_hue(self, value: int) -> bool:
        '''设置色度'''
        self._camera_info.hue.cur = value
        return self.__set_image_value(self.__collect_image_value_settings())

    def set_sharpness(self, value: int) -> bool:
        '''设置锐度'''
        self._camera_info.sharpness.cur = value
        return self.__set_image_value(self.__collect_image_value_settings())

    def set_gamma(self, value: int) -> bool:
        '''设置伽马值'''
        self._camera_info.gamma.cur = value
        return self.__set_image_value(self.__collect_image_value_settings())

    def set_drc(self, value: int) -> bool:
        '''设置动态范围压缩, 消反光'''
        self._camera_info.drc.cur = value
        return self.__set_image_value(self.__collect_image_value_settings())

    def set_defog(self, enable: bool) -> bool:
        '''设置去雾'''
        self._camera_info.defog = enable
        request = {
            'cmd': 'set_motor_demist',
            'set_motor_demist': {
                'chnn': self._chnn,
                'demist': enable
            }
        }
        return self._set_request(request)

    def _get_defog_status(self):
        request = {
            'cmd': 'get_motor_demist',
            'get_motor_demist': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.defog = result['get_motor_demist']['demist']
        except Exception as e:
            logging.warning(e)

    def _get_scene_info(self):
        '''获取场景信息'''
        request = {
            'cmd': 'get_scene_mode',
            'get_scene_mode': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)

        if result is None:
            return

        try:
            self._camera_info.scene_mode = result['get_scene_mode']['mode']

        except Exception as e:
            logging.warning(e)

    def get_pic_stream(self) -> bool:
        '''获取图片流'''
        request = {
            'cmd': 'get_stream_pic',
            'get_stream_pic': {
                'chnn': self._chnn,
                'idx': self._pic_idx
            }
        }
        result = self._get_request(request)
        if result:
            logging.debug(result)
            return True
        return False

    def _get_stream_info(self):
        '''获取视频流信息'''
        request = {
            'cmd': 'get_stream',
            'get_stream': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_stream_ability(self):
        '''获取流能力信息'''
        request = {
            'cmd': 'get_stream_ability',
            'get_stream_ability': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        if result is None:
            return

        try:
            result = result['get_stream_ability']
            support_formats = list()
            support_resolutions = list()

            for fmt in result['fmt']:
                if fmt == 'h264':
                    support_formats.append(SLNCAM_VIDEO_FORMAT_H264)
                elif fmt == 'mjpeg':
                    support_formats.append(SLNCAM_VIDEO_FORMAT_MJPEG)

            for resolution in result['wh']:
                res_split = resolution.split('*')
                support_resolutions.append(SLNcamResolution(int(res_split[0]), int(res_split[1])))

            max_fps = result['frame_rate']['max']

            abilities = list()
            for format in support_formats:
                for resolution in support_resolutions:
                    # 分辨率大于1920 * 1080的帧率需小于30
                    if resolution.w * resolution.h > 1920 * 1080:
                        fps = 30
                    else:
                        fps = max_fps
                    logging.debug('Stream ability, format=%d, resolution=%s, fps=%d' % (format, resolution.size, fps))
                    abilities.append(VideoStreamCapability(format, resolution, fps))

            if len(abilities) > 0:
                self._camera_info.video_stream_capabilities = abilities

        except Exception as e:
            logging.warning('获取流能力信息', e)

    def _get_pic_ability(self):
        '''获取图片能力信息'''
        request = {
            'cmd': 'get_pic_ability',
            'get_pic_ability': {
                'chnn': self._chnn,
                'idx': self._pic_idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

    def _get_image_ability(self):
        '''获取图像能力信息'''
        request = {
            'cmd': 'get_image_ability',
            'get_image_ability': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.exposure.bright.min = result['get_image_ability']['compensation']['min']
            self._camera_info.exposure.bright.max = result['get_image_ability']['compensation']['max']
            self._camera_info.exposure.bright.default = result['get_image_ability']['compensation']['def']

            self._camera_info.exposure.gain.min = result['get_image_ability']['a_gain']['min']
            self._camera_info.exposure.gain.max = result['get_image_ability']['a_gain']['max']
            self._camera_info.exposure.gain.default = result['get_image_ability']['a_gain']['def']

            self._camera_info.exposure.time.min = result['get_image_ability']['shuttertime']['min']
            self._camera_info.exposure.time.max = result['get_image_ability']['shuttertime']['max']
            self._camera_info.exposure.time.default = result['get_image_ability']['shuttertime']['def']

            self._camera_info.contrast.min = result['get_image_ability']['contrast']['min']
            self._camera_info.contrast.max = result['get_image_ability']['contrast']['max']
            self._camera_info.contrast.default = result['get_image_ability']['contrast']['def']

            self._camera_info.saturation.min = result['get_image_ability']['saturation']['min']
            self._camera_info.saturation.max = result['get_image_ability']['saturation']['max']
            self._camera_info.saturation.default = result['get_image_ability']['saturation']['def']

            self._camera_info.hue.min = result['get_image_ability']['hue']['min']
            self._camera_info.hue.max = result['get_image_ability']['hue']['max']
            self._camera_info.hue.default = result['get_image_ability']['hue']['def']

            self._camera_info.sharpness.min = result['get_image_ability']['sharpness']['min']
            self._camera_info.sharpness.max = result['get_image_ability']['sharpness']['max']
            self._camera_info.sharpness.default = result['get_image_ability']['sharpness']['def']

            self._camera_info.gamma.min = result['get_image_ability']['gamma']['min']
            self._camera_info.gamma.max = result['get_image_ability']['gamma']['max']
            self._camera_info.gamma.default = result['get_image_ability']['gamma']['def']

            self._camera_info.white_balance.red.min = result['get_image_ability']['r']['min']
            self._camera_info.white_balance.red.max = result['get_image_ability']['r']['max']
            self._camera_info.white_balance.red.default = result['get_image_ability']['r']['def']

            self._camera_info.white_balance.green.min = result['get_image_ability']['gr']['min']
            self._camera_info.white_balance.green.max = result['get_image_ability']['gr']['max']
            self._camera_info.white_balance.green.default = result['get_image_ability']['gr']['def']

            self._camera_info.white_balance.blue.min = result['get_image_ability']['b']['min']
            self._camera_info.white_balance.blue.max = result['get_image_ability']['b']['max']
            self._camera_info.white_balance.blue.default = result['get_image_ability']['b']['def']

            self._camera_info.drc.min = result['get_image_ability']['drc']['min']
            self._camera_info.drc.max = result['get_image_ability']['drc']['max']
            self._camera_info.drc.default = result['get_image_ability']['drc']['def']

            # 以下为新增字段
            if (result.get('get_image_ability', {}).get('contrast_enhance') is not None):
                self._camera_info.contrast_enhancement.min = result['get_image_ability']['contrast_enhance']['min']
                self._camera_info.contrast_enhancement.max = result['get_image_ability']['contrast_enhance']['max']
                self._camera_info.contrast_enhancement.default = result['get_image_ability']['contrast_enhance']['def']

            if (result.get('get_image_ability', {}).get('color_temperature') is not None):
                self._camera_info.white_balance.temperature.min = result['get_image_ability']['color_temperature']['min']
                self._camera_info.white_balance.temperature.max = result['get_image_ability']['color_temperature']['max']
                self._camera_info.white_balance.temperature.default = result['get_image_ability']['color_temperature']['def']

        except Exception as e:
            logging.warning('获取图像能力信息失败', e)

    def _get_motor_ability(self):
        '''获取电机能力信息'''
        request = {
            'cmd': 'get_motor_ability',
            'get_motor_ability': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.focus.value.min = result['get_motor_ability']['focus_value']['min']
            self._camera_info.focus.value.max = result['get_motor_ability']['focus_value']['max']
            self._camera_info.focus.value.default = result['get_motor_ability']['focus_value']['def']

            self._camera_info.focus.speed.min = result['get_motor_ability']['focus_speed']['min']
            self._camera_info.focus.speed.max = result['get_motor_ability']['focus_speed']['max']
            self._camera_info.focus.speed.default = result['get_motor_ability']['focus_speed']['def']

            self._camera_info.zoom.min = result['get_motor_ability']['ezoom_value']['min']
            self._camera_info.zoom.max = result['get_motor_ability']['ezoom_value']['max']
            self._camera_info.zoom.default = result['get_motor_ability']['ezoom_value']['def']

            self._camera_info.dzoom.min = result['get_motor_ability']['dzoom_value']['min']
            self._camera_info.dzoom.max = result['get_motor_ability']['dzoom_value']['max']
            self._camera_info.dzoom.default = result['get_motor_ability']['dzoom_value']['def']

            self._camera_info.zoom_speed.min = result['get_motor_ability']['ezoom_speed']['min']
            self._camera_info.zoom_speed.max = result['get_motor_ability']['ezoom_speed']['max']
            self._camera_info.zoom_speed.default = result['get_motor_ability']['ezoom_speed']['def']

        except Exception as e:
            logging.warning('获取电机能力信息失败', e)

    def _get_led_ability(self):
        '''获取LED能力信息'''
        request = {
            'cmd': 'get_led_ability',
            'get_led_ability': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.led.part_number = result['get_led_ability']['part_number']
            self._camera_info.led.levelRange.min = result['get_led_ability']['level']['min']
            self._camera_info.led.levelRange.max = result['get_led_ability']['level']['max']
            self._camera_info.led.levelRange.default = result['get_led_ability']['level']['def']

        except Exception as e:
            logging.warning('获取LED能力信息失败', e)

    def _get_image_value_info(self):
        request = {
            'cmd': 'get_image_value',
            'get_image_value': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.contrast.cur = result['get_image_value']['contrast']
            self._camera_info.saturation.cur = result['get_image_value']['saturation']
            self._camera_info.hue.cur = result['get_image_value']['hue']
            self._camera_info.sharpness.cur = result['get_image_value']['sharpness']
            self._camera_info.gamma.cur = result['get_image_value']['gamma']
            self._camera_info.drc.cur = result['get_image_value']['drc']
            # 以下为新增字段
            if (result.get('get_image_value', {}).get('contrast_enhance') is not None):
                self._camera_info.contrast_enhancement.cur = result['get_image_value']['contrast_enhance']

        except Exception as e:
            logging.warning('获取图像值信息失败', e)

    def _get_image_bool_info(self):
        request = {
            'cmd': 'get_image_bool',
            'get_image_bool': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.mirror = result['get_image_bool']['mirror']
            self._camera_info.flip = result['get_image_bool']['flip']
            self._camera_info.power_line_frequency = result['get_image_bool']['flick']
            self._camera_info.cac = result['get_image_bool']['cac']
            self._camera_info.monochrome = result['get_image_bool']['monochrome']
            # 以下为新增字段
            if (result.get('get_image_bool', {}).get('enhance') is not None):
                self._camera_info.sharpness_enhancement = result['get_image_bool']['enhance']

        except Exception as e:
            logging.warning('获取图像布尔信息失败', e)

    def _get_focus_info(self):
        '''获取聚焦信息'''
        request = {
            'cmd': 'get_focus',
            'get_focus': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.focus.mode = result['get_focus']['focus_mode']
            self._camera_info.focus.value.cur = result['get_focus']['focus_value']
            self._camera_info.focus.speed.cur = result['get_focus']['focus_speed']

        except Exception as e:
            logging.warning('获取聚焦信息失败', e)

    def _get_zoom_info(self):
        '''获取变倍信息'''
        request = {
            'cmd': 'get_zoom',
            'get_zoom': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.zoom.cur = result['get_zoom']['ezoom_value']
            self._camera_info.dzoom.cur = result['get_zoom']['dzoom_value']
            self._camera_info.zoom_speed.cur = result['get_zoom']['ezoom_speed']

        except Exception as e:
            logging.warning('获取变倍信息失败', e)

    def _get_focus_region_info(self):
        request = {
            'cmd': 'get_roi',
            'get_roi': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.focus.region.enable = result['get_roi']['roi']
            self._camera_info.focus.region.mode = result['get_roi']['roi_mode']
            self._camera_info.focus.region.x.cur = result['get_roi']['roi_x']
            self._camera_info.focus.region.y.cur = result['get_roi']['roi_y']
        except Exception as e:
            logging.warning('获取聚焦区域信息失败', e)

    def _get_focus_region_info_v2(self):
        '''获取聚焦区域V2信息'''
        request = {
            'cmd': 'get_focus_region',
            'get_focus_region': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.focus.region_v2.enable = result['get_focus_region']['roi']
            self._camera_info.focus.region_v2.x = result['get_focus_region']['roi_x']
            self._camera_info.focus.region_v2.y = result['get_focus_region']['roi_y']
            self._camera_info.focus.region_v2.width = result['get_focus_region']['roi_width']
            self._camera_info.focus.region_v2.height = result['get_focus_region']['roi_height']
        except Exception as e:
            logging.warning('获取聚焦区域V2信息失败', e)

    def _get_exposure_info(self):
        request = {
            'cmd': 'get_exposure',
            'get_exposure': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.exposure.mode = result['get_exposure']['expo_type']
            self._camera_info.exposure.bright.cur = result['get_exposure']['compensation']
            self._camera_info.exposure.time.cur = result['get_exposure']['shuttertime']
            self._camera_info.exposure.gain.cur = result['get_exposure']['a_gain']
        except Exception as e:
            logging.warning('获取曝光信息失败', e)

    def _get_wb_info(self):
        request = {
            'cmd': 'get_white_balance',
            'get_white_balance': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.white_balance.mode = result['get_white_balance']['awb']
            self._camera_info.white_balance.red.cur = result['get_white_balance']['r']
            self._camera_info.white_balance.green.cur = result['get_white_balance']['gb']
            self._camera_info.white_balance.blue.cur = result['get_white_balance']['b']
            # 新增字段
            if (result.get('get_white_balance', {}).get('color_temperature') is not None):
                self._camera_info.white_balance.temperature.cur = result['get_white_balance']['color_temperature']
        except Exception as e:
            logging.warning('获取白平衡信息失败', e)

    def _get_led_info(self):
        request = {
            'cmd': 'get_led',
            'get_led': {
                'chnn': self._chnn,
                'idx': self._idx
            }
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            infos = result['get_led']['led_info']
            for i in range(len(infos)):
                index=infos[i].get('index', i)
                level=infos[i].get('level', 0)
                enable=infos[i].get('enable', False)
                self._camera_info.led.parts.append(LedPartInfo(index, enable, level))
        except Exception as e:
            logging.warning('获取LED信息失败', e)

    def _get_ipv4_info(self):
        request = {
            'cmd': 'get_ipv4'
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.ip_info.dhcp = result['get_ipv4']['dhcp']
            self._camera_info.ip_info.ip = result['get_ipv4']['ip']
            self._camera_info.ip_info.gateway = result['get_ipv4']['gateway']
            self._camera_info.ip_info.netmask = result['get_ipv4']['netmask']
            self._camera_info.ip_info.dns1 = result['get_ipv4']['dns1']
            self._camera_info.ip_info.dns2 = result['get_ipv4']['dns2']
            # 部分相机无mac信息
            self._camera_info.ip_info.mac = result['get_ipv4'].get('mac', '')
        except Exception as e:
            logging.warning('获取IPv4信息失败', e)

    def _get_wireless_ipv4_info(self):
        request = {
            'cmd': 'get_wireless_ipv4'
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.ip_info.wireless_dhcp = result['get_wireless_ipv4']['dhcp']
            self._camera_info.ip_info.wireless_ip = result['get_wireless_ipv4']['ip']
            self._camera_info.ip_info.wireless_gateway = result['get_wireless_ipv4']['gateway']
            self._camera_info.ip_info.wireless_netmask = result['get_wireless_ipv4']['netmask']
            self._camera_info.ip_info.wireless_dns1 = result['get_wireless_ipv4']['dns1']
            self._camera_info.ip_info.wireless_dns2 = result['get_wireless_ipv4']['dns2']
            self._camera_info.ip_info.wireless_mac = result['get_wireless_ipv4']['mac']
        except Exception as e:
            logging.warning('获取无线IPv4信息失败', e)

    def _get_wireless_info(self):
        request = {
            'cmd': 'get_wireless'
        }
        result = self._get_request(request)
        logging.debug(result)

        try:
            self._camera_info.ip_info.wireless_type = result['get_wireless']['type']
            self._camera_info.ip_info.wireless_net = result['get_wireless']['net']
            ap_mode = self._camera_info.ip_info.wireless_type == 'ap'

            if 'ap_ssid' in result['get_wireless']:
                self._camera_info.ip_info.wireless_ssid = result['get_wireless']['ap_ssid']
                self._camera_info.ip_info.wireless_password = result['get_wireless']['ap_passwd']
                self._camera_info.ip_info.wireless_sta_ssid = result['get_wireless']['sta_ssid']
                self._camera_info.ip_info.wireless_sta_password = result['get_wireless']['sta_passwd']
                
                if ap_mode:
                    self._camera_info.ip_info.wireless_ssid = self._camera_info.ip_info.wireless_ap_ssid
                    self._camera_info.ip_info.wireless_password = self._camera_info.ip_info.wireless_ap_password
                else:
                    self._camera_info.ip_info.wireless_ssid = self._camera_info.ip_info.wireless_sta_ssid
                    self._camera_info.ip_info.wireless_password = self._camera_info.ip_info.wireless_sta_password
            else:
                self._camera_info.ip_info.wireless_ssid = result['get_wireless']['ssid']
                self._camera_info.ip_info.wireless_password = result['get_wireless']['password']

                if ap_mode:
                    self._camera_info.ip_info.wireless_ap_ssid = self._camera_info.ip_info.wireless_ssid
                    self._camera_info.ip_info.wireless_ap_password = self._camera_info.ip_info.wireless_password
                else:
                    self._camera_info.ip_info.wireless_sta_ssid = self._camera_info.ip_info.wireless_ssid
                    self._camera_info.ip_info.wireless_sta_password = self._camera_info.ip_info.wireless_password
        except Exception as e:
            logging.warning('获取无线信息失败', e)

    def get_wb_info(self) -> Optional[WhiteBalanceInfo]:
        self._get_wb_info()
        return self._camera_info.white_balance

    def get_exposure_info(self) -> Optional[ExposureInfo]:
        self._get_exposure_info()
        return self._camera_info.exposure
