﻿#include <SDL2/SDL.h>
#include <stdio.h>
#ifdef _WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif
#include <string.h>

#include <ctime>
#include <iomanip>
#include <iostream>
#include <map>
#include <random>
#include <string>

#include "slcam.h"

// 注意：如需使用RGB格式的视频帧，取消以下的注释即可
//       但会消耗额外的资源, 并可能增加画面延时
// #define SLDEMO_RGB

#define SLALIGN(x, a) (((x) + (a) - 1) & ~((a) - 1))

using namespace std;

void my_sleep(int ms)
{
#ifdef _WIN32
    Sleep(ms);
#else
    usleep(ms * 1000);
#endif
}

void log_callback(int level, const char *msg)
{
    std::cout << "Level=" << level << ", msg=" << msg << std::endl;
}

void upgrade_callback(int percent, void *ctx)
{
    std::cout << "Percent=" << percent
              << ", ctx=" << *(reinterpret_cast<int *>(ctx)) << std::endl;
}
int main(int argc, char *argv[])
{
    slcam_api_init();
    slcam_log_set_level(SLCAM_LOG_INFO);
    // slcam_log_set_pattern("*** [%H:%M:%S %z] [thread %t] %v ***");
    // slcam_log_set_callback(log_callback);

    int select = 0;
    SLcamDevInfos enumInfo;

    while (true)
    {
        slcam_enum_devices(&enumInfo);
        std::cout << "##### 检测当前环境Camera设备" << std::endl;
        for (int i = 0; i < enumInfo.cameraNum; i++)
        {
            std::cout << "相机" << i + 1
                      << ": 名称=" << enumInfo.cameras[i].name;
            std::cout << ", UniqueName=" << enumInfo.cameras[i].uniqueName;
            std::cout << ", PID=" << std::hex << enumInfo.cameras[i].model;
            std::cout << std::resetiosflags(std::ios::basefield)
                      << ", USB速率=" << enumInfo.cameras[i].speed << std::endl;
        }
        std::cout << "##### 请选择相机, 0重新搜索可用相机, -1退出选择"
                  << std::endl;
        std::cin >> select;
       // std::cout << "select=" << select << std::endl;

        select = select - 1;

        if (select == (0 - 1))
            continue;

        break;
    }

    if (enumInfo.cameraNum == 0 || select < 0 || select >= enumInfo.cameraNum)
    {
        slcam_api_destroy();
        return 0;
    }

    HSLcam cam = nullptr;
    if (slcam_open(enumInfo.cameras[select].uniqueName, &cam) !=
        SLCAMRET_SUCCESS)
    {
        std::cout << "无法打开相机 " << enumInfo.cameras[select].uniqueName
                  << std::endl;
        slcam_api_destroy();
        system("pause");
        return -1;
    }

    SLcamVideoCaptureCapabilities capabilities;
    memset(&capabilities, 0, sizeof(capabilities));
    slcam_get_capture_capabilities(cam, &capabilities);
    for (int i = 0; i < capabilities.capNum; ++i)
    {
        std::cout << "Cap " << i
                  << ", 宽=" << capabilities.videoCaps[i].resolution.width
                  << ", 高=" << capabilities.videoCaps[i].resolution.height
                  << ", 帧率=" << capabilities.videoCaps[i].maxFps
                  << ", 格式=" << capabilities.videoCaps[i].videoFmt
                  << std::endl;
    }

    int selectedCapIndex = -1;
    while (true)
    {
        std::cout << "##### 请选择需要打开的分辨率、格式, 例如0" << std::endl;
        std::cin >> selectedCapIndex;
        if (selectedCapIndex < 0 || selectedCapIndex >= capabilities.capNum)
        {
            std::cout << "选择错误, 请重新选择!";
            continue;
        }
        break;
    }

    SLcamVideoCaptureCapability selectedCap =
        capabilities.videoCaps[selectedCapIndex];
    int width = selectedCap.resolution.width;
    int height = selectedCap.resolution.height;
    SLcamCaptureContext capCtx;
#ifdef _WIN32
    strcpy_s(capCtx.uniqueName, 128, enumInfo.cameras[select].uniqueName);
#else
    strncpy(capCtx.uniqueName, enumInfo.cameras[select].uniqueName, 128);
#endif
    capCtx.capFmt = selectedCap.videoFmt;
#ifndef SLDEMO_RGB
    capCtx.readFmt = SLCAM_PIX_FORMAT_I420;
#else
    capCtx.readFmt = SLCAM_PIX_FORMAT_RGB24;
#endif
    capCtx.resolution = selectedCap.resolution;
    capCtx.fps = selectedCap.maxFps;

    if (slcam_set_capture_context(cam, &capCtx) != SLCAMRET_SUCCESS)
    {
        std::cout << "相机视频采集参数设置失败!" << std::endl;
        slcam_close(cam);
        slcam_api_destroy();
        return -1;
    }

    SDL_Event event;
    SDL_Window *window = NULL;
    SDL_Renderer *sdlRenderer = NULL;
    SDL_Texture *sdlTexture = NULL;
    SDL_Rect sdlRect;
    const int screen_w = 960;
    const int screen_h = 540;
    sdlRect.x = 0;
    sdlRect.y = 0;
    sdlRect.w = screen_w;
    sdlRect.h = screen_h;
#ifndef SLDEMO_RGB
    Uint32 pixformat = SDL_PIXELFORMAT_IYUV;
#else
    Uint32 pixformat = SDL_PIXELFORMAT_RGB24;
#endif
    SDL_Init(SDL_INIT_VIDEO);
    // Create sdl window
    window = SDL_CreateWindow(
        "Capture Demo", SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED,
        screen_w, screen_h, SDL_WINDOW_RESIZABLE);

    for (int i = 0; i < SDL_GetNumRenderDrivers(); ++i)
    {
        SDL_RendererInfo rendererInfo = {};
        SDL_GetRenderDriverInfo(i, &rendererInfo);
        if (rendererInfo.name != std::string("direct3d"))
        {
            continue;
        }

        sdlRenderer = SDL_CreateRenderer(
            window, i, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);
        break;
    }

    if (!sdlRenderer)
        sdlRenderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_PRESENTVSYNC);

    sdlTexture = SDL_CreateTexture(
        sdlRenderer, pixformat, SDL_TEXTUREACCESS_STREAMING, width, height);

    SLcamVideoFrame frame;
    SDL_bool done = SDL_FALSE;
    Uint32 windowID = SDL_GetWindowID(window);
    int32_t focusValue = 0;
    int32_t maxFocusValue = 0;
    int32_t minFocusValue = 0;
    int32_t defFocusValue = 0;
    int32_t stepFocusValue = 0;
    int32_t state = 0;
    bool isSnap = false;
    int32_t snapNum = 0;
    bool isRec = false;
    int32_t value = 0, min_value = 0, max_value = 0, def_value = 0,
            step_value = 0, user_id = 0, mode = 0;

    std::random_device rd;
    std::mt19937 gen(rd());
    while (true)
    {
        while (SDL_PollEvent(&event))
        {
            switch (event.type)
            {
                case SDL_KEYDOWN:
                    if (event.key.keysym.sym == SDLK_ESCAPE)
                    {
                        done = SDL_TRUE;
                    }
                    else if (event.key.keysym.sym == SDLK_a)
                    {
                        int32_t enable = 0;
                        int32_t mode = 0;
                        int32_t x = 0;
                        int32_t y = 0;
                        slcam_set_focus_region_v1(cam, 6, 7);
                        my_sleep(1000);
                        slcam_get_focus_region_v1(cam, &x, &y);
                        std::cout << "slcam_get_focus_region, enable=" << enable
                                  << ", mode=" << mode << ", x=" << x
                                  << ", y=" << y << std::endl;
                    }
                    else if (event.key.keysym.sym == SDLK_s)
                    {
                        int32_t state = 0;
                        slcam_get_focus_state(cam, &state);
                        cout << "##### 获取当前聚焦状态："
                             << (state == 0 ? "聚焦中" : "聚焦完成") << endl;
                        slcam_get_focus(cam, &value);
                        cout << "##### 获取当前手动聚焦值" << value << endl;
                    }
                    else if (event.key.keysym.sym == SDLK_d)
                    {
                        cout << "##### 设置聚焦模式为手动"
                             << slcam_set_focus_mode(
                                    cam, SLCAM_FOCUS_MODE_MANUAL)
                             << endl;
                        slcam_set_focus(cam, 2);
                        cout << "##### 设置手动聚焦值为2" << endl;
                        cout << "##### 设置聚焦模式为自动"
                             << slcam_set_focus_mode(cam, SLCAM_FOCUS_MODE_AUTO)
                             << endl;
                        // int32_t brightness = 0;
                        // int32_t time = 0;
                        // int32_t gain = 0;
                        // slcam_get_exposure_compensation(cam, &brightness);
                        // slcam_get_exposure_time(cam, &time);
                        // slcam_get_exposure_gain(cam, &gain);
                        // std::cout << "slcam_get_exp, brightness=" <<
                        // brightness
                        //           << ", time=" << time << ", gain=" << gain
                        //           << std::endl;
                    }
                    else if (event.key.keysym.sym == SDLK_z)
                    {
                        int x = 0, y = 0, w = 200, h = 200;
                        slcam_set_exposure_mode(cam, SLCAM_EXPOSURE_MODE_AUTO);
                        my_sleep(1000);
                        if (slcam_set_auto_exposure_region(cam, x, y, w, h) ==
                            SLCAMRET_SUCCESS)
                        {
                            std::cout << "exp region get"
                                      << ", x = " << x << ", y = " << y
                                      << ", w = " << w << ",h = " << h
                                      << std::endl;
                        }
                    }
                    else if (event.key.keysym.sym == SDLK_q)
                    {
                        int32_t led_number = 0;
                        int32_t part_number = 0;
                        int32_t min_level = 0;
                        int32_t max_level = 0;
                        int32_t def_level = 0;
                        int32_t step_level = 0;

                        slcam_get_led_info(
                            cam, &led_number, &part_number, &min_level,
                            &max_level, &def_level, &step_level);

                        std::cout
                            << "slcam_get_led_info, led_number=" << led_number
                            << ", part_number=" << part_number
                            << ", min_level=" << min_level
                            << ", max_level=" << max_level
                            << ", def_level=" << def_level
                            << ", step_level=" << step_level << std::endl;
                    }
                    else if (event.key.keysym.sym == SDLK_w)
                    {
                        int8_t zoom_rel = 0;
                        uint8_t digital_zoom = 0;
                        uint8_t speed = 0;
                        slcam_get_zoom_relative(
                            cam, &zoom_rel, &digital_zoom, &speed);
                        std::cout << "slcam_get_zoom_relative, zoom_rel="
                                  << static_cast<int32_t>(zoom_rel)
                                  << ", digital_zoom="
                                  << static_cast<int32_t>(digital_zoom)
                                  << ", speed=" << static_cast<int32_t>(speed)
                                  << std::endl;
                    }
                    else if (event.key.keysym.sym == SDLK_e)
                    {
                        cout << "##### 修改曝光模式为手动曝光："
                             << slcam_set_exposure_mode(cam, 0) << endl;
                        my_sleep(1000);
                        slcam_get_exposure_mode(cam, &mode);
                        cout << "##### 获取曝光模式为："
                             << (mode == SLCAM_EXPOSURE_MODE_AUTO ? "自动"
                                                                  : "手动")
                             << endl;
                        // 获取快门值范围
                        my_sleep(1000);
                        slcam_get_exposure_time_range(
                            cam, &min_value, &max_value, &def_value,
                            &step_value);
                        cout << "快门:最小值=" << min_value
                             << ", 最大值=" << max_value
                             << "，默认值=" << def_value
                             << ",步长=" << step_value << endl;
                        // 修改快门值并验证是否修改成功
                        cout << "##### 修改快门值为最大："
                             << slcam_set_exposure_time(cam, max_value) << endl;
                        my_sleep(1000);
                        slcam_get_exposure_time(cam, &value);
                        cout << "##### 获取当前快门值为：" << value << endl;
                    }
                    else if (event.key.keysym.sym == SDLK_y)
                    {
                        int32_t value = 0, min_value = 0, max_value = 0,
                                def_value = 0, step_value = 0, user_id = 0;

                        // 修改用户ID
                        cout << "##### 修改用户ID为67108863："
                             << slcam_set_user_id(cam, 6710880) << endl;
                        my_sleep(1500);
                        slcam_get_user_id(cam, &user_id);
                        cout << "##### 获取修改后用户id：" << user_id << endl;
                        cout << "##### 修改用户ID为0："
                             << slcam_set_user_id(cam, 0) << endl;
                        my_sleep(1500);
                        slcam_get_user_id(cam, &user_id);
                        cout << "##### 获取此时用户ID：" << user_id << endl;
                    }
                    else if (event.key.keysym.sym == SDLK_t)
                    {
                        if (slcam_set_zoom_relative(cam, 0, 0, 0) !=
                            SLCAMRET_SUCCESS)
                        {
                            std::cout << "slcam_set_zoom_relative failed!"
                                      << std::endl;
                        }
                    }
                    else if (event.key.keysym.sym == SDLK_u)
                    {
                        uint8_t unique_id[60] = {0};
                        int32_t read_len = 0;
                        if (slcam_get_unique_id(
                                cam, unique_id, sizeof(unique_id), &read_len) ==
                            SLCAMRET_SUCCESS)
                        {
                            std::cout << "slcam_get_unique_id, unique_id=";
                            std::cout << std::showbase << std::hex;
                            for (int i = 0; i < read_len; ++i)
                            {
                                std::cout << static_cast<int32_t>(unique_id[i])
                                          << " ";
                            }
                            std::cout << std::resetiosflags(std::ios::basefield)
                                      << ", read_len=" << read_len << std::endl;
                        }
                    }
                    // 设置 Gamma 模式至 gamma 曲线模式
                    else if (event.key.keysym.sym == SDLK_4)
                    {
                        if (slcam_set_gamma_mode(
                                cam, SLCAM_GAMMA_MODE_BEZIER) ==
                            SLCAMRET_SUCCESS)
                        {
                            std::cout << "Gamma Mode set to: " << mode
                                      << std::endl;
                        }
                        else
                        {
                            std::cout << "Failed to set Gamma Mode!"
                                      << std::endl;
                        }
                        if (slcam_get_gamma_mode(cam, &mode) ==
                            SLCAMRET_SUCCESS)
                        {
                            std::cout << "Current Gamma Mode: " << mode
                                      << std::endl;
                        }
                        else
                        {
                            std::cout << "Failed to get Gamma Mode!"
                                      << std::endl;
                        }
                    }
                    // 设置 Gamma 贝塞尔曲线点位
                    else if (event.key.keysym.sym == SDLK_5)
                    {
                        map<int, int> bezier_map;
                        const int num_points = 25;
                        SLPoint bezier_points[num_points];
                        std::uniform_int_distribution<> xUnf(0, 255);
                        std::uniform_int_distribution<> yUnf(0, 255);

                        // 往map里面添加直至数量到达25
                        // map insert时会去重，并且按照大小排序
                        while (bezier_map.size() < num_points)
                        {
                            int x = xUnf(gen);
                            int y = yUnf(gen);
                            auto it = bezier_map.insert({x, y});
                        }

                        int index = 0;
                        // 把map中存放的x,y放回bezier_points中
                        // 使用迭代器遍历map
                        for (auto it = bezier_map.begin();
                             it != bezier_map.end() && index < num_points; ++it)
                        {
                            bezier_points[index].x = it->first;
                            bezier_points[index].y = it->second;

                            cout << "Index= " << index << ", X="
                                 << static_cast<int>(bezier_points[index].x)
                                 << ",Y= "
                                 << static_cast<int>(bezier_points[index].y)
                                 << endl;
                            ++index;
                        }

                        // 调用设置 Gamma 贝塞尔曲线的函数
                        int32_t result = slcam_set_gamma_bezier_curve(
                            cam, bezier_points, num_points);

                        if (result == SLCAMRET_SUCCESS)
                        {
                            std::cout << "Gamma 贝塞尔曲线设置成功!"
                                      << std::endl;
                        }
                        else
                        {
                            std::cout << "设置 Gamma 贝塞尔曲线失败!"
                                      << std::endl;
                        }
                    }
                    else if (event.key.keysym.sym == SDLK_3)  // 按3控制拍照
                    {
                        isSnap = true;
                    }
                    else if (event.key.keysym.sym == SDLK_r)  // 按r控制录像
                    {
                        if (!isRec)
                        {
                            std::string filePath = "./";
                            filePath += "img/record1.mp4";
                            SLcamRecordSaveInfo save_info;
                            save_info.savePath =
                                const_cast<char *>(filePath.c_str());
                            // 录像分辨率必须与当前分辨率一致
                            save_info.width = capCtx.resolution.width;
                            save_info.height = capCtx.resolution.height;
                            save_info.fps = capCtx.fps;
                            if (slcam_record_start(cam, save_info) ==
                                SLCAMRET_SUCCESS)
                            {
                                std::cout << "##### 开始录像，文件保存位置："
                                          << filePath << std::endl;
                                isRec = true;
                            }
                        }
                        else
                        {
                            isRec = false;
                            slcam_record_stop(cam);
                            std::cout << "##### 停止录像" << std::endl;
                        }
                    }
                    else if (event.key.keysym.sym == SDLK_c)
                    {
                        // 设置ROI
                        std::uniform_int_distribution<> xUnf(
                            0, capCtx.resolution.width);
                        std::uniform_int_distribution<> yUnf(
                            0, capCtx.resolution.height);
                        int getEnable = false;
                        int getX = 0;
                        int getY = 0;
                        int getWidth = 0;
                        int getHeight = 0;
                        slcam_get_roi_region(
                            cam, &getEnable, &getX, &getY, &getWidth,
                            &getHeight);
                        if (!getEnable)
                        {
                            int x, y, width, height;
                            x = SLALIGN(xUnf(gen), 2);
                            width = SLALIGN(xUnf(gen), 8);
                            y = SLALIGN(yUnf(gen), 2);
                            height = SLALIGN(yUnf(gen), 8);
                            slcam_set_roi_region(
                                cam, true, x, y, width, height);
                            std::cout << "Set roi, "
                                      << " enable= " << true << ", x=" << x
                                      << ", y=" << y << ", width=" << width
                                      << ", height=" << height << std::endl;

                            slcam_get_roi_region(
                                cam, &getEnable, &getX, &getY, &getWidth,
                                &getHeight);
                            std::cout << "Get roi, "
                                      << " enable= " << getEnable
                                      << ", x=" << getX << ", y=" << getY
                                      << ", width=" << getWidth
                                      << ", height=" << getHeight << std::endl;
                        }
                        else
                        {
                            slcam_set_roi_region(
                                cam, false, -100, -200, 300, 400);
                        }
                    }
                    else if (event.key.keysym.sym == SDLK_v)
                    {
                        // 切换分辨率
                        selectedCap =
                            capabilities.videoCaps
                                [++selectedCapIndex % capabilities.capNum];
                        capCtx.capFmt = selectedCap.videoFmt;
                        capCtx.resolution = selectedCap.resolution;
                        capCtx.fps = selectedCap.maxFps;
                        slcam_set_capture_context(cam, &capCtx);
                    }
                    break;
                case SDL_QUIT:
                    done = SDL_TRUE;
                    break;
                case SDL_WINDOWEVENT:
                    if (event.window.windowID == windowID &&
                        event.window.event == SDL_WINDOWEVENT_RESIZED)
                    {
                        sdlRect.w = event.window.data1;
                        sdlRect.h = event.window.data2;
                    }
                    else if (
                        event.window.windowID == windowID &&
                        event.window.event == SDL_WINDOWEVENT_MINIMIZED)
                    {
                        while (SDL_WaitEvent(&event))
                        {
                            if (event.window.windowID == windowID &&
                                event.window.event == SDL_WINDOWEVENT_RESTORED)
                            {
                                break;
                            }
                        }
                    }
                    break;
            }
        }

        if (done)
            break;

        if (slcam_read_frame(cam, &frame) == SLCAMRET_SUCCESS)
        {
            if (frame.width != width || frame.height != height)
            {
                width = frame.width;
                height = frame.height;
                std::cout << "Video resolution changed, new width=" << width
                          << ", new height=" << height << std::endl;
                SDL_DestroyTexture(sdlTexture);
                sdlTexture = SDL_CreateTexture(
                    sdlRenderer, pixformat, SDL_TEXTUREACCESS_STREAMING, width,
                    height);
            }
#ifndef SLDEMO_RGB
            SDL_UpdateYUVTexture(
                sdlTexture, NULL, frame.data[0], frame.linesize[0],
                frame.data[1], frame.linesize[1], frame.data[2],
                frame.linesize[2]);
#else
            SDL_UpdateTexture(
                sdlTexture, NULL, frame.data[0], frame.linesize[0]);
#endif
            // 拍照
            if (isSnap)
            {
                isSnap = false;
                char filePath[1024] = {0};
#ifdef _WIN32
                sprintf_s(filePath, "./img/img_%d", snapNum++);
#else
                snprintf(filePath, 1024, "./img/img_%d", snapNum++);
#endif
                SLcamFileSaveInfo fileSaveInfo;
                fileSaveInfo.savePath = filePath;
                fileSaveInfo.format = SLCAM_IMG_FORMAT_BMP;
                fileSaveInfo.frame = &frame;
                if (slcam_file_save_image(cam, fileSaveInfo) ==
                    SLCAMRET_SUCCESS)
                    std::cout << "##### 图片保存成功， 路径为：" << filePath
                              << std::endl;
                fileSaveInfo.format = SLCAM_IMG_FORMAT_PNG;
                slcam_file_save_image(cam, fileSaveInfo);
                fileSaveInfo.format = SLCAM_IMG_FORMAT_JPG;
                slcam_file_save_image(cam, fileSaveInfo);
            }

            // 录像中需不断将待编码帧送入
            if (isRec)
                slcam_record_append_frame(cam, &frame);

            // printf(
            //     "width = %d, height = %d, format = %d, pts=%lld\n",
            //     frame.width, frame.height, frame.fmt, frame.pts);
            slcam_free_frame(&frame);

            SDL_RenderClear(sdlRenderer);
            SDL_RenderCopy(sdlRenderer, sdlTexture, NULL, &sdlRect);
            SDL_RenderPresent(sdlRenderer);
        }
        SDL_Delay(1);
    }

    slcam_close(cam);

    SDL_DestroyTexture(sdlTexture);
    SDL_DestroyRenderer(sdlRenderer);
    SDL_DestroyWindow(window);
    SDL_Quit();

    slcam_api_destroy();
    return 0;
}
