﻿#include <SDL2/SDL.h>
#include <stdio.h>
#include <string.h>

#include <iomanip>
#include <iostream>
#include <thread>
#include <vector>

#include "slcam.h"

SLcamDevInfos gEnumInfo;

void thread_capture(int index)
{
    SDL_Event event;
    SDL_Window *window = NULL;
    SDL_Renderer *sdlRenderer = NULL;
    SDL_Texture *sdlTexture = NULL;
    SDL_Rect sdlRect;
    const int screen_w = 960;
    const int screen_h = 540;
    sdlRect.x = 0;
    sdlRect.y = 0;
    sdlRect.w = screen_w;
    sdlRect.h = screen_h;
    Uint32 pixformat = SDL_PIXELFORMAT_IYUV;

    HSLcam cam = nullptr;

    if (slcam_open(gEnumInfo.cameras[index].uniqueName, &cam) !=
        SLCAMRET_SUCCESS)
    {
        std::cout << "相机打开失败, uniqueName= "
                  << gEnumInfo.cameras[index].uniqueName << std::endl;
        return;
    }
    SLcamVideoCaptureCapabilities caps;
    if (slcam_get_capture_capabilities(cam, &caps) != SLCAMRET_SUCCESS ||
        caps.capNum <= 0)
    {
        std::cout << "获取相机能力集失败, uniqueName="
                  << gEnumInfo.cameras[index].uniqueName << std::endl;
        slcam_close(cam);
        return;
    }

    SLcamCaptureContext capCtx;
    // 默认选择第一个视频格式取流
    SLcamVideoCaptureCapability selectedCap = caps.videoCaps[0];
#ifdef _WIN32
    strcpy_s(capCtx.uniqueName, 128, gEnumInfo.cameras[index].uniqueName);
#else
    strncpy(capCtx.uniqueName, gEnumInfo.cameras[index].uniqueName, 128);
#endif
    capCtx.capFmt = selectedCap.videoFmt;
    capCtx.fps = selectedCap.maxFps;
    capCtx.resolution = selectedCap.resolution;
    capCtx.readFmt = SLCAM_PIX_FORMAT_I420;
    if (slcam_set_capture_context(cam, &capCtx) != SLCAMRET_SUCCESS)
    {
        std::cout << "相机采集参数设置失败, uniqueName="
                  << gEnumInfo.cameras[index].uniqueName << std::endl;
        slcam_close(cam);
        return;
    }

    // Create sdl window
    window = SDL_CreateWindow(
        "Multi Capture Demo", SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED,
        screen_w, screen_h, SDL_WINDOW_RESIZABLE);

    for (int i = 0; i < SDL_GetNumRenderDrivers(); ++i)
    {
        SDL_RendererInfo rendererInfo = {};
        SDL_GetRenderDriverInfo(i, &rendererInfo);
        if (rendererInfo.name != std::string("direct3d"))
        {
            continue;
        }

        sdlRenderer = SDL_CreateRenderer(
            window, i, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);
        break;
    }

    if (!sdlRenderer)
        sdlRenderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_PRESENTVSYNC);

    sdlTexture = SDL_CreateTexture(
        sdlRenderer, pixformat, SDL_TEXTUREACCESS_STREAMING,
        capCtx.resolution.width, capCtx.resolution.height);

    SDL_bool done = SDL_FALSE;
    Uint32 windowID = SDL_GetWindowID(window);
    while (true)
    {
        while (SDL_PollEvent(&event))
        {
            switch (event.type)
            {
                case SDL_KEYDOWN:
                    if (event.key.keysym.sym == SDLK_ESCAPE)
                    {
                        done = SDL_TRUE;
                    }
                    else if (event.key.keysym.sym == SDLK_a)  // 按下A键触发
                    {
                        // slcam_set_white_balance_temperature(cameras[index],
                        // 5000);
                    }
                    break;
                case SDL_QUIT:
                    done = SDL_TRUE;
                case SDL_WINDOWEVENT:
                    if (event.window.windowID == windowID &&
                        event.window.event == SDL_WINDOWEVENT_RESIZED)
                    {
                        sdlRect.w = event.window.data1;
                        sdlRect.h = event.window.data2;
                    }
                    else if (
                        event.window.windowID == windowID &&
                        event.window.event == SDL_WINDOWEVENT_MINIMIZED)
                    {
                        while (SDL_WaitEvent(&event))
                        {
                            if (event.window.windowID == windowID &&
                                event.window.event == SDL_WINDOWEVENT_RESTORED)
                            {
                                break;
                            }
                        }
                    }
                    break;
            }
        }

        if (done)
            break;

        SLcamVideoFrame frame;
        if (slcam_read_frame(cam, &frame) == SLCAMRET_SUCCESS)
        {
            if (frame.data[0] && frame.data[1] && frame.data[2])
            {
                SDL_UpdateYUVTexture(
                    sdlTexture, NULL, frame.data[0], frame.linesize[0],
                    frame.data[1], frame.linesize[1], frame.data[2],
                    frame.linesize[2]);
                slcam_free_frame(&frame);
            }
        }

        SDL_RenderClear(sdlRenderer);
        SDL_RenderCopy(sdlRenderer, sdlTexture, NULL, &sdlRect);
        SDL_RenderPresent(sdlRenderer);
    }

    slcam_close(cam);
    SDL_DestroyTexture(sdlTexture);
    SDL_DestroyRenderer(sdlRenderer);
    SDL_DestroyWindow(window);
}

int main(int argc, char *argv[])
{
    std::vector<std::thread> threads;

    SDL_Init(SDL_INIT_VIDEO);

    slcam_api_init();
    slcam_log_set_level(SLCAM_LOG_INFO);
    slcam_enum_devices(&gEnumInfo);
    std::cout << "##### 检测当前环境Camera设备" << std::endl;
    for (int i = 0; i < gEnumInfo.cameraNum; i++)
    {
        std::cout << "相机" << i << ": 名称=" << gEnumInfo.cameras[i].name;
        std::cout << ", UniqueName=" << gEnumInfo.cameras[i].uniqueName;
        std::cout << ", PID=" << std::hex << gEnumInfo.cameras[i].model;
        std::cout << std::resetiosflags(std::ios::basefield)
                  << ", USB速率=" << gEnumInfo.cameras[i].speed << std::endl;
        threads.emplace_back(thread_capture, i);
    }

    for (auto &t : threads)
        t.join();

    slcam_api_destroy();

    SDL_Quit();
    return 0;
}
