#include "mainwindow.h"

#include <QDateTime>
#include <QDebug>
#include <QFileDialog>
#include <QMessageBox>
#include <QStandardPaths>

#include "imgconverthelper.h"
#include "ui_mainwindow.h"

namespace
{
constexpr int timeOutToKillThreadMs = 2000;
}

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), ui(new Ui::MainWindow)
{
    ui->setupUi(this);
    ui->video_tabWidget->setTabsClosable(true);

    registerMetaTypes();

    slcam_api_init();
    slcam_log_set_level(SLCAM_LOG_INFO);
}

MainWindow::~MainWindow()
{
    if (m_camCapThread)
    {
        m_camCapThread->closeVideoThread();
        m_camCapThread->quit();
        if (!m_camCapThread->wait(timeOutToKillThreadMs))
        {
            qWarning("CameraThread should be terminated.");
            m_camCapThread->terminate();
        }
    }

    if (m_currentCam)
        slcam_close(m_currentCam);

    slcam_api_destroy();
    delete ui;
}

void MainWindow::openUVCCamera(
    const QString &friendlyName, const QString &uniqueName)
{
    Q_UNUSED(friendlyName);
    qDebug() << "Open UVC Camera, name=" << uniqueName;
    if (m_camCapThread)
    {
        m_camCapThread->closeVideoThread();
        m_camCapThread->quit();
        m_camCapThread->wait();
    }

    SLcamCaptureContext ctx;
    QByteArray ba = uniqueName.toLatin1();
#ifdef _WIN32
    strcpy_s(ctx.uniqueName, 128, ba.data());
#else
    strncpy(ctx.unique_name, ba.data(), 128);
#endif
    if (slcam_open(ctx.uniqueName, &m_currentCam) != SLCAMRET_SUCCESS)
    {
        qDebug() << "Failed to open camera, name=" << uniqueName;
        return;
    }

    int targetWidth = 1920;
    int targetHeight = 1080;
    SLcamVideoCaptureCapabilities caps;
    slcam_get_capture_capabilities(m_currentCam, &caps);
    int selectIndex = 0;
    for (int i = 0; i < caps.capNum; ++i)
    {
        if (caps.videoCaps[i].resolution.width >= targetWidth ||
            caps.videoCaps[i].resolution.height >= targetHeight)
        {
            selectIndex = i;
            break;
        }
    }
    ctx.readFmt = SLCAM_PIX_FORMAT_I420;
    ctx.capFmt = caps.videoCaps[selectIndex].videoFmt;
    ctx.resolution.width = caps.videoCaps[selectIndex].resolution.width;
    ctx.resolution.height = caps.videoCaps[selectIndex].resolution.height;
    ctx.fps = caps.videoCaps[selectIndex].maxFps;
    qDebug() << "width=" << ctx.resolution.width << ", height=" << ctx.resolution.height;

    m_camCapThread.reset(new VideoThread(m_currentCam));
    m_videoWidget.reset(new OpenGLVideoWidget());
    ui->video_tabWidget->setCurrentIndex(
        ui->video_tabWidget->insertTab(0, m_videoWidget.get(), "UVC Camera"));
    connect(
        m_camCapThread.get(), &VideoThread::sendFrameForDisplay,
        m_videoWidget.get(), &OpenGLVideoWidget::onFrame);
    connect(
        m_camCapThread.get(), &VideoThread::closeCamera, this,
        &MainWindow::on_closeVideo_btn_clicked);
    connect(
        this, &MainWindow::getFrameForSnap, m_camCapThread.get(),
        &VideoThread::onGetFrameForSnap);
    connect(
        m_camCapThread.get(), &VideoThread::sendFrameForSnap, this,
        &MainWindow::onGetSnapFrame);
    connect(
        this, &MainWindow::startRecording, m_camCapThread.get(),
        &VideoThread::onStartRecording);
    connect(
        this, &MainWindow::stopRecording, m_camCapThread.get(),
        &VideoThread::onStopRecording);

    m_videoWidget->startPlay();
    m_camCapThread->openVideoThread();
    m_camCapThread->setCapctx(ctx);
    getCamInfo(m_camInfo, m_currentCam);
    setWidgetsEnbaleStatus(true);
    updateCamUi(m_camInfo);
    m_camCapThread->start();
}

void MainWindow::getCamInfo(CameraInfo &info, HSLcam cam)
{
    if (!cam)
        return;

    slcam_get_version(cam, &info.version);
    // 曝光
    int32_t value = 0;
    int32_t ret = SLCAMRET_FAILURE;
    ret = slcam_get_exposure_mode(cam, &value);
    info.exposure.enable = (ret == SLCAMRET_SUCCESS);
    if (info.exposure.enable)
    {
        info.exposure.mode = static_cast<SLcamExposureMode>(value);
        // 部分相机BUG: 增大亮度后, 重连相机首次获取亮度值必失败
        // 连续获取两次来规避此BUG
        ret = slcam_get_exposure_compensation(
            cam, &info.exposure.compensation.cur);
        if (ret != SLCAMRET_SUCCESS)
            ret = slcam_get_exposure_compensation(
                cam, &info.exposure.compensation.cur);
        info.exposure.compensation.enable = (ret == SLCAMRET_SUCCESS);

        ret = slcam_get_exposure_time(cam, &info.exposure.time.cur);
        info.exposure.time.enable = (ret == SLCAMRET_SUCCESS);

        ret = slcam_get_exposure_gain(cam, &info.exposure.gain.cur);
        info.exposure.gain.enable = (ret == SLCAMRET_SUCCESS);

        slcam_get_exposure_compensation_range(
            cam, &info.exposure.compensation.min,
            &info.exposure.compensation.max, &info.exposure.compensation.def,
            &info.exposure.compensation.step);
        slcam_get_exposure_time_range(
            cam, &info.exposure.time.min, &info.exposure.time.max,
            &info.exposure.time.def, &info.exposure.time.step);
        slcam_get_exposure_gain_range(
            cam, &info.exposure.gain.min, &info.exposure.gain.max,
            &info.exposure.gain.def, &info.exposure.gain.step);
    }

    // 白平衡
    ret = slcam_get_white_balance_mode(cam, &value);
    info.whiteBalance.enable = (ret == SLCAMRET_SUCCESS);
    if (info.whiteBalance.enable)
    {
        info.whiteBalance.mode = static_cast<SLcamWhiteBalanceMode>(value);
        ret = slcam_get_white_balance_component_red(
            cam, &info.whiteBalance.red.cur);
        info.whiteBalance.red.enable = (ret == SLCAMRET_SUCCESS);

        ret = slcam_get_white_balance_component_green(
            cam, &info.whiteBalance.green.cur);
        info.whiteBalance.green.enable = (ret == SLCAMRET_SUCCESS);

        ret = slcam_get_white_balance_component_blue(
            cam, &info.whiteBalance.blue.cur);
        info.whiteBalance.blue.enable = (ret == SLCAMRET_SUCCESS);

        slcam_get_white_balance_component_red_range(
            cam, &info.whiteBalance.red.min, &info.whiteBalance.red.max,
            &info.whiteBalance.red.def, &info.whiteBalance.red.step);
        slcam_get_white_balance_component_green_range(
            cam, &info.whiteBalance.green.min, &info.whiteBalance.green.max,
            &info.whiteBalance.green.def, &info.whiteBalance.green.step);
        slcam_get_white_balance_component_blue_range(
            cam, &info.whiteBalance.blue.min, &info.whiteBalance.blue.max,
            &info.whiteBalance.blue.def, &info.whiteBalance.blue.step);
    }
}

void MainWindow::updateCamUi(const CameraInfo &camInfo)
{
    m_sendMessageToCamera = false;

    ui->brightness_slider->setRange(
        camInfo.exposure.compensation.min, camInfo.exposure.compensation.max);
    ui->gain_slider->setRange(
        camInfo.exposure.gain.min, camInfo.exposure.gain.max);
    ui->shutter_slider->setRange(
        camInfo.exposure.time.min, camInfo.exposure.time.max);
    if (camInfo.exposure.mode == SLCAM_EXPOSURE_MODE_AUTO)
    {
        ui->autoExp_btn->setChecked(true);
        ui->brightness_slider->setEnabled(true);
        ui->gain_slider->setEnabled(false);
        ui->shutter_slider->setEnabled(false);
        ui->brightness_slider->setValue(camInfo.exposure.compensation.cur);
    }
    else if (camInfo.exposure.mode == SLCAM_EXPOSURE_MODE_MANUAL)
    {
        ui->manualExp_btn->setChecked(true);
        ui->brightness_slider->setEnabled(false);
        ui->gain_slider->setEnabled(true);
        ui->shutter_slider->setEnabled(true);
        ui->brightness_slider->setValue(camInfo.exposure.compensation.cur);
        ui->gain_slider->setValue(camInfo.exposure.gain.cur);
        ui->shutter_slider->setValue(camInfo.exposure.time.cur);
    }

    ui->red_slider->setRange(
        camInfo.whiteBalance.red.min, camInfo.whiteBalance.red.max);
    ui->green_slider->setRange(
        camInfo.whiteBalance.green.min, camInfo.whiteBalance.green.max);
    ui->blue_slider->setRange(
        camInfo.whiteBalance.blue.min, camInfo.whiteBalance.blue.max);

    if (camInfo.whiteBalance.mode == SLCAM_WHITE_BALANCE_MODE_AUTO)
    {
        ui->autoWB_btn->setChecked(true);
        ui->red_slider->setEnabled(false);
        ui->green_slider->setEnabled(false);
        ui->blue_slider->setEnabled(false);
    }
    else if (camInfo.whiteBalance.mode == SLCAM_WHITE_BALANCE_MODE_MANUAL)
    {
        ui->manualWB_btn->setChecked(true);
        ui->red_slider->setEnabled(true);
        ui->green_slider->setEnabled(true);
        ui->blue_slider->setEnabled(true);
        ui->red_slider->setValue(camInfo.whiteBalance.red.cur);
        ui->green_slider->setValue(camInfo.whiteBalance.green.cur);
        ui->blue_slider->setValue(camInfo.whiteBalance.green.cur);
    }

    m_sendMessageToCamera = true;
}

void MainWindow::registerMetaTypes()
{
    qRegisterMetaType<VideoFrame>("VideoFrame");
    qRegisterMetaType<VideoFrame>("VideoFrame&");
}

void MainWindow::setWidgetsEnbaleStatus(bool cameraOpen)
{
    ui->autoExp_btn->setEnabled(cameraOpen);
    ui->manualExp_btn->setEnabled(cameraOpen);
    ui->brightness_slider->setEnabled(cameraOpen);
    ui->gain_slider->setEnabled(cameraOpen);
    ui->shutter_slider->setEnabled(cameraOpen);

    ui->autoWB_btn->setEnabled(cameraOpen);
    ui->manualWB_btn->setEnabled(cameraOpen);
    ui->red_slider->setEnabled(cameraOpen);
    ui->green_slider->setEnabled(cameraOpen);
    ui->blue_slider->setEnabled(cameraOpen);
}

void MainWindow::on_scan_btn_clicked()
{
    SLcamDevInfos devInfos;
    memset(&devInfos, 0, sizeof(devInfos));
    slcam_enum_devices(&devInfos);
    QMap<QString, QString> cameraNames;

    int uvcCameraCount = 0;
    for (int i = 0; i < devInfos.cameraNum; i++)
    {
        // qDebug() << "Camera " << i << ": " <<
        // devInfos.cameras[i].unique_name;
        QString friendlyName = "SLCam_" + QString::number(++uvcCameraCount);
        cameraNames.insert(devInfos.cameras[i].uniqueName, friendlyName);
    }

    if (m_cameraNames == cameraNames)
        return;

    m_cameraNames = cameraNames;
    for (auto it = m_connectingDevices.begin(); it != m_connectingDevices.end();
         ++it)
    {
        if (!m_cameraNames.contains(it.key()))
            m_cameraNames.insert(it.key(), it.value());
    }

    ui->deviceList_comboBox->clear();
    int index = 0;
    for (auto it = m_cameraNames.begin(); it != m_cameraNames.end();
         ++it, ++index)
    {
        ui->deviceList_comboBox->addItem(it.value(), it.key());
    }
}

void MainWindow::on_openVideo_btn_clicked()
{
    if (ui->deviceList_comboBox->count() <= 0 || m_cameraOpend)
        return;

    // 仅允许连接单台相机
    if (m_connectingDevices.count() > 0)
        return;

    QString uniqueName = ui->deviceList_comboBox->currentData().toString();
    QString friendlyName = ui->deviceList_comboBox->currentText();

    if (m_connectingDevices.find(uniqueName) != m_connectingDevices.end())
        return;

    m_connectingDevices.insert(uniqueName, friendlyName);

    m_cameraOpend = !m_cameraOpend;

    openUVCCamera(friendlyName, uniqueName);
}

void MainWindow::on_autoWB_btn_clicked()
{
    if (!m_currentCam)
        return;

    ui->red_slider->setEnabled(false);
    ui->green_slider->setEnabled(false);
    ui->blue_slider->setEnabled(false);
    if (m_sendMessageToCamera)
        slcam_set_white_balance_mode(
            m_currentCam, SLCAM_WHITE_BALANCE_MODE_AUTO);
}

void MainWindow::on_manualWB_btn_clicked()
{
    if (!m_currentCam)
        return;

    m_sendMessageToCamera = false;
    ui->red_slider->setEnabled(true);
    ui->green_slider->setEnabled(true);
    ui->blue_slider->setEnabled(true);

    slcam_set_white_balance_mode(m_currentCam, SLCAM_WHITE_BALANCE_MODE_MANUAL);

    int value = 0;
    slcam_get_white_balance_component_red(m_currentCam, &value);
    ui->red_slider->setValue(value);

    slcam_get_white_balance_component_green(m_currentCam, &value);
    ui->green_slider->setValue(value);

    slcam_get_white_balance_component_blue(m_currentCam, &value);
    ui->blue_slider->setValue(value);

    m_sendMessageToCamera = true;
}

void MainWindow::on_red_slider_valueChanged(int value)
{
    if (!m_currentCam)
        return;

    ui->redValue_lab->setText(QString::number(value));
    if (m_sendMessageToCamera)
        slcam_set_white_balance_component_red(m_currentCam, value);
}

void MainWindow::on_green_slider_valueChanged(int value)
{
    if (!m_currentCam)
        return;

    ui->greenValue_lab->setText(QString::number(value));
    if (m_sendMessageToCamera)
        slcam_set_white_balance_component_green(m_currentCam, value);
}

void MainWindow::on_blue_slider_valueChanged(int value)
{
    if (!m_currentCam)
        return;

    ui->blueValue_lab->setText(QString::number(value));
    if (m_sendMessageToCamera)
        slcam_set_white_balance_component_blue(m_currentCam, value);
}

void MainWindow::on_autoExp_btn_clicked()
{
    if (!m_currentCam)
        return;

    m_sendMessageToCamera = false;
    ui->brightness_slider->setEnabled(true);
    ui->gain_slider->setEnabled(false);
    ui->shutter_slider->setEnabled(false);

    slcam_set_exposure_mode(m_currentCam, SLCAM_EXPOSURE_MODE_AUTO);

    int value = 0;
    slcam_get_exposure_compensation(m_currentCam, &value);
    ui->brightness_slider->setValue(value);

    m_sendMessageToCamera = true;
}

void MainWindow::on_manualExp_btn_clicked()
{
    if (!m_currentCam)
        return;

    m_sendMessageToCamera = false;

    ui->brightness_slider->setEnabled(false);
    ui->gain_slider->setEnabled(true);
    ui->shutter_slider->setEnabled(true);

    slcam_set_exposure_mode(m_currentCam, SLCAM_EXPOSURE_MODE_MANUAL);
    int value = 0;
    slcam_get_exposure_gain(m_currentCam, &value);
    ui->gain_slider->setValue(value);

    slcam_get_exposure_time(m_currentCam, &value);
    ui->shutter_slider->setValue(value);

    m_sendMessageToCamera = true;
}

void MainWindow::on_brightness_slider_valueChanged(int value)
{
    if (!m_currentCam)
        return;

    ui->brightnessValue_lab->setNum(value);
    if (m_sendMessageToCamera)
        slcam_set_exposure_compensation(m_currentCam, value);
}

void MainWindow::on_gain_slider_valueChanged(int value)
{
    if (!m_currentCam)
        return;

    ui->gainValue_lab->setNum(value);
    if (m_sendMessageToCamera)
        slcam_set_exposure_gain(m_currentCam, value);
}

void MainWindow::on_shutter_slider_valueChanged(int value)
{
    if (!m_currentCam)
        return;

    ui->shutterValue_lab->setNum(value);
    if (m_sendMessageToCamera)
        slcam_set_exposure_time(m_currentCam, value);
}

void MainWindow::on_snap_btn_clicked()
{
    emit getFrameForSnap();
}

void MainWindow::onGetSnapFrame(const VideoFrame &frame)
{
    QString picPath =
        QStandardPaths::writableLocation(QStandardPaths::PicturesLocation) +
        "/" +
        QDateTime::currentDateTime().toString(
            QStringLiteral("yyyyMMddhhmmsszzz"));
    QString suffix = ui->imageFormat_cBox->currentText();
    QString filter =
        tr("Images (*.bmp);;Images (*.jpg);;Images (*.png);;Images "
           "(*.tiff)");
    QString selectedFilter = tr("Images (*%1)").arg(suffix);

    QString path = QFileDialog::getSaveFileName(
        this, tr("Save"), picPath, filter, &selectedFilter);

    if (path.isEmpty())
        return;

    if (!path.endsWith(suffix))
        path += suffix;

    QImage img = videoFrame2QImg(frame);
    bool ret = img.save(path, nullptr, 100);
    qDebug() << "pic save result:" << ret << ", path: " << path;
}

void MainWindow::on_video_tabWidget_tabCloseRequested(int index)
{
    if (index == 0)
        on_closeVideo_btn_clicked();
    else
        ui->video_tabWidget->removeTab(index);
}

void MainWindow::on_closeVideo_btn_clicked()
{
    qDebug("Close camera");
    if (!m_currentCam)
        return;

    m_connectingDevices.clear();
    m_cameraOpend = false;

    if (ui->record_btn->isChecked())
        ui->record_btn->click();

    m_camCapThread->closeVideoThread();
    m_camCapThread->quit();
    if (!m_camCapThread->wait(timeOutToKillThreadMs))
    {
        qWarning("Camera close. CameraThread should be terminated.");
        m_camCapThread->terminate();
    }

    m_videoWidget->stopPlay();
    m_videoWidget.reset();
    ui->video_tabWidget->removeTab(0);
    m_camCapThread.reset();
    slcam_close(m_currentCam);
    m_currentCam = nullptr;
    setWidgetsEnbaleStatus(false);
}

void MainWindow::on_record_btn_toggled(bool checked)
{
    if (checked)
    {
        QString recordPath =
            QStandardPaths::writableLocation(QStandardPaths::MoviesLocation) +
            "/" +
            QDateTime::currentDateTime().toString(
                QStringLiteral("yyyyMMddhhmmsszzz"));
        QString suffix = ui->recordFormat_cBox->currentText();
        QString filter =
            tr("Videos (*.mp4)");
        QString selectedFilter = tr("Videos (*%1)").arg(suffix);

        QString path = QFileDialog::getSaveFileName(
            this, tr("Save"), recordPath, filter, &selectedFilter);

        if (path.isEmpty())
            return;

        if (!path.endsWith(suffix))
            path += suffix;

        emit startRecording(path);
    }
    else
    {
        emit stopRecording();
    }
}
