#include "imgconverthelper.h"

#include <string.h>

#include <QDebug>

#include "alignedmalloc.h"
#include "i420buffer.h"

// Aligning pointer to 64 bytes for improved performance, e.g. use SIMD.
static const int kBufferAlignment = 64;

static bool i420ToRGB24(
    const unsigned char *src_y, int src_stride_y, const unsigned char *src_u,
    int src_stride_u, const unsigned char *src_v, int src_stride_v,
    unsigned char *dst_rgb24, int dst_stride_rgb24, int width, int height)
{
    if (!src_y || !src_u || !src_v || !dst_rgb24 || width <= 0 || height == 0)
        return false;

    int rgb[3];
    int yIdx, uIdx, vIdx, idx;
    for (int i = 0; i < height; i++)
    {
        for (int j = 0; j < width; j++)
        {
            yIdx = i * src_stride_y + j;
            vIdx = (i / 2) * src_stride_u + (j / 2);
            uIdx = vIdx;
            int y = src_y[yIdx] - 16;
            int u = src_u[vIdx] - 128;
            int v = src_v[vIdx] - 128;
            rgb[0] = static_cast<int>(1.164 * y + 1.596 * v);  // r分量
            rgb[1] =
                static_cast<int>(1.164 * y - 0.392 * u - 0.813 * v);  // g分量
            rgb[2] = static_cast<int>(1.164 * y + 2.017 * u);         // b分量

            for (int k = 0; k < 3; k++)
            {
                idx = (i * width + j) * 3 + k;
                if (rgb[k] >= 0 && rgb[k] <= 255)
                    dst_rgb24[idx] = rgb[k];
                else
                    dst_rgb24[idx] = (rgb[k] < 0) ? 0 : 255;
            }
        }
    }
    return true;
}

QImage videoFrame2QImg(const SLcamVideoFrame &frame)
{
    if (frame.width <= 0 || frame.height <= 0 || !frame.data[0])
        return QImage();

    switch (frame.fmt)
    {
        case SLcamPixFormat::SLCAM_PIX_FORMAT_ARGB:
            return QImage(
                       frame.data[0], frame.width, frame.height,
                       QImage::Format_ARGB32)
                .copy();
        case SLcamPixFormat::SLCAM_PIX_FORMAT_RGB24:
            return QImage(
                       frame.data[0], frame.width, frame.height,
                       QImage::Format_RGB888)
                .copy();
        case SLcamPixFormat::SLCAM_PIX_FORMAT_RGBA:
            return QImage(
                       frame.data[0], frame.width, frame.height,
                       QImage::Format_RGBA8888)
                .copy();
        case SLcamPixFormat::SLCAM_PIX_FORMAT_I420:
        {
            int stride = frame.width * 3;
            uint8_t *data = static_cast<uint8_t *>(
                alignedMalloc(stride * frame.height, kBufferAlignment));
            i420ToRGB24(
                frame.data[0], frame.linesize[0], frame.data[1],
                frame.linesize[1], frame.data[2], frame.linesize[2], data,
                stride, frame.width, frame.height);
            QImage rgbImg(
                data, frame.width, frame.height, stride, QImage::Format_RGB888,
                alignedFree, data);
            return rgbImg;
        }
        case SLcamPixFormat::SLCAM_PIX_FORMAT_BGRA:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_NV12:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_GRAY8:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_J420:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_BGR24:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_ABGR:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_UNKNOWN:
        default:
            break;
    }
    return QImage();
}

QImage videoFrame2QImg(const VideoFrame &frame)
{
    if (frame.size() == 0)
        return QImage();

    auto frameBuffer = frame.videoFrameBuffer()->toI420();
    if (!frameBuffer)
        return QImage();

    switch (frameBuffer->type())
    {
        case VideoFrameBuffer::Type::kI420:
        {
            int stride = frameBuffer->width() * 3;
            uint8_t *data = static_cast<uint8_t *>(alignedMalloc(
                stride * frameBuffer->height(), kBufferAlignment));
            i420ToRGB24(
                frameBuffer->dataY(), frameBuffer->strideY(),
                frameBuffer->dataU(), frameBuffer->strideU(),
                frameBuffer->dataV(), frameBuffer->strideV(), data, stride,
                frameBuffer->width(), frameBuffer->height());
            QImage rgbImg(
                data, frameBuffer->width(), frameBuffer->height(), stride,
                QImage::Format_RGB888, alignedFree, data);
            return rgbImg;
        }
        case VideoFrameBuffer::Type::kNative:
        case VideoFrameBuffer::Type::kI420A:
        case VideoFrameBuffer::Type::kI422:
        case VideoFrameBuffer::Type::kI444:
        case VideoFrameBuffer::Type::kI010:
        case VideoFrameBuffer::Type::kI210:
        case VideoFrameBuffer::Type::kNV12:
        default:
            break;
    }
    return QImage();
}

VideoFrame makeVideoFrame(const SLcamVideoFrame &frame)
{
    assert(frame.width > 0);
    assert(frame.height > 0);
    if (frame.width <= 0 || frame.height <= 0 || !frame.data[0])
        return VideoFrame();

    switch (frame.fmt)
    {
        case SLcamPixFormat::SLCAM_PIX_FORMAT_I420:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_J420:
        {
            std::shared_ptr<I420Buffer> buffer = I420Buffer::copy(
                frame.width, frame.height, frame.data[0], frame.linesize[0],
                frame.data[1], frame.linesize[1], frame.data[2],
                frame.linesize[2]);
            VideoFrame videoFrame(buffer);
            videoFrame.setTimestampUs(frame.pts);
            return videoFrame;
        }
        case SLcamPixFormat::SLCAM_PIX_FORMAT_RGB24:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_BGR24:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_ARGB:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_RGBA:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_BGRA:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_NV12:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_GRAY8:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_ABGR:
        case SLcamPixFormat::SLCAM_PIX_FORMAT_UNKNOWN:
        default:
        {
            qDebug() << "Unsupport frame format " << frame.fmt
                     << ", makeVideoFrame failed!";
            break;
        }
    }
    return VideoFrame();
}
