#include "videothread.h"

#include <QDebug>

#include "imgconverthelper.h"

VideoThread::VideoThread(HSLcam cam, QObject *parent)
    : QThread(parent),
      m_cam(cam),
      m_isStarting(false),
      m_needFrameForSnap(false),
      m_isRecording(false)
{
}

void VideoThread::openVideoThread()
{
    m_isStarting = true;
}

void VideoThread::closeVideoThread()
{
    m_isStarting = false;
}

void VideoThread::setCapctx(const SLcamCaptureContext &capCtx)
{
    m_capCtx = capCtx;
}

void VideoThread::run()
{
    if (!m_cam)
        return;

    SLcamVideoFrame slFrame;
    int ret = 0;
    ret = slcam_set_capture_context(m_cam, &m_capCtx);
    if (ret != SLCAMRET_SUCCESS)
    {
        qDebug("Open failed!");
        return;
    }

    int failedCount = 0;
    while (m_isStarting)
    {
        if (slcam_read_frame(m_cam, &slFrame) == 0)
        {
            if (m_isRecording)
                slcam_record_append_frame(m_cam, &slFrame);

            failedCount = 0;
            auto videoFrame = makeVideoFrame(slFrame);
            slcam_free_frame(&slFrame);
            emit sendFrameForDisplay(videoFrame);

            if (m_needFrameForSnap)
            {
                emit sendFrameForSnap(videoFrame);
                m_needFrameForSnap = false;
            }
        }
        else
        {
            if (++failedCount > 180)
            {
                qDebug() << "getVideoFrameData failed! FailedCount="
                         << failedCount;
                m_isStarting = false;
                emit closeCamera();
                break;
            }
        }
        QThread::msleep(10);
    }
}

void VideoThread::onGetFrameForSnap()
{
    m_needFrameForSnap = true;
}

void VideoThread::onStartRecording(const QString &path)
{
    if (!m_cam)
        return;

    m_recordFilePath = path;
    SLcamRecordSaveInfo info;
    std::string strPath = m_recordFilePath.toStdString();
    info.savePath = const_cast<char *>(strPath.c_str());
    info.width = m_capCtx.resolution.width;
    info.height = m_capCtx.resolution.height;
    info.fps = m_capCtx.fps;
    if (slcam_record_start(m_cam, info) != SLCAMRET_SUCCESS)
    {
        qWarning("Failed to start recording!");
        return;
    }
    m_isRecording = true;
}

void VideoThread::onStopRecording()
{
    if (!m_cam)
        return;

    if (slcam_record_stop(m_cam) != SLCAMRET_SUCCESS)
        qWarning("Failed to stop recording!");
    m_isRecording = false;
}
