﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using csharpdemo;

namespace csharpdemo1
{
    public partial class Form1 : Form
    {
        private int _mCurrentCam = 0;
        private SLcam.SLcamDevInfo[] _devInfos;
        private SLcam _slcam;
        private SLcam.SLcamVideoCaptureCapabilities _capabilities;
        private Task _task;
        private volatile bool _running = true;
        private SLcam.SLcamCaptureContext _captureContext;
        private SDLPanel sdlPanel;
        private bool _captureImage;
        private bool _recordStart;
        private DateTime _recordStartTime;  // 记录开始录制的时间

        public Form1()
        {
            InitializeComponent();
            SLcam.slcam_api_init();
        }

        private void scan_btn_Click(object sender, EventArgs e)
        {
            // get device nums
            _devInfos = SLcam.EnumerateDevices();
            // 设置日志等级
            SLcam.LogSetLevel(SLcam.SLcamLogLevel.SLCAM_LOG_INFO);
            // 日志回调函数 
            // RegisterLogCallback();
            if (_devInfos == null || _devInfos.Length <= 0) return;

            devList_cBox.Items.Clear();
            for (var i = 0; i < _devInfos.Length; i++)
                devList_cBox.Items.Add(_devInfos[i].name + i);

            devList_cBox.SelectedIndex = _mCurrentCam;
        }

        private void open_btn_Click(object sender, EventArgs e)
        {
            _slcam?.Dispose();
            sdlPanel?.Dispose(true);
            sdlPanel = new SDLPanel();
            sdlPanel.Anchor = AnchorStyles.Top | AnchorStyles.Bottom | AnchorStyles.Left | AnchorStyles.Right;


            _slcam = SLcam.Open(_devInfos[_mCurrentCam].uniqueName);
            // U408升级示例
            // string filePath = "C:\\Users\\14228\\Desktop\\templates\\DS-2K867(4K)V2001_240621.img";
            // _slcam.Upgrade(filePath, UpgradeProgressCallback, IntPtr.Zero);

            if (_slcam?.Handle == null)
            {
                MessageBox.Show("设备打开异常");
                return;
            }

            _slcam.GetCaptureCapabilities(out _capabilities);
            resolu_cBox.Items.Clear();
            for (var i = 0; i < _capabilities.nbCaps; i++)
            {
                var videoCap = _capabilities.videoCaps[i];
                if (videoCap.videoFmt == SLcam.SLcamVideoFormat.SLCAM_VIDEO_FORMAT_NV12)
                    resolu_cBox.Items.Add(videoCap.resolution.width + " * " +
                                          videoCap.resolution.height);
            }

            _captureContext = new SLcam.SLcamCaptureContext();
            // _captureContext.resolution = resolution;
            _captureContext.uniqueName = _devInfos[_mCurrentCam].uniqueName;
            // NV12 or MJPEG
            _captureContext.capFmt = SLcam.SLcamVideoFormat.SLCAM_VIDEO_FORMAT_NV12;
            _captureContext.readFmt = SLcam.SLcamPixFormat.SLCAM_PIX_FORMAT_I420;
            _captureContext.fps = _capabilities.videoCaps[0].maxFps;
            resolu_cBox.SelectedIndex = 0;
            _running = true;
            device_info();
            _task = Task.Run(() => StartCameraCapture(_devInfos, _mCurrentCam));
        }

        private void device_info()
        {
            if (!_slcam.GetFocusRange(out int minValue, out int maxValue, out int defValue, out int stepValue))
            {
                autoFocus_checkBox.Enabled = false;
                minValue = maxValue = defValue = stepValue = 0;
            }
            else
            {
                if (_slcam.GetFocusMode(out int focusMode))
                {
                    if (focusMode == (int)SLcam.SLcamFocusMode.SLCAM_FOCUS_MODE_AUTO)
                    {
                        autoFocus_checkBox.Checked = true;
                    }
                    else if (focusMode == (int)SLcam.SLcamFocusMode.SLCAM_FOCUS_MODE_MANUAL)
                    {
                        autoFocus_checkBox.Checked = false;
                    }
                }
            }


            if (_slcam.GetWhiteBalanceMode(out int mode))
            {
                if (mode == (int)SLcam.SLcamWhiteBalanceMode.SLCAM_WHITE_BALANCE_MODE_MANUAL)
                {
                    autoWB_checkBoz.Checked = false;
                    _slcam.GetWhiteBalanceComponentRedRange(out minValue, out maxValue, out defValue, out stepValue);
                    red_tBar.SetRange(minValue, maxValue);
                    _slcam.GetWhiteBalanceComponentRed(out int value);
                    red_tBar.Value = value;
                    _slcam.GetWhiteBalanceComponentGreenRange(out minValue, out maxValue, out defValue, out stepValue);
                    green_tBar.SetRange(minValue, maxValue);
                    _slcam.GetWhiteBalanceComponentGreen(out value);
                    green_tBar.Value = value;
                    _slcam.GetWhiteBalanceComponentBlueRange(out minValue, out maxValue, out defValue, out stepValue);
                    blue_tBar.SetRange(minValue, maxValue);
                    _slcam.GetWhiteBalanceComponentBlue(out value);
                    blue_tBar.Value = value;
                }
                else
                {
                    autoWB_checkBoz.Checked = true;
                }
            }

            if (_slcam.GetExposureMode(out mode))
            {
                if (mode == (int)SLcam.SLcamExposureMode.SLCAM_EXPOSURE_MODE_MANUAL)
                    autoExp_checkBox.Checked = false;

                else
                    autoExp_checkBox.Checked = true;
            }
        }

        private void StartCameraCapture(SLcam.SLcamDevInfo[] devInfos, int mCurrentCam)
        {
            var stopwatch = new Stopwatch();
            int frameCount = 0;
            stopwatch.Start();

            if (!_slcam.SetCaptureContext(ref _captureContext))
            {
                MessageBox.Show("Initialization Failed");
                return;
            }

            // 设置完参数后  需要等待一段时间后取流（否则可能会取流失败问题）
            // System.Threading.Thread.Sleep(500);

            var vframe = new SLcam.SLcamVideoFrame();
            while (_running)
            {
                if (!_slcam.ReadFrame(ref vframe)) continue;

                // 检查帧尺寸是否与期望的纹理尺寸匹配
                if (vframe.width != sdlPanel.VideoWidth || vframe.height != sdlPanel.VideoHeight)
                {
                    // 可以在这里调整vframe的数据尺寸，或者选择跳过该帧
                    continue;
                }

                frameCount++;

                if (stopwatch.ElapsedMilliseconds >= 3000)
                {
                    var fps = (double)(frameCount / (stopwatch.ElapsedMilliseconds / 1000.0));
                    Invoke((MethodInvoker)(() => { fps_lab.Text = fps.ToString("0.00"); }));

                    // 重置计时器和计数器
                    stopwatch.Restart();
                    frameCount = 0;
                }

                try
                {
                    if (_captureImage)
                    {
                        _captureImage = false;

                        // // Convert YUV data to Bitmap
                        // Bitmap capturedImage = YuvToBitmap(vframe.data, vframe.linesize, vframe.width, vframe.height);
                        //
                        // // Save to desktop or prompt for location
                        // SaveCapturedImage(capturedImage);
                        
                        // SaveCapturedImage2 为使用sdk采图
                        SaveCapturedImage2(vframe);
                    }

                    if (_recordStart)
                    {
                        // 录制中
                        _slcam.AppendFrame(vframe);
                    }

                    this?.Invoke((MethodInvoker)(() =>
                    {
                        sdlPanel.UpdateYUVTexture(vframe.data[0], vframe.data[1], vframe.data[2],
                            vframe.linesize[0], vframe.linesize[1], vframe.linesize[2], vframe.width, vframe.height);
                    }));
                }
                catch (ObjectDisposedException)
                {
                    break;
                }

                System.Threading.Thread.Sleep(10);

                _slcam.FreeFrame(ref vframe);
            }
        }

        private void SaveCapturedImage2(in SLcam.SLcamVideoFrame vframe)
        {
            // 打开一个保存文件对话框，让用户选择保存的路径
            SaveFileDialog saveFileDialog = new SaveFileDialog();
            saveFileDialog.Filter = "All Files|*.*";
            saveFileDialog.Title = "Save Captured Image";
            saveFileDialog.FileName = "capture"; // 默认文件名
            saveFileDialog.DefaultExt = ""; // 默认扩展名

            // 显示提示框并检查用户是否选择了路径
            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                string selectedPath = saveFileDialog.FileName;

                // 创建并设置 SLcamFileSaveInfo 结构体
                var sLcamFileSaveInfo = new SLcam.SLcamFileSaveInfo();
                sLcamFileSaveInfo.format = SLcam.SLcamImgFormat.SLCAM_IMG_FORMAT_JPG; // 假设固定为JPG
                sLcamFileSaveInfo.savePath = selectedPath; // 将用户选择的路径赋值给 save_path

                // Convert vframe to IntPtr
                IntPtr vframePtr = IntPtr.Zero;

                try
                {

                    vframePtr = Marshal.AllocHGlobal(Marshal.SizeOf(vframe));
                    Marshal.StructureToPtr(vframe, vframePtr, false);

 
                    sLcamFileSaveInfo.frame = vframePtr;

                    _slcam.SaveImage(sLcamFileSaveInfo);


                    IntPtr saveInfoPtr = Marshal.AllocHGlobal(Marshal.SizeOf(sLcamFileSaveInfo));
                    Marshal.StructureToPtr(sLcamFileSaveInfo, saveInfoPtr, false);

                    Marshal.FreeHGlobal(saveInfoPtr);
                }
                finally
                {
                    // Free the memory allocated for vframe after usage
                    if (vframePtr != IntPtr.Zero)
                    {
                        Marshal.FreeHGlobal(vframePtr);
                    }
                }
            }
        }

        private void AdjustSDLPanelSize()
        {
            if (sdlPanel == null || panel1.Width == 0 || panel1.Height == 0) return;

            // 原始SDLPanel的宽高比
            float aspectRatio = (float)sdlPanel.VideoWidth / sdlPanel.VideoHeight;

            int panelWidth = panel1.Width;
            int panelHeight = panel1.Height;

            int sdlPanelWidth, sdlPanelHeight;

            // 根据宽高比和父容器的大小计算新的大小，保持比例
            if (panelWidth / (float)panelHeight > aspectRatio)
            {
                // 限制在高度上
                sdlPanelHeight = panelHeight;
                sdlPanelWidth = (int)(sdlPanelHeight * aspectRatio);
            }
            else
            {
                // 限制在宽度上
                sdlPanelWidth = panelWidth;
                sdlPanelHeight = (int)(sdlPanelWidth / aspectRatio);
            }

            // 设置SDLPanel的新大小
            sdlPanel.Size = new Size(sdlPanelWidth, sdlPanelHeight);

            // 调整SDLPanel位置使其居中
            sdlPanel.Location = new Point((panel1.Width - sdlPanel.Width) / 2, (panel1.Height - sdlPanel.Height) / 2);
        }

        private void close_btn_Click(object sender, EventArgs e)
        {
            _running = false;
            _task = null;
            if (sdlPanel != null)
            {
                this.panel1.Controls.Remove(sdlPanel);
                sdlPanel?.Dispose(true);
                sdlPanel = null;
            }

            if (_slcam != null)
            {
                _slcam.Close();
                _slcam = null;
            }
        }

        private void devList_cBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            _mCurrentCam = devList_cBox.SelectedIndex;
        }

        private SLcam.SLcamVideoResolution getResolution(string resolotionStr)
        {
            var strings = resolotionStr.Split('*');
            var resolution = new SLcam.SLcamVideoResolution();
            resolution.width = int.Parse(strings[0].Trim());
            resolution.height = int.Parse(strings[1].Trim());
            return resolution;
        }

        private void resolu_cBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            _captureContext.resolution = getResolution(resolu_cBox.SelectedItem.ToString());
            _slcam.SetCaptureContext(ref _captureContext);
            // 接下来设置它的初始大小和位置
            int newWidth = ClientRectangle.Right - scan_btn.Bounds.Right - 20;
            int newHeight = ClientRectangle.Height - 8;
            panel1.Width = newWidth;
            panel1.Height = newHeight;
            sdlPanel.Size = new Size(_captureContext.resolution.width, _captureContext.resolution.height);
            sdlPanel.Location = new Point(10, 10);
            // 将SDLPanel添加到MainForm的控件集合中
            this.panel1.Controls.Add(sdlPanel);
            sdlPanel.AdjustTextureSize(_captureContext.resolution.width, _captureContext.resolution.height);
            AdjustSDLPanelSize();
        }

        private void autoExp_checkBox_CheckedChanged(object sender, EventArgs e)
        {
            if (autoExp_checkBox.Checked)
                _slcam.SetExposureMode(SLcam.SLcamExposureMode.SLCAM_EXPOSURE_MODE_AUTO);
            else
                _slcam.SetExposureMode(SLcam.SLcamExposureMode.SLCAM_EXPOSURE_MODE_MANUAL);
        }

        private void autoWB_checkBoz_CheckedChanged(object sender, EventArgs e)
        {
            if (autoWB_checkBoz.Checked)
                _slcam.SetWhiteBalanceMode(SLcam.SLcamWhiteBalanceMode.SLCAM_WHITE_BALANCE_MODE_AUTO);
            else
            {
                _slcam.SetWhiteBalanceMode(SLcam.SLcamWhiteBalanceMode.SLCAM_WHITE_BALANCE_MODE_MANUAL);
                _slcam.GetWhiteBalanceComponentRed(out int value);
                red_tBar.Value = value;
                _slcam.GetWhiteBalanceComponentGreen(out value);
                green_tBar.Value = value;
                _slcam.GetWhiteBalanceComponentBlue(out value);
                blue_tBar.Value = value;
            }
        }

        private void autoFocus_checkBox_CheckedChanged(object sender, EventArgs e)
        {
            if (autoFocus_checkBox.Checked)
                _slcam.SetFocusMode(SLcam.SLcamFocusMode.SLCAM_FOCUS_MODE_AUTO);
            else
            {
                _slcam.SetFocusMode(SLcam.SLcamFocusMode.SLCAM_FOCUS_MODE_MANUAL);
            }
        }

        private void red_tBar_ValueChanged(object sender, EventArgs e)
        {
            _slcam.SetWhiteBalanceComponentRed(red_tBar.Value);
        }

        private void blue_tBar_ValueChanged(object sender, EventArgs e)
        {
            _slcam.SetWhiteBalanceComponentBlue(blue_tBar.Value);
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            _running = false;
            _task = null;
            if (sdlPanel != null)
            {
                sdlPanel?.Dispose(true);
                this.panel1.Controls.Remove(sdlPanel);
                sdlPanel = null;
            }

            if (_slcam != null)
            {
                _slcam.Close();
                _slcam = null;
            }

            SLcam.ApiDestroy();
        }

        private void Form1_SizeChanged(object sender, EventArgs e)
        {
            int newWidth = ClientRectangle.Right - scan_btn.Bounds.Right - 20;
            int newHeight = ClientRectangle.Height - 8;

            panel1.Width = newWidth;
            panel1.Height = newHeight;
            AdjustSDLPanelSize();
        }

        private void green_tBar_ValueChanged(object sender, EventArgs e)
        {
            _slcam.SetWhiteBalanceComponentGreen(green_tBar.Value);
        }

        public static void MyLogCallbackFunction(int level, string msg)
        {
            Console.WriteLine($"Log Level: {level}, Message: {msg}");
        }

        public void RegisterLogCallback()
        {
            var callbackDelegate = new SLcam.SLcamLogCallback(MyLogCallbackFunction);
            SLcam.slcam_log_set_callback(callbackDelegate);
        }

        private static void UpgradeProgressCallback(int percent, IntPtr ctx)
        {
            // Handle the progress update
            Console.WriteLine($"Upgrade progress: {percent}%");
            if (percent == 100)
            {
                Console.WriteLine($"Upgrade progress success");
            }
        }

        private void saveImg_btn_Click(object sender, EventArgs e)
        {
            _captureImage = true;
        }


        public Bitmap YuvToBitmap(IntPtr[] yuvData, int[] linesize, int width, int height)
        {
            // 创建一个Bitmap对象
            Bitmap bmp = new Bitmap(width, height, PixelFormat.Format24bppRgb);

            // 锁定Bitmap的位图数据
            BitmapData bmpData = bmp.LockBits(new Rectangle(0, 0, width, height),
                ImageLockMode.WriteOnly,
                bmp.PixelFormat);

            // 获取Bitmap数据的指针
            IntPtr ptr = bmpData.Scan0;

            // 创建一个临时RGB数据数组
            int bytes = Math.Abs(bmpData.Stride) * height;
            byte[] rgbValues = new byte[bytes];

            // 转换YUV数据到RGB数据
            DecodeYUV420(rgbValues, yuvData, linesize, width, height);

            // 将RGB数据复制到Bitmap数据
            Marshal.Copy(rgbValues, 0, ptr, bytes);

            // 解锁Bitmap数据
            bmp.UnlockBits(bmpData);

            return bmp;
        }

        private static void DecodeYUV420(byte[] rgbData, IntPtr[] yuvData, int[] linesize, int width, int height)
        {
            // 将IntPtr转换为byte数组
            byte[] yPlane = new byte[linesize[0] * height];
            byte[] uPlane = new byte[linesize[1] * (height / 2)];
            byte[] vPlane = new byte[linesize[2] * (height / 2)];

            Marshal.Copy(yuvData[0], yPlane, 0, yPlane.Length);
            Marshal.Copy(yuvData[1], uPlane, 0, uPlane.Length);
            Marshal.Copy(yuvData[2], vPlane, 0, vPlane.Length);

            int frameSize = width * height;

            for (int j = 0, yp = 0; j < height; j++)
            {
                int yIndex = j * linesize[0];
                int uvIndex = (j >> 1) * linesize[1];
                for (int i = 0; i < width; i++, yp++)
                {
                    int y = yPlane[yIndex + i] & 0xFF;
                    int u = uPlane[uvIndex + (i >> 1)] & 0xFF;
                    int v = vPlane[uvIndex + (i >> 1)] & 0xFF;

                    y = y < 16 ? 16 : y;

                    int r = (int)(1.164f * (y - 16) + 1.596f * (v - 128));
                    int g = (int)(1.164f * (y - 16) - 0.813f * (v - 128) - 0.391f * (u - 128));
                    int b = (int)(1.164f * (y - 16) + 2.018f * (u - 128));

                    r = r < 0 ? 0 : (r > 255 ? 255 : r);
                    g = g < 0 ? 0 : (g > 255 ? 255 : g);
                    b = b < 0 ? 0 : (b > 255 ? 255 : b);

                    int index = yp * 3;
                    rgbData[index] = (byte)b;
                    rgbData[index + 1] = (byte)g;
                    rgbData[index + 2] = (byte)r;
                }
            }
        }

        private void SaveCapturedImage(Bitmap image)
        {
            SaveFileDialog saveFileDialog = new SaveFileDialog
            {
                Filter = "PNG Image|*.png",
                Title = "Save Captured Image"
            };

            if (saveFileDialog.ShowDialog() == DialogResult.OK && saveFileDialog.FileName != "")
            {
                try
                {
                    image.Save(saveFileDialog.FileName, ImageFormat.Png);
                    MessageBox.Show(this, $"Image saved to {saveFileDialog.FileName}");
                }
                catch (Exception ex)
                {
                    MessageBox.Show($"Failed to save image: {ex.Message}");
                }
            }

            // image.Dispose();
        }

        private void videoStart_btn_Click(object sender, EventArgs e)
        {
            // 创建并显示保存文件对话框，让用户选择保存视频的路径
            SaveFileDialog saveFileDialog = new SaveFileDialog();
            saveFileDialog.Filter = "MP4 Video|*.mp4|All Files|*.*";
            saveFileDialog.Title = "Save Video Recording";
            saveFileDialog.FileName = "video.mp4";  // 默认文件名
            saveFileDialog.DefaultExt = ".mp4";     // 默认扩展名

            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                string selectedPath = saveFileDialog.FileName;

                // 创建并设置 SLcamRecordSaveInfo 结构体
                var sLcamRecordSaveInfo = new SLcam.SLcamRecordSaveInfo
                {
                    savePath = selectedPath,
                    width = _captureContext.resolution.width,               
                    height = _captureContext.resolution.height,             
                    fps = _captureContext.fps                    
                };
                
                _slcam.StartRecording(sLcamRecordSaveInfo);
                
                _recordStart = true;
                _recordStartTime = DateTime.Now;
            }
        }


        private void videoStop_btn_Click(object sender, EventArgs e)
        {
            // 切分辨率和断连时会中止录像
            _recordStart = false;
            _slcam?.StopRecording();
            // 计算录制持续时间
            TimeSpan recordDuration = DateTime.Now - _recordStartTime;
            double totalSeconds = recordDuration.TotalSeconds;

            // 输出录制的总秒数
            MessageBox.Show($"录制时间：{totalSeconds} 秒");
        }


        // 设置Roi相关
        public void setRoi()
        {
            /*
             * x，y需为偶数，宽高需要为8的倍数 （YUV）
             * 裁剪后需要同步更改panel控件的大小，否则会造成画面卡死
             */
            // _slcam.SetRoiRegion(1, 8, 8, 1080, 1080);
            // int newWidth = ClientRectangle.Right - scan_btn.Bounds.Right - 20;
            // int newHeight = ClientRectangle.Height - 8;
            // panel1.Width = newWidth;
            // panel1.Height = newHeight;
            // sdlPanel.Size = new Size(1080, 1080);
            // sdlPanel.Location = new Point(10, 10);
            // // 将SDLPanel添加到MainForm的控件集合中
            // this.panel1.Controls.Add(sdlPanel);
            // sdlPanel.AdjustTextureSize(1080, 1080);
            // AdjustSDLPanelSize();
            
            // 复原，将enable设置为0（画面恢复为未裁剪之前，使用GetRoiRegion可获取之前设置过的大小 ）
            // _slcam.SetRoiRegion(0, 3, 5, 1080, 1080);
            // int newWidth = ClientRectangle.Right - scan_btn.Bounds.Right - 20;
            // int newHeight = ClientRectangle.Height - 8;
            // panel1.Width = newWidth;
            // panel1.Height = newHeight;
            // sdlPanel.Size = new Size(_captureContext.resolution.width, _captureContext.resolution.height);
            // sdlPanel.Location = new Point(10, 10);
            // // 将SDLPanel添加到MainForm的控件集合中
            // this.panel1.Controls.Add(sdlPanel);
            // sdlPanel.AdjustTextureSize(_captureContext.resolution.width, _captureContext.resolution.height);
            // AdjustSDLPanelSize();
        }
    }
}