﻿using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace csharpdemo2
{
    public partial class Form1 : Form
    {
        private int _mCurrentCam = 0;
        private SLcam.SLcamDevInfo[] _devInfos;
        private SLcam _slcam;
        private SLcam.SLcamVideoCaptureCapabilities _capabilities;
        private Task _task;
        private volatile bool _running = true;
        private SLcam.SLcamCaptureContext _captureContext;
        private bool _captureImage;

        public Form1()
        {
            InitializeComponent();
            SLcam.slcam_api_init();
            pictureBox1.Width = ClientRectangle.Right - scan_btn.Bounds.Right - 20;
            pictureBox1.Height = ClientRectangle.Height - 8;
        }

        private void scan_btn_Click(object sender, EventArgs e)
        {
            // get device nums
            _devInfos = SLcam.EnumerateDevices();
            // 设置日志等级
            SLcam.LogSetLevel(SLcam.SLcamLogLevel.SLCAM_LOG_TRACE);
            if (_devInfos == null || _devInfos.Length <= 0) return;

            devList_cBox.Items.Clear();
            for (var i = 0; i < _devInfos.Length; i++)
                devList_cBox.Items.Add(_devInfos[i].name + i);

            devList_cBox.SelectedIndex = _mCurrentCam;
        }

        private void open_btn_Click(object sender, EventArgs e)
        {
            _slcam?.Dispose();
            _slcam = SLcam.Open(_devInfos[_mCurrentCam].uniqueName);
            if (_slcam?.Handle == null)
            {
                MessageBox.Show("设备打开异常");
                return;
            }

            _slcam.GetCaptureCapabilities(out _capabilities);
            SLcam.LogSetLevel(SLcam.SLcamLogLevel.SLCAM_LOG_TRACE);
            // _slcam.RegisterLogCallback();
            resolu_cBox.Items.Clear();
            for (var i = 0; i < _capabilities.nbCaps; i++)
            {
                var videoCap = _capabilities.videoCaps[i];
                if (videoCap.videoFmt == SLcam.SLcamVideoFormat.SLCAM_VIDEO_FORMAT_MJPEG)
                    resolu_cBox.Items.Add(videoCap.resolution.width + " * " +
                                          videoCap.resolution.height);
            }

            _captureContext = new SLcam.SLcamCaptureContext();
            // _captureContext.resolution = resolution;
            _captureContext.uniqueName = _devInfos[_mCurrentCam].uniqueName;
            _captureContext.capFmt = SLcam.SLcamVideoFormat.SLCAM_VIDEO_FORMAT_MJPEG;
            _captureContext.readFmt = SLcam.SLcamPixFormat.SLCAM_PIX_FORMAT_BGR24;
            _captureContext.fps = _capabilities.videoCaps[0].maxFps;
            resolu_cBox.SelectedIndex = 0;
            _running = true;
            device_info();
            _task = Task.Run(() => StartCameraCapture(_devInfos, _mCurrentCam));
        }

        private void device_info()
        {
            if (!_slcam.GetFocusRange(out int minValue, out int maxValue, out int defValue, out int stepValue))
            {
                autoFocus_checkBox.Enabled = false;
                minValue = maxValue = defValue = stepValue = 0;
            }
            else
            {
                if (_slcam.GetFocusMode(out int focusMode))
                {
                    if (focusMode == (int)SLcam.SLcamFocusMode.SLCAM_FOCUS_MODE_AUTO)
                    {
                        autoFocus_checkBox.Checked = true;
                    }
                    else if (focusMode == (int)SLcam.SLcamFocusMode.SLCAM_FOCUS_MODE_MANUAL)
                    {
                        autoFocus_checkBox.Checked = false;
                    }
                }
            }


            if (_slcam.GetWhiteBalanceMode(out int mode))
            {
                if (mode == (int)SLcam.SLcamWhiteBalanceMode.SLCAM_WHITE_BALANCE_MODE_MANUAL)
                {
                    autoWB_checkBoz.Checked = false;
                    _slcam.GetWhiteBalanceComponentRedRange(out minValue, out maxValue, out defValue, out stepValue);
                    red_tBar.SetRange(minValue, maxValue);
                    _slcam.GetWhiteBalanceComponentRed(out int value);
                    red_tBar.Value = value;
                    _slcam.GetWhiteBalanceComponentGreenRange(out minValue, out maxValue, out defValue, out stepValue);
                    green_tBar.SetRange(minValue, maxValue);
                    _slcam.GetWhiteBalanceComponentGreen(out value);
                    green_tBar.Value = value;
                    _slcam.GetWhiteBalanceComponentBlueRange(out minValue, out maxValue, out defValue, out stepValue);
                    blue_tBar.SetRange(minValue, maxValue);
                    _slcam.GetWhiteBalanceComponentBlue(out value);
                    blue_tBar.Value = value;
                }
                else
                {
                    autoWB_checkBoz.Checked = true;
                }
            }

            if (_slcam.GetExposureMode(out mode))
            {
                if (mode == (int)SLcam.SLcamExposureMode.SLCAM_EXPOSURE_MODE_MANUAL)
                    autoExp_checkBox.Checked = false;

                else
                    autoExp_checkBox.Checked = true;
            }
        }

        private void StartCameraCapture(SLcam.SLcamDevInfo[] devInfos, int mCurrentCam)
        {
            if (!_slcam.SetCaptureContext(ref _captureContext))
            {
                MessageBox.Show("Initialization Failed");
                return;
            }

            var stopwatch = new Stopwatch();
            int frameCount = 0;
            stopwatch.Start();

            // 设置完参数后  需要等待一段时间后取流（否则可能会取流失败问题）
            System.Threading.Thread.Sleep(500);

            Bitmap preBitmap = null;
            var vframe = new SLcam.SLcamVideoFrame();
            while (_running)
            {
                if (!_slcam.ReadFrame(ref vframe)) continue;
                frameCount++; // 此变量跟踪处理的帧数

                if (stopwatch.ElapsedMilliseconds >= 3000) // 每3秒更新一次FPS
                {
                    var fps = (double)(frameCount / (stopwatch.ElapsedMilliseconds / 1000.0));
                    // 更新UI
                    Invoke((MethodInvoker)(() => { fps_lab.Text = $"FPS: {fps.ToString("0.00")}"; }));
                    // 重置计时器和计数器
                    stopwatch.Restart();
                    frameCount = 0;
                }

                var disp = new Bitmap(vframe.width, vframe.height, vframe.linesize[0], PixelFormat.Format24bppRgb,
                    vframe.data[0]);
                // save to img
                if (_captureImage)
                {
                    _captureImage = false;
                    // Save to desktop or prompt for location
                    SaveCapturedImage(disp);
                }

                try
                {
                    pictureBox1.Invoke(new MethodInvoker(() =>
                    {
                        preBitmap?.Dispose();
                        var bitmap = new Bitmap(disp);
                        pictureBox1.Image = bitmap;
                        pictureBox1.Invalidate();
                        preBitmap = bitmap;
                    }));
                    _slcam.FreeFrame(ref vframe);
                    disp?.Dispose();
                    // System.Threading.Thread.Sleep(10);
                }
                catch (ObjectDisposedException)
                {
                    break;
                }
            }

            pictureBox1.Image = null;
        }

        private void close_btn_Click(object sender, EventArgs e)
        {
            _running = false;
            _task = null;
            _slcam?.Close();
            _slcam = null;
            pictureBox1.Image = null;
        }

        private void devList_cBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            _mCurrentCam = devList_cBox.SelectedIndex;
        }

        private SLcam.SLcamVideoResolution getResolution(string resolotionStr)
        {
            var strings = resolotionStr.Split('*');
            var resolution = new SLcam.SLcamVideoResolution();
            resolution.width = int.Parse(strings[0].Trim());
            resolution.height = int.Parse(strings[1].Trim());
            return resolution;
        }

        private void resolu_cBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            _captureContext.resolution = getResolution(resolu_cBox.SelectedItem.ToString());
            _slcam.SetCaptureContext(ref _captureContext);
        }

        private void autoExp_checkBox_CheckedChanged(object sender, EventArgs e)
        {
            if (autoExp_checkBox.Checked)
                _slcam.SetExposureMode(SLcam.SLcamExposureMode.SLCAM_EXPOSURE_MODE_AUTO);
            else
                _slcam.SetExposureMode(SLcam.SLcamExposureMode.SLCAM_EXPOSURE_MODE_MANUAL);
            // throw new System.NotImplementedException();
        }

        private void autoWB_checkBoz_CheckedChanged(object sender, EventArgs e)
        {
            if (autoWB_checkBoz.Checked)
                _slcam.SetWhiteBalanceMode(SLcam.SLcamWhiteBalanceMode.SLCAM_WHITE_BALANCE_MODE_AUTO);
            else
            {
                _slcam.SetWhiteBalanceMode(SLcam.SLcamWhiteBalanceMode.SLCAM_WHITE_BALANCE_MODE_MANUAL);
                _slcam.GetWhiteBalanceComponentRed(out int value);
                red_tBar.Value = value;
                _slcam.GetWhiteBalanceComponentGreen(out value);
                green_tBar.Value = value;
                _slcam.GetWhiteBalanceComponentBlue(out value);
                blue_tBar.Value = value;
            }

            // throw new System.NotImplementedException();
        }

        private void autoFocus_checkBox_CheckedChanged(object sender, EventArgs e)
        {
            if (autoFocus_checkBox.Checked)
                _slcam.SetFocusMode(SLcam.SLcamFocusMode.SLCAM_FOCUS_MODE_AUTO);
            else
            {
                _slcam.SetFocusMode(SLcam.SLcamFocusMode.SLCAM_FOCUS_MODE_MANUAL);
            }
        }

        private void red_tBar_ValueChanged(object sender, EventArgs e)
        {
            _slcam.SetWhiteBalanceComponentRed(red_tBar.Value);
        }

        private void blue_tBar_ValueChanged(object sender, EventArgs e)
        {
            _slcam.SetWhiteBalanceComponentBlue(blue_tBar.Value);
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            _running = false;
            _task = null;
            if (_slcam != null)
            {
                _slcam.Close();
                _slcam = null;
            }

            pictureBox1.Image = null;
            SLcam.ApiDestroy();
        }

        private void Form1_SizeChanged(object sender, EventArgs e)
        {
            int newWidth = ClientRectangle.Right - scan_btn.Bounds.Right - 20;
            int newHeight = ClientRectangle.Height - 8;

            pictureBox1.Width = newWidth;
            pictureBox1.Height = newHeight;
        }

        private void green_tBar_ValueChanged(object sender, EventArgs e)
        {
            _slcam.SetWhiteBalanceComponentGreen(green_tBar.Value);
        }

        private void saveImg_btn_Click(object sender, EventArgs e)
        {
            _captureImage = true;
        }

        private void SaveCapturedImage(Bitmap image)
        {
            SaveFileDialog saveFileDialog = new SaveFileDialog
            {
                Filter = "PNG Image|*.png",
                Title = "Save Captured Image"
            };

            if (saveFileDialog.ShowDialog() == DialogResult.OK && saveFileDialog.FileName != "")
            {
                try
                {
                    image.Save(saveFileDialog.FileName, ImageFormat.Png);
                    MessageBox.Show(this,$"Image saved to {saveFileDialog.FileName}");
                }
                catch (Exception ex)
                {
                    MessageBox.Show($"Failed to save image: {ex.Message}");
                }
                
            }

            // image.Dispose();
        }
        
        // 设置Roi相关
        public void setRoi()
        {
            /*
             * x，y需为偶数，宽高需要为8的倍数 （YUV）
             * 裁剪后需要同步更改panel控件的大小，否则会造成画面卡死
             */
            // _slcam.SetRoiRegion(1, 8, 8, 1080, 1080);
            // int newWidth = ClientRectangle.Right - scan_btn.Bounds.Right - 20;
            // int newHeight = ClientRectangle.Height - 8;
            // panel1.Width = newWidth;
            // panel1.Height = newHeight;
            // sdlPanel.Size = new Size(1080, 1080);
            // sdlPanel.Location = new Point(10, 10);
            // // 将SDLPanel添加到MainForm的控件集合中
            // this.panel1.Controls.Add(sdlPanel);
            // sdlPanel.AdjustTextureSize(1080, 1080);
            // AdjustSDLPanelSize();

            // 复原，将enable设置为0（画面恢复为未裁剪之前，使用GetRoiRegion可获取之前设置过的大小 ）
            // _slcam.SetRoiRegion(0, 3, 5, 1080, 1080);
            // int newWidth = ClientRectangle.Right - scan_btn.Bounds.Right - 20;
            // int newHeight = ClientRectangle.Height - 8;
            // panel1.Width = newWidth;
            // panel1.Height = newHeight;
            // sdlPanel.Size = new Size(_captureContext.resolution.width, _captureContext.resolution.height);
            // sdlPanel.Location = new Point(10, 10);
            // // 将SDLPanel添加到MainForm的控件集合中
            // this.panel1.Controls.Add(sdlPanel);
            // sdlPanel.AdjustTextureSize(_captureContext.resolution.width, _captureContext.resolution.height);
            // AdjustSDLPanelSize();
        }
    }
}