﻿using Microsoft.Win32.SafeHandles;
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.ConstrainedExecution;
using System.Runtime.InteropServices;
using System.Security.Permissions;
using System.Text;
using System.Threading.Tasks;

namespace slcam_labview
{
    public class SLcam
    {
        private const string SLCamDll = "slcam.dll";
        private const CallingConvention cc = CallingConvention.Cdecl;
        private string _unique_name;
        private SLcamCaptureContext _context;
        private const int SLCAM_MAX_DEVICES = 16;
        private const int SLCAM_MAX_CAP_SIZE = 32;


        public enum SLCamResultCode : uint
        {
            SLCAMRET_SUCCESS = 0x00000000,
            SLCAMRET_FAILURE = 0x80000000,
        }

        public enum SLcamModel : ushort
        {
            SLCAM_MODEL_M112U = 0x0112,
            SLCAM_MODEL_M114U = 0x0114,
            SLCAM_MODEL_M202U = 0x0202,
            SLCAM_MODEL_C310 = 0xC310,
            SLCAM_MODEL_C311 = 0xC311,
            SLCAM_MODEL_C312 = 0xC312,
            SLCAM_MODEL_C313 = 0xC313,
            SLCAM_MODEL_C314 = 0xC314,
            SLCAM_MODEL_U106 = 0x1106,
            SLCAM_MODEL_U108 = 0x1108,
            SLCAM_MODEL_U112 = 0x1112,
            SLCAM_MODEL_U120 = 0x1120,
            SLCAM_MODEL_A311U = 0xA311,
            SLCAM_MODEL_A312U = 0xA312,
            SLCAM_MODEL_A313 = 0xA313,
            SLCAM_MODEL_A314 = 0xA314,
            SLCAM_MODEL_A321U = 0xA321,
            SLCAM_MODEL_A322U = 0xA322,
            SLCAM_MODEL_U405 = 0x1405,
            SLCAM_MODEL_U406 = 0x1406,
            SLCAM_MODEL_U408 = 0x1408,
            SLCAM_MODEL_U205 = 0x1205,
            SLCAM_MODEL_U208 = 0x1208,
            SLCAM_MODEL_U305 = 0x1305,
            SLCAM_MODEL_U306 = 0x1306,
            SLCAM_MODEL_B201 = 0x00F9,
            SLCAM_MODEL_L313 = 0x2313,
            SLCAM_MODEL_L314 = 0x2314,
            SLCAM_MODEL_UNSUPPORT = 0xFFFF
        }

        public enum SLcamLogLevel
        {
            SLCAM_LOG_TRACE,
            SLCAM_LOG_DEBUG,
            SLCAM_LOG_INFO,
            SLCAM_LOG_WARNING,
            SLCAM_LOG_ERROR,
            SLCAM_LOG_CRITICAL,
            SLCAM_LOG_OFF
        }

        public enum SLcamFocusMode
        {
            SLCAM_FOCUS_MODE_MANUAL = 0,
            SLCAM_FOCUS_MODE_AUTO,
            SLCAM_FOCUS_MODE_ONCE
        }

        public enum SLcamExposureMode
        {
            SLCAM_EXPOSURE_MODE_MANUAL = 0,
            SLCAM_EXPOSURE_MODE_AUTO
        }

        public enum SLcamWhiteBalanceMode
        {
            SLCAM_WHITE_BALANCE_MODE_MANUAL = 0,
            SLCAM_WHITE_BALANCE_MODE_AUTO
        }

        public enum SLcamPowerLineFrequence
        {
            SLCAM_POWER_LINE_FREQUENCE_60HZ = 0,
            SLCAM_POWER_LINE_FREQUENCE_50HZ
        }

        public enum SLcamVideoFormat
        {
            SLCAM_VIDEO_FORMAT_UNKNOWN = -1,
            SLCAM_VIDEO_FORMAT_I420,
            SLCAM_VIDEO_FORMAT_J420,
            SLCAM_VIDEO_FORMAT_IYUV,
            SLCAM_VIDEO_FORMAT_RGB24,
            SLCAM_VIDEO_FORMAT_BGR24,
            SLCAM_VIDEO_FORMAT_ABGR,
            SLCAM_VIDEO_FORMAT_ARGB,
            SLCAM_VIDEO_FORMAT_RGBA,
            SLCAM_VIDEO_FORMAT_BGRA,
            SLCAM_VIDEO_FORMAT_RGB565,
            SLCAM_VIDEO_FORMAT_YUY2,
            SLCAM_VIDEO_FORMAT_UYVY,
            SLCAM_VIDEO_FORMAT_MJPEG,
            SLCAM_VIDEO_FORMAT_H264,
            SLCAM_VIDEO_FORMAT_H265,
            SLCAM_VIDEO_FORMAT_NV12
        }

        public enum SLcamPixFormat
        {
            SLCAM_PIX_FORMAT_UNKNOWN = -1,
            SLCAM_PIX_FORMAT_I420,
            SLCAM_PIX_FORMAT_J420,
            SLCAM_PIX_FORMAT_IYUV,
            SLCAM_PIX_FORMAT_RGB24,
            SLCAM_PIX_FORMAT_BGR24,
            SLCAM_PIX_FORMAT_ABGR,
            SLCAM_PIX_FORMAT_ARGB,
            SLCAM_PIX_FORMAT_RGBA,
            SLCAM_PIX_FORMAT_BGRA,
            SLCAM_PIX_FORMAT_RGB565,
            SLCAM_PIX_FORMAT_YUY2,
            SLCAM_PIX_FORMAT_UYVY,
            SLCAM_PIX_FORMAT_NV12,
            SLCAM_PIX_FORMAT_GRAY8
        }


        public enum SLcamImgFormat
        {
            SLCAM_IMG_FORMAT_UNKNOWN = -1,
            SLCAM_IMG_FORMAT_PNG,
            SLCAM_IMG_FORMAT_JPG,
            SLCAM_IMG_FORMAT_BMP
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct SLcamVideoResolution
        {
            public int width;
            public int height;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct SLcamVideoFrame
        {
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
            public IntPtr[] data;

            public SLcamPixFormat fmt;
            public int width;
            public int height;

            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
            public int[] linesize;

            public long pts;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct SLcamCaptureContext
        {
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 128)]
            public string unique_name;

            public SLcamVideoResolution resolution;
            public SLcamVideoFormat cap_fmt;
            public SLcamPixFormat read_fmt;
            public int fps;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct SLcamVideoCaptureCapability
        {
            public SLcamVideoResolution resolution;
            public int max_fps;
            public SLcamVideoFormat video_fmt;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct SLcamVideoCaptureCapabilities
        {
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = SLCAM_MAX_CAP_SIZE)]
            public SLcamVideoCaptureCapability[] video_caps;

            public int nb_caps;
        }

        /**
         * Low speed   USB1.0
         * Full speed  USB1.1
         * High speed  USB2.0
         * Super speed USB3.0
         */
        public enum SLcamDevUsbSpeed
        {
            SLCAM_USB_LOW_SPEED = 0,
            SLCAM_USB_FULL_SPEED,
            SLCAM_USB_HIGH_SPEED,
            SLCAM_USB_SUPER_SPEED
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct SLcamDevInfo
        {
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 128)]
            public string name;

            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 128)]
            public string unique_name;

            public ushort vendorId;
            public SLcamModel model;
            public SLcamDevUsbSpeed speed;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct SLcamDevInfos
        {
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = SLCAM_MAX_DEVICES)]
            public SLcamDevInfo[] cameras;

            public int nb_cameras;
        }

        public struct SLcamFileSaveInfo
        {
            public SLcamImgFormat format;
            public string save_path;
            public IntPtr frame;
        }

        public struct SLcamRecordSaveInfo
        {
            public string save_path;
            public int width;
            public int height;
            public int fps;
        }

        public enum SLcamGainUnit
        {
            SLCAM_EXPOSURE_GAIN_MAGNIFICATION = 0,
            SLCAM_EXPOSURE_GAIN_DB
        }
        public struct SLPoint
        {
            public byte x;
            public byte y;
        }

        #region MyRegion

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_api_init();

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_api_destroy();

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_log_set_level(SLcamLogLevel level);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_log_set_pattern(string pattern);

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void SLcamLogCallback(int level, string msg);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_log_set_callback(SLcamLogCallback callback);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_enum_devices(ref SLcamDevInfos info);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_open(string unique_name, out IntPtr cam);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_capture_capabilities(IntPtr cam,
            out SLcamVideoCaptureCapabilities capabilities);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_capture_context(IntPtr cam, ref SLcamCaptureContext ctx);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_close(IntPtr cam);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_read_frame(IntPtr cam, ref SLcamVideoFrame frame);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_free_frame(ref SLcamVideoFrame frame);

        /**
         * Set focus mode
         * @param  mode  0 manual, 1 auto, 2 once/trigger once
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_focus_mode(IntPtr cam, int mode);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_focus_mode(IntPtr cam, out int mode);

        /**
         * Get focus state
         * @param  state  0 focusing, 1 focus done
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_focus_state(IntPtr cam, out int state);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_focus(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_focus(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_focus_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_auto_focus_range(IntPtr cam, int min_value, int max_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_auto_focus_range(IntPtr cam, out int min_value, out int max_value);

        /**
         * Increase/Decrease one focus value. Only for M Serise.
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_increase_focus(IntPtr cam);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_decrease_focus(IntPtr cam);

        /**
         * Set focus region, in auto focus mode(old version)
         * Support all focus camera
         * @param   x     x pos
         *          y     y pos
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_focus_region_v1(IntPtr cam, int x, int y);

        /**
         * Get focus region(old version)
         * Support all focus camera
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_focus_region_v1(IntPtr cam, out int x, out int y);

        /**
         * Set focus region, in auto focus mode
         * Support specific version camera (c31x 1.4.0/A311u 1.0.0/m202u 1.2.0/m112u 1.1.1)
         * @param  enable 0 disable, 1 enable
         *         mode   0 small,   1 big
         *                 (small  0 <= x <= 16, 0 <= y <= 14)
         *                 (big    0 <= x <= 4,  0 <= y <= 4)
         *          x     x pos
         *          y     y pos
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_focus_region_v2(IntPtr cam, int enable, int mode, int x, int y);

        /**
         * Get focus region
         * Support specific version camera (c31x 1.4.0/A311u 1.0.0/m202u 1.2.0/m112u 1.1.1)
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_focus_region_v2(IntPtr cam, out int enable, out int mode, out int x,
            out int y);

        /** Get focus region range
         *  Only get the range of small mode
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_focus_region_range(IntPtr cam, out int min_x, out int min_y,
            out int max_x, out int max_y, out int def_x, out int def_y, out int step_x, out int step_y);

        /**
         * Set focus region, in auto focus mode
         * @param enable   0 disable, 1 enable
         * @param  x       x pos
         * @param  y       y pos
         * @param  width   width
         * @param  height  height
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int slcam_set_focus_region_v3(IntPtr cam, int enable, int x, int y, int width,
            int height);

        /**
         * Get focus region
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int slcam_get_focus_region_v3(IntPtr cam, out int enable, out int x, out int y,
            out int width, out int height);

        /**
         * Get firmware version
         * C31xu/U1xx/M112u/M202u: real_version = double(version) / 100;
         * B201: real_version = double(version & 0xFFFF) / 100;
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_version(IntPtr cam, out int version);

        /**
         * Set power line frequence
         * @param  flag  0 60hz, 1 50hz
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_power_line_frequency(IntPtr cam, int flag);

        /**
         * Get power line frequence
         * @param flag  0 60hz, 1 50hz
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_power_line_frequency(IntPtr cam, out int flag);

        /**
         * Set horizontal flip
         * @param  enable  0 disable, 1 enable
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_mirror(IntPtr cam, int enable);

        /**
         * Get horizontal flip
         * @param  enable 0 disable, 1 enable
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_mirror(IntPtr cam, out int enable);

        /**
         * Set vertical flip
         * @param  enable  0 disable, 1 enable
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_flip(IntPtr cam, int enable);

        /**
         * Get vertical flip
         * @param  enable  0 disable, 1 enable
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_flip(IntPtr cam, out int enable);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_hue(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_hue(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_hue_range(IntPtr cam, out int min_value,
            out int max_value, out int def_value, out int step_value);


        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_sharpness(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_sharpness(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_sharpness_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_saturation(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_saturation(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_saturation_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_contrast(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_contrast(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_contrast_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        /**
         * Set zoom and dzoom value, value range(zoom_min + dzoom_min, zoom_max +
         * dzoom_max) NOTICE: When value > zoom_max, enable dzoom, dzoom_value = value -
         * zoom_value dzoom : digital zoom
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_zoom(IntPtr cam, int value);

        /**
         * Get zoom and dzoom value
         * When value > zoom_max, zoom_value = zoom_max, dzoom_value = value - zoom_value
         * When value <= zoom_max, zoom_value = value, dzoom_value = 0
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_zoom(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_ozoom_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_dzoom_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        /**
         * Set zoom relative
         * @param  zoom_rel  0 stop, 1 zoom out, -1 zoom in
         *         digital_zoom  invaild
         *         speed         invaild
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_zoom_relative(IntPtr cam, sbyte zoom_rel, byte digital_zoom,
            byte speed);

        // Get zoom relative
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_zoom_relative(IntPtr cam, out sbyte zoom_rel, out byte digital_zoom,
            out byte speed);

        /**
         * Set zoom speed
         * @param  speed  0 low, 1 medium, 2 high
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_zoom_speed(IntPtr cam, int speed);

        /**
         * Get zoom speed
         * @param  speed  0 low, 1 medium, 2 high
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_zoom_speed(IntPtr cam, out int speed);

        /**
         * Set exposure mode
         * @param  mode  0 manual, 1 auto
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_exposure_mode(IntPtr cam, int mode);

        /**
         * Get exposure mode
         * @param mode  0 manual, 1 auto
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_exposure_mode(IntPtr cam, out int mode);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_exposure_compensation(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_exposure_compensation(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_exposure_compensation_range(IntPtr cam, out int min_value,
            out int max_value,
            out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_exposure_time(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_exposure_time(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_exposure_time_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
        private static extern int slcam_set_long_exposure_time(IntPtr cam, int value);

        /**
         *获取当前长曝光时间
         *cam 相机句柄
         *value 获取的长曝光时间值
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
        private static extern int slcam_get_long_exposure_time(IntPtr cam, out int value);

        /**
         *获取长曝光时间范围等信息
         * cam 相机句柄
         * min_value 最小值的指针
         * max_value 最大值的指针
         * def_value 默认值的指针
         * step_value 步长的指针
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
        private static extern int slcam_get_long_exposure_time_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_exposure_gain(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_exposure_gain(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_exposure_gain_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_exposure_gain_unit(IntPtr cam, int unit);

        /**
         * Get gain unit
         * @param unit  0 magnification, 1 db
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
        private static extern int slcam_get_exposure_gain_unit(IntPtr cam, out int unit);


        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_gamma(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_gamma(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_gamma_range(IntPtr cam, out int min_value, out int max_value,
            out int def_value, out int step_value);

        /**
         * white balance mode
         * @param  mode  0 manual, 1 auto
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_white_balance_mode(IntPtr cam, int mode);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_white_balance_mode(IntPtr cam, out int mode);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_white_balance_temperature(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_white_balance_temperature(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_white_balance_temperature_range(IntPtr cam, out int min_value,
            out int max_value, out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_white_balance_component_red(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_white_balance_component_red(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_white_balance_component_red_range(IntPtr cam, out int min_value,
            out int max_value, out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_white_balance_component_green(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_white_balance_component_green(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_white_balance_component_green_range(IntPtr cam, out int min_value,
            out int max_value, out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_white_balance_component_blue(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_white_balance_component_blue(IntPtr cam, out int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_white_balance_component_blue_range(IntPtr cam, out int min_value,
            out int max_value, out int def_value, out int step_value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_auto_exposure_region(IntPtr cam, int x, int y, int w, int h);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_auto_exposure_region(IntPtr cam, out int x, out int y, out int w,
            out int h);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_auto_whitebalance_region(IntPtr cam, int x, int y, int w, int h);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_auto_whitebalance_region(IntPtr cam, out int x, out int y, out int w,
            out int h);

        /**
         * Change scene
         * @param flag  0 default scene, 1 metallography scene
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_scene(IntPtr cam, int flag);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_scene(IntPtr cam, out int flag);

        /**
         * Dynamic Range Compression
         * @param value range: 0-127
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_drc(IntPtr cam, int value);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_drc(IntPtr cam, out int value);

        /**
         * Chromatic Abberation Correction, for depurple
         * @param enable 0 disable, 1 enable
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_cac(IntPtr cam, int enable);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_cac(IntPtr cam, out int enable);

        /**
         * Local Dynamic Contrast Improvement
         * @param enable 0 disable, 1 enable
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_ldci(IntPtr cam, int enable);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_ldci(IntPtr cam, out int enable);

        /**
         * Weak Texture Enhancement
         * @param enable 0 disable, 1 enable
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_bayer_shp(IntPtr cam, int enable);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_bayer_shp(IntPtr cam, out int enable);

        /**
         * LED control
         * @param
         *        led_index  led index
         *        part_index part index
         *        level      led brightness
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_led(IntPtr cam, int led_index, int part_index, int level);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_led(IntPtr cam, int led_index, int part_index, out int level);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_led_info(IntPtr cam, out int led_number, out int part_number,
            out int min_level, out int max_level, out int def_level, out int step_level);

        /**
         * A3XX LED control
         * @param up,left,right,down  brightness of the led postion
         * part_index for led pos: 0-up, 1-left, 2-right, 3-down
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_a3xx_all_led(IntPtr cam, int up, int left, int right, int down);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_a3xx_all_led(IntPtr cam, out int up, out int left, out int right,
            out int down);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
        public static extern int slcam_set_a3xx_led(IntPtr cam, int partIndex, int enable, int level);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
        public static extern int slcam_get_a3xx_led(IntPtr cam, int partIndex, out int enable, out int level);

        /**
         * defog mode, heat to prevent fogging up
         * @param  enable  0 disable, 1 enable
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_defog(IntPtr cam, int enable);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_defog(IntPtr cam, out int enable);

        /**
         * Set user id
         * @param  userID  range(0x0-0x3FFFFFF)
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_user_id(IntPtr cam, int user_id);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_get_user_id(IntPtr cam, out int user_id);

        /**
         * Tell camera current windows version.
         * @param windows_version  7 windows7, 10 windows10, 11 windows11, other <windows7
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_windows_version(IntPtr cam, int windows_version);

        /**
       * Get model suffix
       * @param suffix model suffix
       * For example, U205A's suffix is 'A', U205's suffix is 'NUL'
       * Only one char, lowercase
       */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int slcam_get_model_suffix(IntPtr cam, out string suffix);


        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void SLcamProgressCallback(int percent, IntPtr ctx);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_upgrade(IntPtr cam, string filePath, SLcamProgressCallback callback,
            IntPtr ctx);

        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        private static extern int slcam_set_device_name(IntPtr cam, string name);

        /**
         * Get unique id
         * @param unique_id  unique id (max length 60)
         * @param len        input unique_id length
         * @param read_len   unique_id actual length
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int slcam_get_unique_id(IntPtr cam, byte[] uniqueId, int len, out int readLen);

        /**
     * get gamma mode
     * @param mode  0 ,1 bezier_curve
     */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int slcam_get_gamma_mode(IntPtr cam, out int mode);

        /**
         * set gamma mode
         * @param mode  0 ,1 bezier_curve
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int slcam_set_gamma_mode(IntPtr cam, int mode);

        /**
         * Set Gamma Bezier Curve.
         * @param points  The array of SLPoint defining the curve.
         * @param size    The number of points (max 25).
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int slcam_set_gamma_bezier_curve(IntPtr cam, SLPoint[] points, int size);

        /**
         * Get Gamma Bezier Curve.
         * @param points  The array to store retrieved points.
         * @param size    The number of points (max 25).
         * @param real_size   The pointer to store the number of points.
         */
        [DllImport(SLCamDll, ExactSpelling = true, CallingConvention = cc)]
        public static extern int slcam_get_gamma_bezier_curve(IntPtr cam, [Out] SLPoint[] points, int size,
            out int real_size);

        [DllImport("ole32.dll")]
        public static extern int CoInitializeEx(IntPtr pvReserved, uint dwCoInit);

        [DllImport("ole32.dll")]
        public static extern void CoUninitialize();

        public const uint COINIT_APARTMENTTHREADED = 0x2; // STA 模式

        #endregion dllimport

        private IntPtr _handle;
        private int[] _myArray;

        private bool CheckHResult(int ret)
        {
            return (ret >= 0);
        }

        public bool apiInit()
        {
            CoUninitialize();
            // CoInitializeEx(IntPtr.Zero, COINIT_APARTMENTTHREADED);

            //RegisterLogCallback();
            //slcam_log_set_level(SLcamLogLevel.SLCAM_LOG_TRACE);
            return slcam_api_init() >= 0;
        }

        public bool apiDestroy()
        {
            _handle = IntPtr.Zero;
            return slcam_api_destroy() >= 0;
        }

        private SLcamDevInfo[] EnumerateDevices()
        {
            SLcamDevInfos devInfos = new SLcamDevInfos();
            int result = slcam_enum_devices(ref devInfos);
            if (result != 0) return Array.Empty<SLcamDevInfo>();
            return devInfos.nb_cameras > 0
                ? devInfos.cameras.Take(devInfos.nb_cameras).ToArray()
                : Array.Empty<SLcamDevInfo>();
        }

        public int getDevicesNum()
        {
            SLcamDevInfos devInfos = new SLcamDevInfos();
            slcam_enum_devices(ref devInfos);
            return devInfos.nb_cameras;
        }

        public bool logSetLevel(SLcamLogLevel level)
        {
            return slcam_log_set_level(level) >= 0;
        }

        public bool logSetPattern(string pattern)
        {
            return slcam_log_set_pattern(pattern) >= 0;
        }

        public IntPtr openByIndex(int index)
        {
            SLcamDevInfos devInfos = new SLcamDevInfos();
            slcam_enum_devices(ref devInfos);
            if (index >= devInfos.nb_cameras) return IntPtr.Zero;
            var slcamOpen = slcam_open(devInfos.cameras[index].unique_name, out var cam);
            _unique_name = devInfos.cameras[index].unique_name;
            _handle = cam;
            return cam;
        }

        private IntPtr open(string unique_name)
        {
            var slcamOpen = slcam_open(unique_name, out var tmpHandle);
            _handle = tmpHandle;
            return _handle;
        }

        public string[] getVideoFormat()
        {
            var resolutions = new HashSet<string>();
            slcam_get_capture_capabilities(_handle, out var capabilities);

            for (int i = 0; i < capabilities.nb_caps; i++)
            {
                // 获取枚举值的名称
                string enumName = Enum.GetName(typeof(SLcamVideoFormat), capabilities.video_caps[i].video_fmt);
                if (enumName != null)
                {
                    var formatName = enumName.Substring(enumName.LastIndexOf('_') + 1);
                    resolutions.Add(formatName);
                }
            }

            return resolutions.ToArray();
        }

        public string[] getCaptureCapabilities(string format)
        {
            var resolutions = new HashSet<string>();
            slcam_get_capture_capabilities(_handle, out var capabilities);

            string enumFormatName = "SLCAM_VIDEO_FORMAT_" + format;
            var enumValue = (SLcamVideoFormat)Enum.Parse(typeof(SLcamVideoFormat), enumFormatName);

            for (int i = 0; i < capabilities.nb_caps; i++)
            {
                if (capabilities.video_caps[i].video_fmt == enumValue)
                    resolutions.Add(
                        $"{capabilities.video_caps[i].resolution.width}*{capabilities.video_caps[i].resolution.height}");
            }

            return resolutions.ToArray();
        }

        public bool getResolutions(string resolution, out int width, out int height)
        {
            if (resolution.Length <= 0)
            {
                width = height = 0;
                return false;
            }

            var strings = resolution.Split('*');
            width = int.Parse(strings[0]);
            height = int.Parse(strings[1]);
            return true;
        }

        public bool close()
        {
            var slcamClose = slcam_close(_handle);
            // _handle = IntPtr.Zero;
            _unique_name = null;
            return CheckHResult(slcamClose);
        }

        private bool getCaptureCapabilities(out SLcamVideoCaptureCapabilities capabilities)
        {
            if (HandleInvalid())
            {
                capabilities = new SLcamVideoCaptureCapabilities();
                return false;
            }

            return CheckHResult(slcam_get_capture_capabilities(_handle, out capabilities));
        }

        public string getUniqueName(int index)
        {
            if (_unique_name.Length >= 0)
                return _unique_name;
            return "";
        }

        public bool setCaptureContext(SLcamVideoResolution resolution, string format)
        {
            if (HandleInvalid())
                return false;
            string enumFormatName = "SLCAM_VIDEO_FORMAT_" + format;
            var enumValue = (SLcamVideoFormat)Enum.Parse(typeof(SLcamVideoFormat), enumFormatName);
            _context = new SLcamCaptureContext
            {
                unique_name = _unique_name,
                cap_fmt = enumValue,
                read_fmt = SLcamPixFormat.SLCAM_PIX_FORMAT_BGRA,
                resolution = resolution,
                fps = 50
            };
            bool result = CheckHResult(slcam_set_capture_context(_handle, ref _context));
            return result;
        }

        public bool setResolution(SLcamVideoResolution resolution)
        {
            if (HandleInvalid())
            {
                return false;
            }

            if (_context.resolution.width <= 0 || _context.resolution.height <= 0) return false;

            _context.resolution = resolution;
            return CheckHResult(slcam_set_capture_context(_handle, ref _context));
        }

        private bool setCaptureContext(ref SLcamCaptureContext context)
        {
            if (HandleInvalid())
            {
                context = new SLcamCaptureContext();
                return false;
            }

            return CheckHResult(slcam_set_capture_context(_handle, ref context));
        }

        public bool readFrame(ref SLcamVideoFrame frame)
        {
            if (HandleInvalid())
            {
                frame = new SLcamVideoFrame();
                return false;
            }

            var slcamReadFrame = slcam_read_frame(_handle, ref frame);
            _myArray = new int[frame.height * frame.width];

            return CheckHResult(slcamReadFrame);
        }

        public int[] copyDataFrom(IntPtr[] data)
        {
            var dataPtr = data[0];
            // 从非托管内存中复制数据：
            Marshal.Copy(dataPtr, _myArray, 0, _myArray.Length);

            return _myArray;
        }

        public bool FreeFrame(ref SLcamVideoFrame frame)
        {
            return !HandleInvalid() && CheckHResult(slcam_free_frame(ref frame));
        }

        public bool SetFocusMode(SLcamFocusMode mode)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_focus_mode(_handle, (int)mode));
        }

        public bool GetFocusMode(out int mode)
        {
            mode = default;
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_get_focus_mode(_handle, out mode));
        }

        public bool GetFocusState(out int state)
        {
            state = default;
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_get_focus_state(_handle, out state));
        }

        public bool SetFocus(int value)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_focus(_handle, value));
        }

        public bool GetFocus(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_focus(_handle, out value));
        }

        public bool GetFocusRange(out int minValue, out int maxValue, out int defValue, out int stepValue)
        {
            if (HandleInvalid())
            {
                minValue = maxValue = defValue = stepValue = default;
                return false;
            }

            return CheckHResult(slcam_get_focus_range(_handle, out minValue, out maxValue, out defValue,
                out stepValue));
        }


        public bool SetAutoFocusRange(int minValue, int maxValue)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_auto_focus_range(_handle, minValue, maxValue));
        }

        public bool GetAutoFocusRange(out int minValue, out int maxValue)
        {
            if (HandleInvalid())
            {
                minValue = default;
                maxValue = default;
                return false;
            }

            return CheckHResult(slcam_get_auto_focus_range(_handle, out minValue, out maxValue));
        }

        public bool SetFocusRegionV1(int x, int y)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_focus_region_v1(_handle, x, y));
        }

        public bool GetFocusRegionV1(out int x, out int y)
        {
            if (HandleInvalid())
            {
                x = y = default;
                return false;
            }

            return CheckHResult(slcam_get_focus_region_v1(_handle, out x, out y));
        }

        public bool SetFocusRegionV2(int enable, int mode, int x, int y)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_focus_region_v2(_handle, enable, mode, x, y));
        }

        public bool GetFocusRegionV2(out int enable, out int mode, out int x, out int y)
        {
            if (HandleInvalid())
            {
                enable = mode = x = y = default;
                return false;
            }

            return CheckHResult(slcam_get_focus_region_v2(_handle, out enable, out mode, out x, out y));
        }

        public bool GetFocusRegionRange(out int min_x, out int min_y, out int max_x, out int max_y, out int def_x,
            out int def_y, out int step_x, out int step_y)
        {
            if (HandleInvalid())
            {
                min_x = min_y = max_x = max_y = def_x = def_y = step_x = step_y = default;
                return false;
            }

            return CheckHResult(slcam_get_focus_region_range(_handle, out min_x, out min_y, out max_x, out max_y,
                out def_x, out def_y, out step_x, out step_y));
        }

        public bool SetFocusRegionV3(int enable, int x, int y, int width, int height)
        {
            if (HandleInvalid())
            {
                return false;
            }

            int result = slcam_set_focus_region_v3(_handle, enable, x, y, width, height);
            return CheckHResult(result);
        }

        public bool GetFocusRegionV3(out int enable, out int x, out int y, out int width, out int height)
        {
            enable = x = y = width = height = default;

            if (HandleInvalid())
            {
                return false;
            }

            int result = slcam_get_focus_region_v3(_handle, out int enableInt, out x, out y, out width, out height);
            return CheckHResult(result);
        }

        public bool GetVersion(out int version)
        {
            if (HandleInvalid())
            {
                version = default;
                return false;
            }

            return CheckHResult(slcam_get_version(_handle, out version));
        }

        public bool SetPowerLineFrequency(int flag)
        {
            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_set_power_line_frequency(_handle, flag));
        }

        public bool GetPowerLineFrequency(out int flag)
        {
            if (HandleInvalid())
            {
                flag = default;
                return false;
            }

            return CheckHResult(slcam_get_power_line_frequency(_handle, out flag));
        }

        public bool SetMirror(int enable)
        {
            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_set_mirror(_handle, enable));
        }

        public bool GetMirror(out int enable)
        {
            if (HandleInvalid())
            {
                enable = default;
                return false;
            }

            return CheckHResult(slcam_get_mirror(_handle, out enable));
        }

        public bool SetFlip(int enable)
        {
            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_set_flip(_handle, enable));
        }

        public bool GetFlip(out int enable)
        {
            if (HandleInvalid())
            {
                enable = default;
                return false;
            }

            return CheckHResult(slcam_get_flip(_handle, out enable));
        }

        public bool SetHue(int value)
        {
            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_set_hue(_handle, value));
        }

        public bool GetHue(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_hue(_handle, out value));
        }

        public bool GetHueRange(out int min_value, out int max_value, out int def_value, out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_hue_range(_handle, out min_value, out max_value, out def_value,
                out step_value));
        }

        public bool SetSharpness(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_sharpness(_handle, value));
        }

        public bool GetSharpness(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_sharpness(_handle, out value));
        }

        public bool GetSharpnessRange(out int min_value, out int max_value, out int def_value, out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_sharpness_range(_handle, out min_value, out max_value, out def_value,
                out step_value));
        }

        public bool SetSaturation(int value)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_saturation(_handle, value));
        }

        public bool GetSaturation(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_saturation(_handle, out value));
        }

        public bool GetSaturationRange(out int min_value, out int max_value, out int def_value, out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }


            return CheckHResult(slcam_get_saturation_range(_handle, out min_value, out max_value, out def_value,
                out step_value));
        }

        public bool SetContrast(int value)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_contrast(_handle, value));
        }

        public bool GetContrast(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_contrast(_handle, out value));
        }

        public bool GetContrastRange(out int min_value, out int max_value, out int def_value, out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_contrast_range(_handle, out min_value, out max_value, out def_value,
                out step_value));
        }

        public bool SetZoom(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_zoom(_handle, value));
        }

        public bool GetZoom(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_zoom(_handle, out value));
        }

        public bool GetOpticalZoomRange(out int min_value, out int max_value, out int def_value, out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_ozoom_range(_handle, out min_value, out max_value, out def_value,
                out step_value));
        }

        public bool GetDigitalZoomRange(out int min_value, out int max_value, out int def_value, out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_dzoom_range(_handle, out min_value, out max_value, out def_value,
                out step_value));
        }

        public bool SetZoomSpeed(int speed)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_zoom_speed(_handle, speed));
        }

        public bool GetZoomSpeed(out int speed)
        {
            if (HandleInvalid())
            {
                speed = default;
                return false;
            }

            return CheckHResult(slcam_get_zoom_speed(_handle, out speed));
        }

        public bool SetExposureMode(SLcamExposureMode mode)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_exposure_mode(_handle, (int)mode));
        }

        public bool GetExposureMode(out int mode)
        {
            if (HandleInvalid())
            {
                mode = default;
                return false;
            }

            return CheckHResult(slcam_get_exposure_mode(_handle, out mode));
        }

        public bool SetExposureCompensation(int value)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_exposure_compensation(_handle, value));
        }

        public bool GetExposureCompensation(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_exposure_compensation(_handle, out value));
        }

        public bool GetExposureCompensationRange(out int min_value, out int max_value, out int def_value,
            out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_exposure_compensation_range(_handle, out min_value, out max_value,
                out def_value, out step_value));
        }

        public bool SetExposureTime(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_exposure_time(_handle, value));
        }

        public bool GetExposureTime(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_exposure_time(_handle, out value));
        }

        public bool GetExposureTimeRange(out int min_value, out int max_value, out int def_value, out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_exposure_time_range(_handle, out min_value, out max_value, out def_value,
                out step_value));
        }

        public bool SetLongExposureTime(int value)
        {
            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_set_long_exposure_time(_handle, value));
        }

        public bool GetLongExposureTime(out int value)
        {
            value = default;

            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_get_long_exposure_time(_handle, out value));
        }

        public bool GetLongExposureTimeRange(out int minValue, out int maxValue, out int defValue, out int stepValue)
        {
            minValue = maxValue = defValue = stepValue = default;

            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_get_long_exposure_time_range(_handle, out minValue, out maxValue, out defValue,
                out stepValue));
        }

        public bool SetExposureGain(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_exposure_gain(_handle, value));
        }

        public bool GetExposureGain(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_exposure_gain(_handle, out value));
        }

        public bool GetExposureGainRange(out int min_value, out int max_value, out int def_value, out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_exposure_gain_range(_handle, out min_value, out max_value, out def_value,
                out step_value));
        }

        public bool SetExposureGainUnit(int unit)
        {
            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_set_exposure_gain_unit(_handle, unit));
        }

        public bool GetExposureGainUnit(out int unit)
        {
            if (HandleInvalid())
            {
                unit = default;
                return false;
            }

            return CheckHResult(slcam_get_exposure_gain_unit(_handle, out unit));
        }

        public bool SetGamma(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_gamma(_handle, value));
        }

        public bool GetGamma(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_gamma(_handle, out value));
        }

        public bool GetGammaRange(out int min_value, out int max_value, out int def_value, out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_gamma_range(_handle, out min_value, out max_value, out def_value,
                out step_value));
        }

        public bool SetWhiteBalanceMode(SLcamWhiteBalanceMode mode)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_white_balance_mode(_handle, (int)mode));
        }

        public bool GetWhiteBalanceMode(out int mode)
        {
            if (HandleInvalid())
            {
                mode = default;
                return false;
            }

            return CheckHResult(slcam_get_white_balance_mode(_handle, out mode));
        }

        public bool SetWhiteBalanceTemperature(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_white_balance_temperature(_handle, value));
        }

        public bool GetWhiteBalanceTemperature(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_white_balance_temperature(_handle, out value));
        }

        public bool GetWhiteBalanceTemperatureRange(out int min_value, out int max_value, out int def_value,
            out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_white_balance_temperature_range(_handle, out min_value, out max_value,
                out def_value, out step_value));
        }

        public bool SetWhiteBalanceComponentRed(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_white_balance_component_red(_handle, value));
        }

        public bool GetWhiteBalanceComponentRed(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_white_balance_component_red(_handle, out value));
        }

        public bool GetWhiteBalanceComponentRedRange(out int min_value, out int max_value, out int def_value,
            out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_white_balance_component_red_range(_handle, out min_value, out max_value,
                out def_value, out step_value));
        }

        public bool SetWhiteBalanceComponentGreen(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_white_balance_component_green(_handle, value));
        }

        public bool GetWhiteBalanceComponentGreen(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_white_balance_component_green(_handle, out value));
        }

        public bool GetWhiteBalanceComponentGreenRange(out int min_value, out int max_value, out int def_value,
            out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_white_balance_component_green_range(_handle, out min_value, out max_value,
                out def_value, out step_value));
        }

        public bool SetWhiteBalanceComponentBlue(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_white_balance_component_blue(_handle, value));
        }

        public bool GetWhiteBalanceComponentBlue(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_white_balance_component_blue(_handle, out value));
        }

        public bool GetWhiteBalanceComponentBlueRange(out int min_value, out int max_value, out int def_value,
            out int step_value)
        {
            if (HandleInvalid())
            {
                min_value = max_value = def_value = step_value = default;
                return false;
            }

            return CheckHResult(slcam_get_white_balance_component_blue_range(_handle, out min_value, out max_value,
                out def_value, out step_value));
        }

        public bool SetAutoExposureRegion(int x, int y, int w, int h)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_auto_exposure_region(_handle, x, y, w, h));
        }

        public bool GetAutoExposureRegion(out int x, out int y, out int w, out int h)
        {
            if (HandleInvalid())
            {
                x = y = w = h = default;
                return false;
            }

            return CheckHResult(slcam_get_auto_exposure_region(_handle, out x, out y, out w, out h));
        }

        public bool SetAutoWhiteBalanceRegion(int x, int y, int w, int h)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_auto_whitebalance_region(_handle, x, y, w, h));
        }

        public bool GetAutoWhiteBalanceRegion(out int x, out int y, out int w, out int h)
        {
            if (HandleInvalid())
            {
                x = y = w = h = default;
                return false;
            }

            return CheckHResult(slcam_get_auto_whitebalance_region(_handle, out x, out y, out w, out h));
        }

        public bool SetScene(int flag)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_scene(_handle, flag));
        }

        public bool GetScene(out int flag)
        {
            if (HandleInvalid())
            {
                flag = default;
                return false;
            }

            return CheckHResult(slcam_get_scene(_handle, out flag));
        }

        public bool SetDRC(int value)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_drc(_handle, value));
        }

        public bool GetDRC(out int value)
        {
            if (HandleInvalid())
            {
                value = default;
                return false;
            }

            return CheckHResult(slcam_get_drc(_handle, out value));
        }

        public bool SetCAC(int enable)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_cac(_handle, enable));
        }

        public bool GetCAC(out int enable)
        {
            if (HandleInvalid())
            {
                enable = default;
                return false;
            }

            return CheckHResult(slcam_get_cac(_handle, out enable));
        }

        public bool SetLDCI(int enable)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_ldci(_handle, enable));
        }

        public bool GetLDCI(out int enable)
        {
            if (HandleInvalid())
            {
                enable = default;
                return false;
            }

            return CheckHResult(slcam_get_ldci(_handle, out enable));
        }

        public bool SetBayerSHp(int enable)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_bayer_shp(_handle, enable));
        }

        public bool GetBayerSHp(out int enable)
        {
            if (HandleInvalid())
            {
                enable = default;
                return false;
            }

            return CheckHResult(slcam_get_bayer_shp(_handle, out enable));
        }

        public bool SetLED(int led_index, int part_index, int level)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_led(_handle, led_index, part_index, level));
        }

        public bool GetLED(int led_index, int part_index, out int level)
        {
            if (HandleInvalid())
            {
                level = default;
                return false;
            }

            return CheckHResult(slcam_get_led(_handle, led_index, part_index, out level));
        }

        public bool GetLEDInfo(out int led_number, out int part_number, out int min_level, out int max_level,
            out int def_level, out int step_level)
        {
            if (HandleInvalid())
            {
                led_number = part_number = min_level = max_level = def_level = step_level = default;
                return false;
            }

            return CheckHResult(slcam_get_led_info(_handle, out led_number, out part_number, out min_level,
                out max_level, out def_level, out step_level));
        }

        public bool SetDefog(int enable)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_defog(_handle, enable));
        }

        public bool GetDefog(out int enable)
        {
            if (HandleInvalid())
            {
                enable = default;
                return false;
            }

            return CheckHResult(slcam_get_defog(_handle, out enable));
        }

        public bool SetUserID(int user_id)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_user_id(_handle, user_id));
        }

        public bool GetUserID(out int user_id)
        {
            if (HandleInvalid())
            {
                user_id = default;
                return false;
            }

            return CheckHResult(slcam_get_user_id(_handle, out user_id));
        }

        public bool IncreaseFocus()
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_increase_focus(_handle));
        }

        public bool DecreaseFocus()
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_decrease_focus(_handle));
        }

        public bool SetZoomRelative(sbyte zoom_rel, byte digital_zoom, byte speed)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_zoom_relative(_handle, zoom_rel, digital_zoom, speed));
        }

        public bool GetZoomRelative(out sbyte zoom_rel, out byte digital_zoom, out byte speed)
        {
            zoom_rel = default;
            digital_zoom = default;
            speed = default;

            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_get_zoom_relative(_handle, out zoom_rel, out digital_zoom, out speed));
        }

        public bool SetA3XXAllLED(int up, int left, int right, int down)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_a3xx_all_led(_handle, up, left, right, down));
        }

        public bool GetA3XXAllLED(out int up, out int left, out int right, out int down)
        {
            up = default;
            left = default;
            right = default;
            down = default;

            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_get_a3xx_all_led(_handle, out up, out left, out right, out down));
        }

        public bool SetA3xxLed(int partIndex, bool enable, int brightnessLevel)
        {
            if (HandleInvalid())
                return false;

            int enableValue = enable ? 1 : 0;
            return CheckHResult(slcam_set_a3xx_led(_handle, partIndex, enableValue, brightnessLevel));
        }

        public bool GetA3xxLed(int partIndex, out bool enable, out int brightnessLevel)
        {
            if (HandleInvalid())
            {
                enable = false;
                brightnessLevel = 0;
                return false;
            }

            int enableValue;
            bool result = CheckHResult(slcam_get_a3xx_led(_handle, partIndex, out enableValue, out brightnessLevel));
            enable = enableValue != 0;
            return result;
        }

        public bool GetModelSuffix(out string suffix)
        {
            suffix = string.Empty;

            if (HandleInvalid())
            {
                return false;
            }

            int result = slcam_get_model_suffix(_handle, out suffix);
            return CheckHResult(result);
        }

        public bool SetWindowsVersion(int windows_version)
        {
            if (HandleInvalid())
                return false;

            return CheckHResult(slcam_set_windows_version(_handle, windows_version));
        }


        public static void MyLogCallbackFunction(int level, string msg)
        {
            string desktopPath = Environment.GetFolderPath(Environment.SpecialFolder.Desktop);
            string logFilePath = Path.Combine(desktopPath, "log.txt");

            using (StreamWriter writer = new StreamWriter(logFilePath, true))
            {
                writer.WriteLine($"Log Level: {level}, Message: {msg}");
            }
        }

        public void RegisterLogCallback()
        {
            var callbackDelegate = new SLcam.SLcamLogCallback(MyLogCallbackFunction);
            SLcam.slcam_log_set_callback(callbackDelegate);
        }

        public bool Upgrade(string filePath, SLcamProgressCallback callback, IntPtr ctx)
        {
            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_upgrade(_handle, filePath, callback, ctx));
        }

        public bool SetDeviceName(string name)
        {
            if (HandleInvalid())
            {
                return false;
            }

            return CheckHResult(slcam_set_device_name(_handle, name));
        }

        public bool GetUniqueId(out byte[] uniqueId, int maxLength, out int actualLength)
        {
            uniqueId = new byte[maxLength];
            actualLength = 0;

            if (HandleInvalid())
            {
                return false;
            }

            int result = slcam_get_unique_id(_handle, uniqueId, maxLength, out actualLength);

            if (!CheckHResult(result))
            {
                return false;
            }

            // 截取实际长度的 ID
            Array.Resize(ref uniqueId, actualLength);
            return true;
        }

        public bool GetGammaMode(out int mode)
        {
            mode = default;
            return !HandleInvalid() && CheckHResult(slcam_get_gamma_mode(_handle, out mode));
        }
    
        public bool SetGammaMode(int mode)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_gamma_mode(_handle, mode));
        }
    
        public bool SetGammaBezierCurve(SLPoint[] points, int size)
        {
            return !HandleInvalid() && CheckHResult(slcam_set_gamma_bezier_curve(_handle, points, size));
        }
    
        public bool GetGammaBezierCurve(out SLPoint[] points, int size, out int realSize)
        {
            points = new SLPoint[size];
            realSize = default;
            return !HandleInvalid() && CheckHResult(slcam_get_gamma_bezier_curve(_handle, points, size, out realSize));
        }

        private bool HandleInvalid()
        {
            if (_handle == null || _handle == IntPtr.Zero)
                return true;
            return false;
        }
    }
}