import ctypes
import filecmp
import logging
import os
import platform
import shutil
import sys
import time

# 第三方库，用于视频显示
os.environ['PYGAME_HIDE_SUPPORT_PROMPT'] = '1'
import pygame

import slcam


def copy_files(source_folders, dest_folder):
    os.makedirs(dest_folder, exist_ok=True)

    for sources_folder in source_folders:
        for root, dirs, files in os.walk(sources_folder):
            for file in files:
                source_path = os.path.join(root, file)
                relative_path = os.path.relpath(root, sources_folder)

                if relative_path == '.':
                    relative_path = ''

                dest_dir = os.path.join(dest_folder, relative_path)
                os.makedirs(dest_dir, exist_ok=True)

                dest_path = os.path.join(dest_dir, file)

                if not os.path.exists(dest_path) or not filecmp.cmp(source_path, dest_path, shallow=False):
                    try:
                        shutil.copy2(source_path, dest_path)
                        # print(f'Copy succeed {source_path} -> {dest_path}')
                    except Exception as e:
                        print(f'Copy failed {source_path}: {e}')


def load_dll_path():
    script_dir = os.path.dirname(os.path.abspath(__file__))
    arch = platform.architecture()[0]
    arch_dir = 'x64' if '64' in arch else 'x86'

    if sys.platform == 'win32':
        platform_dir = 'win'
        dll_dir = 'bin'
    elif sys.platform.startswith('linux'):
        platform_dir = 'linux'
        dll_dir = 'lib'
    else:
        platform_dir = 'mac'
        dll_dir = 'lib'
        arch_dir = ''

    sdk_path = os.path.abspath(os.path.join(script_dir, '../../sdk', platform_dir, 'slcam', dll_dir, arch_dir))
    ffmpeg_path = os.path.abspath(os.path.join(script_dir, '../../sdk', platform_dir, 'FFmpeg', dll_dir, arch_dir))
    run_path = os.path.abspath(os.path.join(script_dir, 'build', arch_dir))
    copy_files([sdk_path, ffmpeg_path], run_path)
    return run_path


if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG)
    # 拷贝依赖库至指定路径并加载
    dll_path = load_dll_path()
    slcam.SLcam.set_load_dll_path(dll_path)
    slcam.SLcam.init()
    time.sleep(1)

    while True:
        dev_infos = slcam.SLcam.enum_devices()
        for i in range(dev_infos.cameraNum):
            dev_info = dev_infos.cameras[i]
            print('相机%d, 型号=%X, unique_name=%s' % (i + 1, dev_info.model, dev_info.uniqueName))
        select = int(input('请选择相机, 0重新搜索可用相机, -1退出选择\n'))
        if select == 0:
            continue
        else:
            break

    select = select - 1
    if select < 0 or select >= dev_infos.cameraNum:
        print('相机选择错误, 退出程序')
        slcam.SLcam.destroy()
        exit(0)

    dev_info = dev_infos.cameras[select]
    camera = slcam.SLcam.open(dev_info.uniqueName)

    capabilities = camera.get_capture_capabilities()
    for i in range(capabilities.capNum):
        capability = capabilities.cap[i]
        print('视频流能力%d, 格式=%d, 分辨率=%dx%d, 帧率=%d' %
              (
                  i + 1, capability.videoFmt, capability.resolution.width, capability.resolution.height,
                  capability.maxFps))

    select_capability = int(input('请选择需打开的视频流能力编号:\n')) - 1
    if select_capability < 0 or select_capability >= capabilities.capNum:
        print('视频流能力选择错误, 退出程序')
        camera.close()
        slcam.SLcam.destroy()
        exit(0)

    capability = capabilities.cap[select_capability]
    cap_ctx = slcam.SLcamCaptureContext()
    cap_ctx.uniqueName = dev_info.uniqueName
    cap_ctx.resolution = capability.resolution
    cap_ctx.capFmt = capability.videoFmt
    cap_ctx.readFmt = slcam.SLCAM_PIX_FORMAT_RGB24
    cap_ctx.fps = capability.maxFps
    camera.set_capture_context(cap_ctx)

    pygame.init()
    flags = pygame.SHOWN | pygame.RESIZABLE
    screen_w = 960
    screen_h = 540
    screen = pygame.display.set_mode((screen_w, screen_h), flags)
    pygame.display.set_caption('UVC SDK Demo')
    pygame.key.stop_text_input()
    clock = pygame.time.Clock()
    running = True

    capture_frame = False
    recording = False
    save_file_index = 0
    get_frame_failed_count = 0
    while running:
        for event in pygame.event.get():
            if event.type == pygame.KEYDOWN:
                # 返回键退出程序
                if event.key == pygame.K_ESCAPE:
                    running = False
                # 按A拍照
                elif event.key == pygame.K_a:
                    capture_frame = True
                # 按B开始录像
                elif event.key == pygame.K_b:
                    record_Info = slcam.SLcamRecordSaveInfo()
                    record_Info.savePath = os.path.expanduser('~/Videos/record.mp4').encode('utf-8')
                    record_Info.width = cap_ctx.resolution.width
                    record_Info.height = cap_ctx.resolution.height
                    record_Info.fps = cap_ctx.fps
                    camera.record_start(record_Info)
                    recording = True
                    print('开始录像, 保存至: %s' % (record_Info.savePath))
                # 按C停止录像
                elif event.key == pygame.K_c:
                    camera.record_stop()
                    recording = False
                    print('停止录像')
                elif event.key == pygame.K_d:
                    camera.set_flip(True)
                elif event.key == pygame.K_e:
                    camera.set_flip(False)
                elif event.key == pygame.K_j:
                    # 一键聚焦
                    mode = slcam.SLCAM_FOCUS_MODE_ONCE
                    camera.set_focus_mode(mode)
                elif event.key == pygame.K_k:
                    # 自动曝光示例
                    mode = slcam.SLCAM_EXPOSURE_MODE_AUTO
                    camera.set_exposure_mode(mode)
                elif event.key == pygame.K_l:
                    # 手动曝光示例
                    mode = slcam.SLCAM_EXPOSURE_MODE_MANUAL
                    camera.set_exposure_mode(mode)
                elif event.key == pygame.K_m:
                    # 自动白平衡示例
                    mode = slcam.SLCAM_WHITE_BALANCE_MODE_AUTO
                    camera.set_white_balance_mode(mode)
                elif event.key == pygame.K_n:
                    # 手动白平衡示例
                    mode = slcam.SLCAM_WHITE_BALANCE_MODE_MANUAL
                    camera.set_white_balance_mode(mode)
                # elif ......
            elif event.type == pygame.QUIT:
                running = False
            elif event.type == pygame.WINDOWRESIZED:
                screen_w = event.x
                screen_h = event.y

        try:
            frame = slcam.SLcamVideoFrame()
            ret = camera.read_frame(frame)

            if ret != slcam.SLCAMRET_SUCCESS:
                get_frame_failed_count = get_frame_failed_count + 1
                if get_frame_failed_count > 300:
                    print('UVC camera disconnect!')
                    running = False
            else:
                get_frame_failed_count = 0
                rgb_bytes = ctypes.string_at(frame.data[0],
                                             frame.linesize[0] * frame.height)
                img = pygame.image.frombytes(rgb_bytes, (frame.width, frame.height), 'RGB')

                if capture_frame:
                    file_save_info = slcam.SLcamFileSaveInfo()
                    file_save_info.format = slcam.SLCAM_IMG_FORMAT_JPG
                    # 无需增加文件尾缀
                    file_save_info.savePath = os.path.expanduser('~/Pictures/img%d' % (save_file_index)).encode('utf-8')
                    file_save_info.frame = ctypes.pointer(frame)
                    camera.file_save_image(file_save_info)
                    save_file_index = save_file_index + 1
                    print('图片已保存至: %s' % (file_save_info.savePath))
                    capture_frame = False

                if recording:
                    camera.record_append_frame(frame)

                img = pygame.transform.scale(img, (screen_w, screen_h))
                screen.blit(img, (0, 0))
                slcam.SLcam.free_frame(frame)
        except Exception as e:
            # print(f'Frame acquisition error: {str(e)}')
            pass

        pygame.display.flip()
        clock.tick(100)  # limits FPS to 100

    pygame.quit()
    camera.close()
    slcam.SLcam.destroy()
