import platform
import sys, os
import re
import ctypes
import numpy as np
import shutil
import filecmp
import slcam
from OpenGL import GL
from PyQt5.QtCore import pyqtSignal, pyqtSlot, QTimer, QSignalBlocker, Qt, QElapsedTimer
from PyQt5.QtWidgets import (QApplication, QWidget, QLabel, QPushButton, QComboBox,
                             QCheckBox, QSlider, QGroupBox, QGridLayout, QVBoxLayout,
                             QHBoxLayout, QMessageBox, QSizePolicy, QOpenGLWidget, QFileDialog, QMessageBox)


def copy_files(source_folders, dest_folder):
    os.makedirs(dest_folder, exist_ok=True)

    for sources_folder in source_folders:
        for root, dirs, files in os.walk(sources_folder):
            for file in files:
                source_path = os.path.join(root, file)
                relative_path = os.path.relpath(root, sources_folder)

                if relative_path == '.':
                    relative_path = ''

                dest_dir = os.path.join(dest_folder, relative_path)
                os.makedirs(dest_dir, exist_ok=True)

                dest_path = os.path.join(dest_dir, file)

                if not os.path.exists(dest_path) or not filecmp.cmp(source_path, dest_path, shallow=False):
                    try:
                        shutil.copy2(source_path, dest_path)
                        # print(f'Copy succeed {source_path} -> {dest_path}')
                    except Exception as e:
                        print(f'Copy failed {source_path}: {e}')


class OpenGLVideoWidget(QOpenGLWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.program = None
        self.textures = [0, 0, 0]
        self.video_size = (0, 0)
        self.aspect_ratio = 16 / 9

        self.vertices = np.array([
            -1.0, -1.0,
            1.0, -1.0,
            -1.0, 1.0,
            1.0, 1.0
        ], dtype=np.float32)

        self.tex_coords = np.array([
            0.0, 1.0,
            1.0, 1.0,
            0.0, 0.0,
            1.0, 0.0
        ], dtype=np.float32)

    def initializeGL(self):
        self.program = GL.glCreateProgram()
        vertex_shader = GL.glCreateShader(GL.GL_VERTEX_SHADER)
        vs_source = '''
           #version 330 core
           layout (location = 0) in vec4 vertexIn;
           layout (location = 1) in vec2 textureIn;
           out vec2 textureOut;
           void main(void)
           {
               gl_Position = vertexIn;
               textureOut = textureIn;
           }
           '''
        GL.glShaderSource(vertex_shader, vs_source)
        GL.glCompileShader(vertex_shader)
        GL.glAttachShader(self.program, vertex_shader)

        fragment_shader = GL.glCreateShader(GL.GL_FRAGMENT_SHADER)
        fs_source = '''
           #version 330 core
           in vec2 textureOut;
           uniform sampler2D tex_y;
           uniform sampler2D tex_u;
           uniform sampler2D tex_v;
           out vec4 FragColor;
           void main(void)
           {
               vec3 yuv;
               vec3 rgb;
               yuv.x = texture(tex_y, textureOut).r - 0.063;
               yuv.y = texture(tex_u, textureOut).r - 0.5;
               yuv.z = texture(tex_v, textureOut).r - 0.5;
               rgb = mat3(1.164,  1.164,  1.164,
                          0.0, -0.392,  2.017,
                          1.596, -0.813,  0.0) * yuv;
               FragColor = vec4(rgb, 1.0);
           }
           '''
        GL.glShaderSource(fragment_shader, fs_source)
        GL.glCompileShader(fragment_shader)
        GL.glAttachShader(self.program, fragment_shader)

        GL.glLinkProgram(self.program)

        self.textures = GL.glGenTextures(3)
        for i in range(3):
            GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[i])
            GL.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_MAG_FILTER, GL.GL_LINEAR)
            GL.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_MIN_FILTER, GL.GL_LINEAR)
            GL.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_WRAP_S, GL.GL_CLAMP_TO_EDGE)
            GL.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_WRAP_T, GL.GL_CLAMP_TO_EDGE)

    def paintGL(self):
        if self.video_size == (0, 0):
            return

        GL.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT)
        GL.glUseProgram(self.program)

        view_w, view_h = self.width(), self.height()
        video_ratio = self.aspect_ratio

        if view_w / view_h > video_ratio:
            render_h = view_h
            render_w = int(render_h * video_ratio)
        else:
            render_w = view_w
            render_h = int(render_w / video_ratio)

        x = (view_w - render_w) // 2
        y = (view_h - render_h) // 2
        GL.glViewport(x, y, render_w, render_h)

        vertex_attr = GL.glGetAttribLocation(self.program, 'vertexIn')
        GL.glEnableVertexAttribArray(vertex_attr)
        GL.glVertexAttribPointer(vertex_attr, 2, GL.GL_FLOAT, GL.GL_FALSE, 0, self.vertices)

        tex_attr = GL.glGetAttribLocation(self.program, 'textureIn')
        GL.glEnableVertexAttribArray(tex_attr)
        GL.glVertexAttribPointer(tex_attr, 2, GL.GL_FLOAT, GL.GL_FALSE, 0, self.tex_coords)

        for i, name in enumerate(['tex_y', 'tex_u', 'tex_v']):
            GL.glActiveTexture(GL.GL_TEXTURE0 + i)
            GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[i])
            GL.glUniform1i(GL.glGetUniformLocation(self.program, name), i)

        GL.glDrawArrays(GL.GL_TRIANGLE_STRIP, 0, 4)
        GL.glFlush()

    def resizeGL(self, w, h):
        GL.glViewport(0, 0, w, h)

    def update_yuv(self, y_data, u_data, v_data, width, height):
        if (width, height) != self.video_size:
            self.init_textures(width, height)
            self.video_size = (width, height)
            self.aspect_ratio = width / height

        self.update_texture_data(y_data, u_data, v_data)
        self.update()

    def init_textures(self, width, height):
        GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[0])
        GL.glTexImage2D(GL.GL_TEXTURE_2D, 0, GL.GL_RED,
                        width, height, 0, GL.GL_RED,
                        GL.GL_UNSIGNED_BYTE, None)
        GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[1])
        GL.glTexImage2D(GL.GL_TEXTURE_2D, 0, GL.GL_RED,
                        width // 2, height // 2, 0, GL.GL_RED,
                        GL.GL_UNSIGNED_BYTE, None)
        GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[2])
        GL.glTexImage2D(GL.GL_TEXTURE_2D, 0, GL.GL_RED,
                        width // 2, height // 2, 0, GL.GL_RED,
                        GL.GL_UNSIGNED_BYTE, None)

    def update_texture_data(self, y_data, u_data, v_data):
        self.makeCurrent()

        y_ptr = y_data.ctypes.data_as(ctypes.POINTER(ctypes.c_ubyte))
        u_ptr = u_data.ctypes.data_as(ctypes.POINTER(ctypes.c_ubyte))
        v_ptr = v_data.ctypes.data_as(ctypes.POINTER(ctypes.c_ubyte))

        for i, data_ptr in enumerate([y_ptr, u_ptr, v_ptr]):
            GL.glBindTexture(GL.GL_TEXTURE_2D, self.textures[i])
            GL.glTexSubImage2D(GL.GL_TEXTURE_2D, 0, 0, 0,
                               self.video_size[0] // (2 if i else 1),
                               self.video_size[1] // (2 if i else 1),
                               GL.GL_RED, GL.GL_UNSIGNED_BYTE, data_ptr)

        self.doneCurrent()


class MainWidget(QWidget):
    def __init__(self):
        super().__init__()
        # Set the library file load path
        slcam.SLcam.set_load_dll_path(self.load_dll_path())
        slcam.SLcam.init()
        self.hcam = None
        self.dev_infos = None
        self.__ctx = None
        self.is_capture = False
        self.capture_file_index = 0

        self.frame_count = 0
        self.fps = 0
        self.fps_label = QLabel('FPS: 0.00')

        self.fps_timer = QTimer(self)
        self.fps_timer.timeout.connect(self.update_fps)
        self.fps_timer.start(1000)

        self.update_frame_timer = QTimer(self)
        self.update_frame_timer.timeout.connect(self.read_frame)
        self.setWindowTitle('SLCamera')
        self.setMinimumSize(1280, 850)
        self.init_ui()
        self.filePath = None

    @staticmethod
    def load_dll_path():
        script_dir = os.path.dirname(os.path.abspath(__file__))
        arch = platform.architecture()[0]
        arch_dir = 'x64' if '64' in arch else 'x86'

        if sys.platform == 'win32':
            platform_dir = 'win'
            dll_dir = 'bin'
        elif sys.platform.startswith('linux'):
            platform_dir = 'linux'
            dll_dir = 'lib'
        else:
            platform_dir = 'mac'
            dll_dir = 'lib'
            arch_dir = ''

        sdk_path = os.path.abspath(os.path.join(script_dir, '../../sdk', platform_dir, 'slcam', dll_dir, arch_dir))
        ffmpeg_path = os.path.abspath(os.path.join(script_dir, '../../sdk', platform_dir, 'FFmpeg', dll_dir, arch_dir))
        run_path = os.path.abspath(os.path.join(script_dir, 'build', arch_dir))
        copy_files([sdk_path, ffmpeg_path], run_path)
        return run_path

    def init_ui(self):
        main_grid = QGridLayout(self)
        main_grid.setColumnStretch(0, 1)
        main_grid.setColumnStretch(1, 4)
        main_grid.setSpacing(0)

        ctrl_panel = QWidget()
        ctrl_panel.setFixedWidth(300)
        ctrl_layout = QVBoxLayout(ctrl_panel)

        self.scan_btn = QPushButton('Scan')
        self.camera_cmb = QComboBox()
        ctrl_layout.addWidget(self.scan_btn)
        ctrl_layout.addWidget(self.camera_cmb)

        # Resolution Setting
        res_group = QGroupBox('Resolution Settings')
        self.res_cmb = QComboBox()
        res_group.setLayout(QVBoxLayout())
        res_group.layout().addWidget(self.res_cmb)

        # Exposure
        exp_group = QGroupBox('Exposure Control')
        self.auto_exp_cbox = QCheckBox('Auto Exposure')
        self.exp_bright_lab = QLabel('0')
        self.exp_time_lab = QLabel('0')
        self.exp_gain_lab = QLabel('0')
        self.exp_bright_slider = QSlider(Qt.Orientation.Horizontal)
        self.exp_time_slider = QSlider(Qt.Orientation.Horizontal)
        self.exp_gain_slider = QSlider(Qt.Orientation.Horizontal)
        self.exp_bright_slider.setTracking(False)
        self.exp_time_slider.setTracking(False)
        self.exp_gain_slider.setTracking(False)
        exp_group.setLayout(QVBoxLayout())
        exp_group.layout().addLayout(self.create_param_layout('Bright', self.exp_bright_slider, self.exp_bright_lab))
        exp_group.layout().addLayout(self.create_param_layout('Time', self.exp_time_slider, self.exp_time_lab))
        exp_group.layout().addLayout(self.create_param_layout('Gain', self.exp_gain_slider, self.exp_gain_lab))
        exp_group.layout().addWidget(self.auto_exp_cbox)

        # White Balance
        wb_group = QGroupBox('White Balance')
        self.auto_wb_cbox = QCheckBox('Auto WB')
        self.red_lab = QLabel('0')
        self.green_lab = QLabel('0')
        self.blue_lab = QLabel('0')
        self.red_slider = QSlider(Qt.Orientation.Horizontal)
        self.green_slider = QSlider(Qt.Orientation.Horizontal)
        self.blue_slider = QSlider(Qt.Orientation.Horizontal)
        self.red_slider.setTracking(False)
        self.green_slider.setTracking(False)
        self.blue_slider.setTracking(False)
        wb_group.setLayout(QVBoxLayout())
        wb_group.layout().addLayout(self.create_param_layout('Red', self.red_slider, self.red_lab))
        wb_group.layout().addLayout(self.create_param_layout('Green', self.green_slider, self.green_lab))
        wb_group.layout().addLayout(self.create_param_layout('Blue', self.blue_slider, self.blue_lab))
        wb_group.layout().addWidget(self.auto_wb_cbox)

        func_group = QGroupBox('Function')
        self.hue_lab = QLabel('0')
        self.saturation_lab = QLabel('0')
        self.contrast_lab = QLabel('0')
        self.sharpness_lab = QLabel('0')
        self.gamma_lab = QLabel('0')
        self.hue_slider = QSlider(Qt.Orientation.Horizontal)
        self.saturation_slider = QSlider(Qt.Orientation.Horizontal)
        self.contrast_slider = QSlider(Qt.Orientation.Horizontal)
        self.sharpness_slider = QSlider(Qt.Orientation.Horizontal)
        self.gamma_slider = QSlider(Qt.Orientation.Horizontal)
        self.hue_slider.setTracking(False)
        self.saturation_slider.setTracking(False)
        self.contrast_slider.setTracking(False)
        self.sharpness_slider.setTracking(False)
        self.gamma_slider.setTracking(False)
        self.power_line_frequency_cbox = QCheckBox('Power Line Frequency')
        self.flip_cbox = QCheckBox('Flip')
        self.mirror_cbox = QCheckBox('Mirror')
        func_vbox_layout = QVBoxLayout()
        func_vbox_layout.addLayout(self.create_param_layout('Hue', self.hue_slider, self.hue_lab))
        func_vbox_layout.addLayout(self.create_param_layout('Saturation', self.saturation_slider, self.saturation_lab))
        func_vbox_layout.addLayout(self.create_param_layout('Contrast', self.contrast_slider, self.contrast_lab))
        func_vbox_layout.addLayout(self.create_param_layout('Sharpness', self.sharpness_slider, self.sharpness_lab))
        func_vbox_layout.addLayout(self.create_param_layout('Gamma', self.gamma_slider, self.gamma_lab))
        func_vbox_layout.addWidget(self.power_line_frequency_cbox)
        func_vbox_layout.addWidget(self.flip_cbox)
        func_vbox_layout.addWidget(self.mirror_cbox)
        func_group.setLayout(func_vbox_layout)

        self.open_btn = QPushButton('Open Camera')
        self.close_btn = QPushButton('Close Camera')
        self.snap_btn = QPushButton('Capture')

        ctrl_layout.addWidget(self.open_btn)
        ctrl_layout.addWidget(self.close_btn)
        ctrl_layout.addWidget(res_group)
        ctrl_layout.addWidget(exp_group)
        ctrl_layout.addWidget(wb_group)
        ctrl_layout.addWidget(func_group)
        ctrl_layout.addWidget(self.snap_btn)
        ctrl_layout.addStretch()

        self.fps_label = QLabel('FPS: 0.00')
        self.fps_label.setStyleSheet('font-size: 14px; color: black;')
        ctrl_layout.addWidget(self.fps_label)
        self.fps_label.setAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignBottom)

        self.video_wiget = OpenGLVideoWidget()
        self.video_wiget.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)

        main_grid.addWidget(ctrl_panel, 0, 0)
        main_grid.addWidget(self.video_wiget, 0, 1)

        self.scan_btn.clicked.connect(self.on_scan_btn_clicked)
        self.open_btn.clicked.connect(self.on_open_btn_clicked)
        self.close_btn.clicked.connect(self.on_close_btn_clicked)
        self.auto_exp_cbox.toggled.connect(self.on_auto_exp_cbox_toggled)
        self.auto_wb_cbox.toggled.connect(self.on_auto_wb_cbox_toggled)
        self.exp_bright_slider.valueChanged.connect(self.on_exp_bright_slider_changed)
        self.exp_time_slider.valueChanged.connect(self.on_exp_time_slider_changed)
        self.exp_gain_slider.valueChanged.connect(self.on_exp_gain_slider_changed)
        self.red_slider.valueChanged.connect(self.on_red_slider_changed)
        self.green_slider.valueChanged.connect(self.on_green_slider_changed)
        self.blue_slider.valueChanged.connect(self.on_blue_slider_changed)
        self.hue_slider.valueChanged.connect(self.on_hue_slider_changed)
        self.saturation_slider.valueChanged.connect(self.on_saturation_slider_changed)
        self.contrast_slider.valueChanged.connect(self.on_contrast_slider_changed)
        self.sharpness_slider.valueChanged.connect(self.on_sharpness_slider_changed)
        self.gamma_slider.valueChanged.connect(self.on_gamma_slider_changed)
        self.power_line_frequency_cbox.toggled.connect(self.on_power_line_frequency_cbox_toggled)
        self.flip_cbox.toggled.connect(self.on_flip_cbox_toggled)
        self.mirror_cbox.toggled.connect(self.on_mirror_cbox_toggled)
        self.camera_cmb.currentIndexChanged.connect(self.on_camera_cmb_index_changed)
        self.res_cmb.currentIndexChanged.connect(self.on_res_cmb_index_changed)
        self.snap_btn.clicked.connect(self.on_snap_btn_clicked)

        self.exp_bright_slider.valueChanged.connect(lambda v: self.exp_bright_lab.setText(str(v)))
        self.exp_time_slider.valueChanged.connect(lambda v: self.exp_time_lab.setText(str(v)))
        self.exp_gain_slider.valueChanged.connect(lambda v: self.exp_gain_lab.setText(str(v)))
        self.red_slider.valueChanged.connect(lambda v: self.red_lab.setText(str(v)))
        self.green_slider.valueChanged.connect(lambda v: self.green_lab.setText(str(v)))
        self.blue_slider.valueChanged.connect(lambda v: self.blue_lab.setText(str(v)))
        self.hue_slider.valueChanged.connect(lambda v: self.hue_lab.setText(str(v)))
        self.saturation_slider.valueChanged.connect(lambda v: self.saturation_lab.setText(str(v)))
        self.contrast_slider.valueChanged.connect(lambda v: self.contrast_lab.setText(str(v)))
        self.sharpness_slider.valueChanged.connect(lambda v: self.sharpness_lab.setText(str(v)))
        self.gamma_slider.valueChanged.connect(lambda v: self.gamma_lab.setText(str(v)))

        self.set_controls_enable(False)

    def create_param_layout(self, label, slider, value_label):
        hbox = QHBoxLayout()
        hbox.addWidget(QLabel(label))
        hbox.addWidget(slider)
        hbox.addSpacing(10)
        hbox.addWidget(value_label)
        return hbox

    def on_scan_btn_clicked(self):
        self.dev_infos = slcam.SLcam.enum_devices()
        self.camera_cmb.clear()
        if self.dev_infos.cameraNum <= 0:
            QMessageBox.information(self, 'No Devices', 'No cameras detected')
            return

        for i in range(self.dev_infos.cameraNum):
            devInfo = self.dev_infos.cameras[i]
            self.camera_cmb.addItem(
                f'Camera {i + 1}',
                userData={'index': i, 'dev_info': devInfo}
            )

    def on_open_btn_clicked(self):
        if self.hcam:
            self.on_close_btn_clicked()
            return

        if self.camera_cmb.currentIndex() == -1:
            QMessageBox.warning(self, 'Selection Error', 'No camera selected')
            return

        try:
            devIndex = self.camera_cmb.currentIndex()
            devInfo = self.dev_infos.cameras[devIndex]

            self.hcam = slcam.SLcam.open(devInfo.uniqueName)
            if not self.hcam:
                raise RuntimeError('Camera open failed')

            capabilities = self.hcam.get_capture_capabilities()

            self.res_cmb.clear()
            for i in range(capabilities.capNum):
                videoCap = capabilities.cap[i]
                resolutionStr = f'{videoCap.resolution.width} * {videoCap.resolution.height}'
                if videoCap.videoFmt == slcam.SLCAM_VIDEO_FORMAT_MJPEG:
                    resolutionStr += ' MJPEG'
                elif videoCap.videoFmt == slcam.SLCAM_VIDEO_FORMAT_NV12:
                    resolutionStr += ' NV12'

                self.res_cmb.addItem(resolutionStr, videoCap.videoFmt)

            self.__ctx = slcam.SLcamCaptureContext()
            self.__ctx.uniqueName = devInfo.uniqueName
            self.__ctx.capFmt = slcam.SLCAM_VIDEO_FORMAT_MJPEG
            self.__ctx.readFmt = slcam.SLCAM_PIX_FORMAT_I420
            self.__ctx.fps = capabilities.cap[0].maxFps
            if self.res_cmb.count() > 0:
                self.res_cmb.setCurrentIndex(0)
                self.on_res_cmb_index_changed(0)
            else:
                return

            self.init_controls()
            self.update_frame_timer.start(10)

        except Exception as e:
            QMessageBox.critical(self, 'Error', str(e))
            self.on_close_btn_clicked()

    def update_fps(self):
        self.fps_label.setText(f'FPS: {self.frame_count:.2f}')
        self.frame_count = 0

    def read_frame(self):
        try:
            frame = slcam.SLcamVideoFrame()
            ret = self.hcam.read_frame(frame)

            if ret != slcam.SLCAMRET_SUCCESS:
                return

            # if frame.fmt != slcam.SLCAM_PIX_FORMAT_I420:
            #     self.hcam.FreeFrame(frame)
            #     return
            self.frame_count += 1

            if self.is_capture:
                self.save_captured_image(frame)
                self.is_capture = False

            y_plane = np.ctypeslib.as_array(
                ctypes.cast(frame.data[0], ctypes.POINTER(ctypes.c_ubyte)),
                shape=(frame.height, frame.width)
            )
            u_plane = np.ctypeslib.as_array(
                ctypes.cast(frame.data[1], ctypes.POINTER(ctypes.c_ubyte)),
                shape=(frame.height // 2, frame.width // 2)
            )
            v_plane = np.ctypeslib.as_array(
                ctypes.cast(frame.data[2], ctypes.POINTER(ctypes.c_ubyte)),
                shape=(frame.height // 2, frame.width // 2)
            )

            self.video_wiget.update_yuv(y_plane, u_plane, v_plane, frame.width, frame.height)
            self.hcam.free_frame(frame)

        except Exception as e:
            # print(f'Frame processing error: {str(e)}')
            pass

    def closeEvent(self, event):
        self.on_close_btn_clicked()
        slcam.SLcam.destroy()
        super().closeEvent(event)

    def on_close_btn_clicked(self):
        self.update_frame_timer.stop()
        if self.hcam:
            self.hcam.close()
            self.hcam = None
            self.video_wiget.video_size = (0, 0)
            self.video_wiget.update()
            self.set_controls_enable(False)

    def on_snap_btn_clicked(self):
        self.is_capture = True

    def set_controls_enable(self, enable):
        self.res_cmb.setEnabled(enable)
        self.auto_wb_cbox.setEnabled(enable)
        self.auto_exp_cbox.setEnabled(enable)
        self.exp_bright_slider.setEnabled(enable)
        self.exp_time_slider.setEnabled(enable)
        self.exp_gain_slider.setEnabled(enable)
        self.red_slider.setEnabled(enable)
        self.green_slider.setEnabled(enable)
        self.blue_slider.setEnabled(enable)
        self.hue_slider.setEnabled(enable)
        self.saturation_slider.setEnabled(enable)
        self.contrast_slider.setEnabled(enable)
        self.sharpness_slider.setEnabled(enable)
        self.gamma_slider.setEnabled(enable)
        self.power_line_frequency_cbox.setEnabled(enable)
        self.flip_cbox.setEnabled(enable)
        self.mirror_cbox.setEnabled(enable)

    def init_controls(self):
        self.set_controls_enable(True)
        with QSignalBlocker(self.exp_bright_slider), \
                QSignalBlocker(self.exp_time_slider), \
                QSignalBlocker(self.exp_gain_slider), \
                QSignalBlocker(self.red_slider), \
                QSignalBlocker(self.green_slider), \
                QSignalBlocker(self.blue_slider), \
                QSignalBlocker(self.hue_slider), \
                QSignalBlocker(self.saturation_slider), \
                QSignalBlocker(self.contrast_slider), \
                QSignalBlocker(self.sharpness_slider), \
                QSignalBlocker(self.gamma_slider):
            exp_bright_range = self.hcam.get_exposure_compensation_range()
            self.exp_bright_slider.setRange(exp_bright_range.min, exp_bright_range.max)

            exp_time_range = self.hcam.get_exposure_time_range()
            self.exp_time_slider.setRange(exp_time_range.min, exp_time_range.max)

            exp_gain_range = self.hcam.get_exposure_gain_range()
            self.exp_gain_slider.setRange(exp_gain_range.min, exp_gain_range.max)

            wb_range = self.hcam.get_white_balance_component_red_range()
            self.red_slider.setRange(wb_range.min, wb_range.max)
            self.green_slider.setRange(wb_range.min, wb_range.max)
            self.blue_slider.setRange(wb_range.min, wb_range.max)

            hue_range = self.hcam.get_hue_range()
            self.hue_slider.setRange(hue_range.min, hue_range.max)

            saturation_range = self.hcam.get_saturation_range()
            self.saturation_slider.setRange(saturation_range.min, saturation_range.max)

            contrast_range = self.hcam.get_contrast_range()
            self.contrast_slider.setRange(contrast_range.min, contrast_range.max)

            sharpness_range = self.hcam.get_sharpness_range()
            self.sharpness_slider.setRange(sharpness_range.min, sharpness_range.max)

            gamma_range = self.hcam.get_gamma_range()
            self.gamma_slider.setRange(gamma_range.min, gamma_range.max)

        self.exp_bright_slider.setValue(self.hcam.get_exposure_compensation())
        self.exp_time_slider.setValue(self.hcam.get_exposure_time())
        self.exp_gain_slider.setValue(self.hcam.get_exposure_gain())
        self.red_slider.setValue(self.hcam.get_white_balance_component_red())
        self.green_slider.setValue(self.hcam.get_white_balance_component_green())
        self.blue_slider.setValue(self.hcam.get_white_balance_component_blue())
        self.hue_slider.setValue(self.hcam.get_hue())
        self.saturation_slider.setValue(self.hcam.get_saturation())
        self.contrast_slider.setValue(self.hcam.get_contrast())
        self.sharpness_slider.setValue(self.hcam.get_sharpness())
        self.gamma_slider.setValue(self.hcam.get_gamma())

        mode = self.hcam.get_exposure_mode()
        self.auto_exp_cbox.setChecked(mode == slcam.SLCAM_EXPOSURE_MODE_AUTO)

        mode = self.hcam.get_white_balance_mode()
        self.auto_wb_cbox.setChecked(mode == slcam.SLCAM_WHITE_BALANCE_MODE_AUTO)

        self.power_line_frequency_cbox.setChecked(
            self.hcam.get_power_line_frequency() == slcam.SLCAM_POWER_LINE_FREQUENCE_50HZ)
        self.flip_cbox.setChecked(self.hcam.get_flip() == 1)
        self.mirror_cbox.setChecked(self.hcam.get_mirror() == 1)

        self.exp_bright_slider.setEnabled(self.auto_exp_cbox.isChecked())
        self.exp_time_slider.setEnabled(not self.auto_exp_cbox.isChecked())
        self.exp_gain_slider.setEnabled(not self.auto_exp_cbox.isChecked())

        self.red_slider.setEnabled(not self.auto_wb_cbox.isChecked())
        self.green_slider.setEnabled(not self.auto_wb_cbox.isChecked())
        self.blue_slider.setEnabled(not self.auto_wb_cbox.isChecked())

    def on_auto_exp_cbox_toggled(self, checked):
        self.exp_bright_slider.setValue(self.hcam.get_exposure_compensation())
        self.exp_time_slider.setValue(self.hcam.get_exposure_time())
        self.exp_gain_slider.setValue(self.hcam.get_exposure_gain())
        self.hcam.set_exposure_mode(slcam.SLCAM_EXPOSURE_MODE_AUTO if checked else slcam.SLCAM_EXPOSURE_MODE_MANUAL)

        self.exp_bright_slider.setEnabled(checked)
        self.exp_time_slider.setEnabled(not checked)
        self.exp_gain_slider.setEnabled(not checked)

    def on_exp_bright_slider_changed(self, value):
        self.hcam.set_exposure_compensation(value)

    def on_exp_time_slider_changed(self, value):
        self.hcam.set_exposure_time(value)

    def on_exp_gain_slider_changed(self, value):
        self.hcam.set_exposure_gain(value)

    def on_auto_wb_cbox_toggled(self, checked):
        self.red_slider.setValue(self.hcam.get_white_balance_component_red())
        self.green_slider.setValue(self.hcam.get_white_balance_component_green())
        self.blue_slider.setValue(self.hcam.get_white_balance_component_blue())
        self.hcam.set_white_balance_mode(
            slcam.SLCAM_WHITE_BALANCE_MODE_AUTO if checked else slcam.SLCAM_WHITE_BALANCE_MODE_MANUAL)

        self.red_slider.setEnabled(not checked)
        self.green_slider.setEnabled(not checked)
        self.blue_slider.setEnabled(not checked)

    def on_red_slider_changed(self, value):
        self.hcam.set_white_balance_component_red(value)

    def on_green_slider_changed(self, value):
        self.hcam.set_white_balance_component_green(value)

    def on_blue_slider_changed(self, value):
        self.hcam.set_white_balance_component_blue(value)

    def on_hue_slider_changed(self, value):
        self.hcam.set_hue(value)

    def on_saturation_slider_changed(self, value):
        self.hcam.set_saturation(value)

    def on_contrast_slider_changed(self, value):
        self.hcam.set_contrast(value)

    def on_sharpness_slider_changed(self, value):
        self.hcam.set_sharpness(value)

    def on_gamma_slider_changed(self, value):
        self.hcam.set_gamma(value)

    def on_power_line_frequency_cbox_toggled(self, checked):
        self.hcam.set_power_line_frequency(checked)

    def on_flip_cbox_toggled(self, checked):
        self.hcam.set_flip(checked)

    def on_mirror_cbox_toggled(self, checked):
        self.hcam.set_mirror(checked)

    def on_camera_cmb_index_changed(self, index):
        pass

    def on_res_cmb_index_changed(self, index):
        if self.__ctx is None or index == -1:
            return
        resolutionStr = self.res_cmb.currentText()
        match = re.match(r'(\d+)\s*\*\s*(\d+)\s*(MJPEG|NV12)?', resolutionStr)

        if match:
            width = int(match.group(1))
            height = int(match.group(2))
            fmt = match.group(3) if match.group(3) else None

            self.__ctx.resolution = slcam.SLcamVideoResolution(width=width, height=height)

            if fmt == 'MJPEG':
                self.__ctx.videoFmt = slcam.SLCAM_VIDEO_FORMAT_MJPEG
            elif fmt == 'NV12':
                self.__ctx.videoFmt = slcam.SLCAM_VIDEO_FORMAT_NV12

            self.hcam.set_capture_context(self.__ctx)

    def save_captured_image(self, vframe):
        options = QFileDialog.Option.DontUseNativeDialog
        file_name, selected_filter = QFileDialog.getSaveFileName(
            None,
            'Save Captured Image',
            os.path.expanduser('~/Pictures/capture%d.jpg' % (self.capture_file_index)),
            'JPEG Image (*.jpg);;PNG Image (*.png);;Windows Bitmap (*.bmp)',
            options=options
        )

        if not file_name:
            return

        if 'JPEG' in selected_filter:
            ext = '.jpg'
            save_format = slcam.SLCAM_IMG_FORMAT_JPG
        elif 'PNG' in selected_filter:
            ext = '.png'
            save_format = slcam.SLCAM_IMG_FORMAT_PNG
        elif 'Bitmap' in selected_filter:
            ext = '.bmp'
            save_format = slcam.SLCAM_IMG_FORMAT_BMP
        else:
            ext = '.jpg'
            save_format = slcam.SLCAM_IMG_FORMAT_JPG

        # Ensure correct file extension
        file_name = os.path.splitext(file_name)[0]
        os.makedirs(os.path.dirname(file_name), exist_ok=True)

        save_info = slcam.SLcamFileSaveInfo()
        save_info.format = save_format
        save_info.savePath = file_name.encode('utf-8')

        try:
            vframe_ptr = ctypes.pointer(vframe)
            save_info.frame = vframe_ptr
            self.hcam.file_save_image(save_info)
            self.capture_file_index = self.capture_file_index + 1

            QMessageBox.information(None, 'Save Successful', f'Image saved to:\n{file_name}')
        except Exception as e:
            QMessageBox.critical(None, 'Save Failed', f'Failed to save image:\n{str(e)}')

    #  Suitable for U408 upgrade
    # def open_upgrade_file_dialog(self):
    #     options = QFileDialog.Options()
    #     filePath, _ = QFileDialog.getOpenFileName(self, 'Select Firmware File', '',
    #                                               'All Files (*)', options=options)
    #     if filePath:
    #         print('cur path = ' + filePath)
    #     self.filePath = filePath

    # def update_progress(self, percent, ctx):
    #     # self.progressLabel.setText(f'Progress: {percent}%')
    #     print('cur percent = ' + percent)
    #     if percent == 100:
    #         QMessageBox.information(self, 'Success', 'Upgrade completed successfully!')

    # def start_upgrade(self):
    #     try:
    #         # context_data = ctypes.create_string_buffer(b'Upgrade Context Data')\
    #         ctx = 'Some context information'
    #         self.hcam.upgrade(self.filePath, self.update_progress, ctx)
    #     except Exception as e:
    #         QMessageBox.critical(self, 'Error', str(e))


if __name__ == '__main__':
    app = QApplication(sys.argv)
    window = MainWidget()
    window.show()
    sys.exit(app.exec())
