import sys, ctypes, os.path

SLCAM_MAX_DEVICES = 16
SLCAM_MAX_CAP_SIZE = 32

# SLcamRet
SLCAMRET_SUCCESS = 0x00000000
SLCAMRET_FAILURE = 0x80000000

# SLcamModel
SLCAM_MODEL_M112U = 0x0112
SLCAM_MODEL_M114U = 0x0114
SLCAM_MODEL_M202U = 0x0202
SLCAM_MODEL_M122 = 0x0122
SLCAM_MODEL_M124 = 0x0124
SLCAM_MODEL_M212 = 0x0212
SLCAM_MODEL_C310 = 0xC310
SLCAM_MODEL_C311 = 0xC311
SLCAM_MODEL_C312 = 0xC312
SLCAM_MODEL_C313 = 0xC313
SLCAM_MODEL_C314 = 0xC314
SLCAM_MODEL_U106 = 0x1106
SLCAM_MODEL_U108 = 0x1108
SLCAM_MODEL_U112 = 0x1112
SLCAM_MODEL_U120 = 0x1120
SLCAM_MODEL_A311U = 0xA311
SLCAM_MODEL_A312U = 0xA312
SLCAM_MODEL_A313 = 0xA313
SLCAM_MODEL_A314 = 0xA314
SLCAM_MODEL_A321U = 0xA321
SLCAM_MODEL_A322U = 0xA322
SLCAM_MODEL_A323 = 0xA323
SLCAM_MODEL_A324 = 0xA324
SLCAM_MODEL_U405 = 0x1405
SLCAM_MODEL_U406 = 0x1406
SLCAM_MODEL_U408 = 0x1408
SLCAM_MODEL_U202 = 0x1202
SLCAM_MODEL_U205 = 0x1205
SLCAM_MODEL_U208 = 0x1208
SLCAM_MODEL_U304 = 0x1304
SLCAM_MODEL_U305 = 0x1305
SLCAM_MODEL_U306 = 0x1306
SLCAM_MODEL_B201 = 0x00F9
SLCAM_MODEL_L311 = 0x2311
SLCAM_MODEL_L313 = 0x2313
SLCAM_MODEL_L314 = 0x2314
SLCAM_MODEL_UNSUPPORT = 0xFFFF

# SLcamLogLevel
SLCAM_LOG_TRACE = 0
SLCAM_LOG_DEBUG = 1
SLCAM_LOG_INFO = 2
SLCAM_LOG_WARNING = 3
SLCAM_LOG_ERROR = 4
SLCAM_LOG_CRITICAL = 5
SLCAM_LOG_OFF = 6

# SLcamFocusMode
SLCAM_FOCUS_MODE_MANUAL = 0
SLCAM_FOCUS_MODE_AUTO = 1
SLCAM_FOCUS_MODE_ONCE = 2

# SLcamExposureMode
SLCAM_EXPOSURE_MODE_MANUAL = 0
SLCAM_EXPOSURE_MODE_AUTO = 1

# SLcamWhiteBalanceMode
SLCAM_WHITE_BALANCE_MODE_MANUAL = 0
SLCAM_WHITE_BALANCE_MODE_AUTO = 1

# SLcamPowerLineFrequence
SLCAM_POWER_LINE_FREQUENCE_60HZ = 0
SLCAM_POWER_LINE_FREQUENCE_50HZ = 1

# SLcamExposureGainUnit
SLCAM_EXPOSURE_GAIN_MAGNIFICATION = 0
SLCAM_EXPOSURE_GAIN_DB = 1

# SLcamVideoFormat
SLCAM_VIDEO_FORMAT_UNKNOWN = -1
SLCAM_VIDEO_FORMAT_I420 = 0
SLCAM_VIDEO_FORMAT_J420 = 1
SLCAM_VIDEO_FORMAT_IYUV = 2
SLCAM_VIDEO_FORMAT_RGB24 = 3
SLCAM_VIDEO_FORMAT_BGR24 = 4
SLCAM_VIDEO_FORMAT_ABGR = 5
SLCAM_VIDEO_FORMAT_ARGB = 6
SLCAM_VIDEO_FORMAT_RGBA = 7
SLCAM_VIDEO_FORMAT_BGRA = 8
SLCAM_VIDEO_FORMAT_RGB565 = 9
SLCAM_VIDEO_FORMAT_YUY2 = 10
SLCAM_VIDEO_FORMAT_UYVY = 11
SLCAM_VIDEO_FORMAT_MJPEG = 12
SLCAM_VIDEO_FORMAT_H264 = 13
SLCAM_VIDEO_FORMAT_H265 = 14
SLCAM_VIDEO_FORMAT_NV12 = 15

# SLcamPixFormat
SLCAM_PIX_FORMAT_UNKNOWN = -1
SLCAM_PIX_FORMAT_I420 = 0
SLCAM_PIX_FORMAT_J420 = 1
SLCAM_PIX_FORMAT_IYUV = 2
SLCAM_PIX_FORMAT_RGB24 = 3
SLCAM_PIX_FORMAT_BGR24 = 4
SLCAM_PIX_FORMAT_ABGR = 5
SLCAM_PIX_FORMAT_ARGB = 6
SLCAM_PIX_FORMAT_RGBA = 7
SLCAM_PIX_FORMAT_BGRA = 8
SLCAM_PIX_FORMAT_RGB565 = 9
SLCAM_PIX_FORMAT_YUY2 = 10
SLCAM_PIX_FORMAT_UYVY = 11
SLCAM_PIX_FORMAT_NV12 = 12
SLCAM_PIX_FORMAT_GRAY8 = 13

# SLcamImgFormat
SLCAM_IMG_FORMAT_UNKNOWN = -1
SLCAM_IMG_FORMAT_PNG = 0
SLCAM_IMG_FORMAT_JPG = 1
SLCAM_IMG_FORMAT_BMP = 2

# SLcamDevUsbSpeed
SLCAM_USB_LOW_SPEED = 0
SLCAM_USB_FULL_SPEED = 1
SLCAM_USB_HIGH_SPEED = 2
SLCAM_USB_SUPER_SPEED = 3

# SLcamGammaMode
SLCAM_GAMMA_MODE_NORMAL = 0
SLCAM_GAMMA_MODE_BEZIER = 1


class SLcamRetException(Exception):
    def __init__(self, ret):
        self.ret = ret


class SLcamVideoResolution(ctypes.Structure):
    _fields_ = [
        ('width', ctypes.c_int32),
        ('height', ctypes.c_int32)
    ]


class SLcamVideoFrame(ctypes.Structure):
    _fields_ = [
        ('data', ctypes.c_void_p * 4),
        ('fmt', ctypes.c_int),  # SLcamPixFormat
        ('width', ctypes.c_int32),
        ('height', ctypes.c_int32),
        ('linesize', ctypes.c_int32 * 4),
        ('pts', ctypes.c_int64)
    ]


class SLcamCaptureContext(ctypes.Structure):
    _fields_ = [
        ('uniqueName', ctypes.c_char * 128),
        ('resolution', SLcamVideoResolution),
        ('capFmt', ctypes.c_int),  # SLcamVideoFormat
        ('readFmt', ctypes.c_int),  # SLcamPixFormat
        ('fps', ctypes.c_int32)
    ]


class SLcamVideoCaptureCapability(ctypes.Structure):
    _fields_ = [
        ('resolution', SLcamVideoResolution),
        ('maxFps', ctypes.c_int32),
        ('videoFmt', ctypes.c_int)  # SLcamVideoFormat
    ]


class SLcamVideoCaptureCapabilities(ctypes.Structure):
    _fields_ = [
        ('cap', SLcamVideoCaptureCapability * SLCAM_MAX_CAP_SIZE),
        ('capNum', ctypes.c_int32)
    ]


class SLcamDevInfo(ctypes.Structure):
    _fields_ = [
        ('name', ctypes.c_char * 128),
        ('uniqueName', ctypes.c_char * 128),
        ('vendorId', ctypes.c_uint16),
        ('model', ctypes.c_uint16),  # SLcamModel
        ('speed', ctypes.c_int)  # SLcamDevUsbSpeed
    ]


class SLcamDevInfos(ctypes.Structure):
    _fields_ = [
        ('cameras', SLcamDevInfo * SLCAM_MAX_DEVICES),
        ('cameraNum', ctypes.c_int32)
    ]


class SLcamFileSaveInfo(ctypes.Structure):
    _fields_ = [
        ('format', ctypes.c_int),  # SLcamImgFormat
        ('savePath', ctypes.c_char_p),
        ('frame', ctypes.POINTER(SLcamVideoFrame))
    ]


class SLcamRecordSaveInfo(ctypes.Structure):
    _fields_ = [
        ('savePath', ctypes.c_char_p),
        ('width', ctypes.c_int32),
        ('height', ctypes.c_int32),
        ('fps', ctypes.c_int32)
    ]


class SLPoint(ctypes.Structure):
    _fields_ = [
        ('x', ctypes.c_uint8),
        ('y', ctypes.c_uint8)
    ]


class SLcamRangeInfo:
    def __init__(self, min_value, max_value, default_value, step_value):
        self.min = min_value
        self.max = max_value
        self.default = default_value
        self.step = step_value

    def __repr__(self):
        return f'RangeInfo(min={self.min}, max={self.max}, default={self.default}, step={self.step})'


class _SLcam(ctypes.Structure):
    pass  # 不透明类型


HSLcam = ctypes.POINTER(_SLcam)  # 对应 C 的 typedef struct _SLcam* HSLcam


class SLcam:
    __lib = None
    __load_dll_path = None
    __log_cb = None

    __LogCallback = ctypes.CFUNCTYPE(None, ctypes.c_int32, ctypes.c_char_p)
    __ProgressCallback = ctypes.CFUNCTYPE(None, ctypes.c_int32, ctypes.c_void_p)

    def __init__(self, h):
        self.__h = h
        self.__progress_fun = None
        self.__progress_ctx = None
        self.__progress_cb = None

    def __del__(self):
        self.close()

    def __exit__(self):
        self.close()

    def __enter__(self):
        return self

    def __nonzero__(self):
        return self.__h is not None

    def __bool__(self):
        return self.__h is not None

    @staticmethod
    def __errcheck(result, fun, args):
        if result < 0:
            raise SLcamRetException(result)
        return args

    @classmethod
    def set_load_dll_path(cls, path: str):
        cls.__load_dll_path = path

    @classmethod
    def init(cls):
        cls.__init_lib()
        return cls.__lib.slcam_api_init()

    @classmethod
    def destroy(cls):
        cls.__init_lib()
        return cls.__lib.slcam_api_destroy()

    @classmethod
    def set_log_level(cls, level):
        cls.__init_lib()
        cls.__lib.slcam_log_set_level(level)

    @classmethod
    def set_log_pattern(cls, pattern):
        cls.__init_lib()
        cls.__lib.slcam_log_set_pattern(pattern)

    @classmethod
    def set_log_callback(cls, cb):
        cls.__init_lib()
        cls.__log_cb = cls.__LogCallback(cb)
        cls.__lib.slcam_log_set_callback(cls.__log_cb)

    @classmethod
    def enum_devices(cls):
        cls.__init_lib()
        devInfos = SLcamDevInfos()
        devInfos.cameraNum = 0
        cls.__lib.slcam_enum_devices(ctypes.byref(devInfos))
        return devInfos

    @classmethod
    def open(cls, uniqueName):
        cls.__init_lib()
        h = HSLcam()
        ret = cls.__lib.slcam_open(uniqueName, ctypes.byref(h))
        if ret != SLCAMRET_SUCCESS:
            raise SLcamRetException(ret)
        if not h:
            raise RuntimeError('Failed to open camera: null handle')
        return __class__(h)

    def close(self):
        if self.__h:
            self.__lib.slcam_close(self.__h)
            self.__h = None

    def get_capture_capabilities(self):
        self.__init_lib()
        cap = SLcamVideoCaptureCapabilities()
        self.__lib.slcam_get_capture_capabilities(self.__h, ctypes.byref(cap))
        return cap

    def set_capture_context(self, ctx):
        self.__init_lib()
        self.__lib.slcam_set_capture_context(self.__h, ctypes.byref(ctx))  # SLcamCaptureContext

    def read_frame(self, frame):
        self.__init_lib()
        return self.__lib.slcam_read_frame(self.__h, ctypes.byref(frame))  # SLcamVideoFrame

    # SLcamVideoFrame frame
    @classmethod
    def free_frame(cls, frame):
        cls.__init_lib()
        cls.__lib.slcam_free_frame(ctypes.byref(frame))

    # SLcamRecordSaveInfo fileSaveInfo
    def file_save_image(self, fileSaveInfo):
        self.__lib.slcam_file_save_image(self.__h, fileSaveInfo)

    def record_start(self, recordSaveInfo):
        self.__lib.slcam_record_start(self.__h, recordSaveInfo)  # SLcamFileSaveInfo

    def record_append_frame(self, frame):
        self.__lib.slcam_record_append_frame(self.__h, ctypes.byref(frame))

    def record_stop(self):
        self.__lib.slcam_record_stop(self.__h)

    def set_focus_mode(self, mode):
        self.__lib.slcam_set_focus_mode(self.__h, mode)  # SLcamFocusMode

    def get_focus_mode(self):
        mode = ctypes.c_int32()
        self.__lib.slcam_get_focus_mode(self.__h, ctypes.byref(mode))
        return mode.value

    def get_focus_mode_ex(self):
        mode = ctypes.c_int32()
        self.__lib.slcam_get_focus_mode_ex(self.__h, ctypes.byref(mode))
        return mode.value

    def get_focus_state(self):
        state = ctypes.c_int32()
        self.__lib.slcam_get_focus_state(self.__h, ctypes.byref(state))
        return state.value

    def set_focus(self, value):
        self.__lib.slcam_set_focus(self.__h, value)

    def get_focus(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_focus(self.__h, ctypes.byref(value))
        return value.value

    def get_focus_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_focus_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                         ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_auto_focus_range(self, min_value, max_value):
        self.__lib.slcam_set_auto_focus_range(self.__h, min_value, max_value)

    def get_auto_focus_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        self.__lib.slcam_get_auto_focus_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value))
        return min_value.value, max_value.value

    # 增加一点聚焦值, 仅M系列可用
    def increase_focus(self):
        self.__lib.slcam_increase_focus(self.__h)

    # 减少一点聚焦值, 仅M系列可用
    def decrease_focus(self):
        self.__lib.slcam_decrease_focus(self.__h)

    def set_focus_region_v1(self, x, y):
        self.__lib.slcam_set_focus_region_v1(self.__h, x, y)

    def get_focus_region_v1(self):
        x = ctypes.c_int32()
        y = ctypes.c_int32()
        self.__lib.slcam_get_focus_region_v1(self.__h, ctypes.byref(x), ctypes.byref(y))
        return x.value, y.value

    def set_focus_region_v2(self, enable, mode, x, y):
        self.__lib.slcam_set_focus_region_v2(self.__h, enable, mode, x, y)

    def get_focus_region_v2(self):
        enable = ctypes.c_int32()
        mode = ctypes.c_int32()
        x = ctypes.c_int32()
        y = ctypes.c_int32()
        self.__lib.slcam_get_focus_region_v2(self.__h, ctypes.byref(enable), ctypes.byref(mode),
                                             ctypes.byref(x), ctypes.byref(y))
        return enable.value, mode.value, x.value, y.value

    def get_focus_region_range(self):
        min_x = ctypes.c_int32()
        min_y = ctypes.c_int32()
        max_x = ctypes.c_int32()
        max_y = ctypes.c_int32()
        def_x = ctypes.c_int32()
        def_y = ctypes.c_int32()
        step_x = ctypes.c_int32()
        step_y = ctypes.c_int32()
        self.__lib.slcam_get_focus_region_range(self.__h, ctypes.byref(min_x), ctypes.byref(min_y),
                                                ctypes.byref(max_x), ctypes.byref(max_y),
                                                ctypes.byref(def_x), ctypes.byref(def_y),
                                                ctypes.byref(step_x), ctypes.byref(step_y))
        return min_x.value, min_y.value, max_x.value, max_y.value, def_x.value, def_y.value, step_x.value, step_y.value

    def set_focus_region_v3(self, enable, x, y, width, height):
        self.__lib.slcam_set_focus_region_v3(self.__h, enable, x, y, width, height)

    def get_focus_region_v3(self):
        enable = ctypes.c_int32()
        x = ctypes.c_int32()
        y = ctypes.c_int32()
        width = ctypes.c_int32()
        height = ctypes.c_int32()
        self.__lib.slcam_get_focus_region_v3(self.__h, ctypes.byref(enable), ctypes.byref(x), ctypes.byref(y),
                                             ctypes.byref(width), ctypes.byref(height))
        return enable.value, x.value, y.value, width.value, height.value

    def get_version(self):
        version = ctypes.c_int32()
        self.__lib.slcam_get_version(self.__h, ctypes.byref(version))
        return version.value

    def set_power_line_frequency(self, flag):
        self.__lib.slcam_set_power_line_frequency(self.__h, flag)

    def get_power_line_frequency(self):
        flag = ctypes.c_int32()
        self.__lib.slcam_get_power_line_frequency(self.__h, ctypes.byref(flag))
        return flag.value

    def set_mirror(self, enable):
        self.__lib.slcam_set_mirror(self.__h, enable)

    def get_mirror(self):
        enable = ctypes.c_int32()
        self.__lib.slcam_get_mirror(self.__h, ctypes.byref(enable))
        return enable.value

    def set_flip(self, enable):
        self.__lib.slcam_set_flip(self.__h, enable)

    def get_flip(self):
        enable = ctypes.c_int32()
        self.__lib.slcam_get_flip(self.__h, ctypes.byref(enable))
        return enable.value

    def set_hue(self, value):
        self.__lib.slcam_set_hue(self.__h, value)

    def get_hue(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_hue(self.__h, ctypes.byref(value))
        return value.value

    def get_hue_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_hue_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                       ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_sharpness(self, value):
        self.__lib.slcam_set_sharpness(self.__h, value)

    def get_sharpness(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_sharpness(self.__h, ctypes.byref(value))
        return value.value

    def get_sharpness_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_sharpness_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                             ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_saturation(self, value):
        self.__lib.slcam_set_saturation(self.__h, value)

    def get_saturation(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_saturation(self.__h, ctypes.byref(value))
        return value.value

    def get_saturation_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_saturation_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                              ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_contrast(self, value):
        self.__init_lib()
        self.__lib.slcam_set_contrast(self.__h, value)

    def get_contrast(self):
        self.__init_lib()
        value = ctypes.c_int32()
        self.__lib.slcam_get_contrast(self.__h, ctypes.byref(value))
        return value.value

    def get_contrast_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_contrast_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                            ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_zoom(self, value):
        self.__init_lib()
        self.__lib.slcam_set_zoom(self.__h, value)

    def get_zoom(self):
        self.__init_lib()
        value = ctypes.c_int32()
        self.__lib.slcam_get_zoom(self.__h, ctypes.byref(value))
        return value.value

    def get_ozoom_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_ozoom_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                         ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def get_dzoom_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_dzoom_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                         ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_zoom_relative(self, zoom_rel, digital_zoom, speed):
        self.__lib.slcam_set_zoom_relative(self.__h, ctypes.c_int8(zoom_rel), ctypes.c_uint8(digital_zoom),
                                           ctypes.c_uint8(speed))

    def get_zoom_relative(self):
        zoom_rel = ctypes.c_int8()
        digital_zoom = ctypes.c_uint8()
        speed = ctypes.c_uint8()
        self.__lib.slcam_get_zoom_relative(self.__h, ctypes.byref(zoom_rel), ctypes.byref(digital_zoom),
                                           ctypes.byref(speed))
        return zoom_rel.value, digital_zoom.value, speed.value

    def set_zoom_speed(self, speed):
        self.__lib.slcam_set_zoom_speed(self.__h, ctypes.c_int32(speed))  # SLcamDevUsbSpeed

    def get_zoom_speed(self):
        speed = ctypes.c_int32()
        self.__lib.slcam_get_zoom_speed(self.__h, ctypes.byref(speed))
        return speed.value

    def set_exposure_mode(self, mode):
        self.__lib.slcam_set_exposure_mode(self.__h, mode)  # SLcamExposureMode

    def get_exposure_mode(self):
        mode = ctypes.c_int32()
        self.__lib.slcam_get_exposure_mode(self.__h, ctypes.byref(mode))
        return mode.value

    def set_exposure_compensation(self, value):
        self.__lib.slcam_set_exposure_compensation(self.__h, value)

    def get_exposure_compensation(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_exposure_compensation(self.__h, ctypes.byref(value))
        return value.value

    def get_exposure_compensation_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_exposure_compensation_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                                         ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_exposure_time(self, value):
        self.__lib.slcam_set_exposure_time(self.__h, ctypes.c_int32(value))

    def get_exposure_time(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_exposure_time(self.__h, ctypes.byref(value))
        return value.value

    def get_exposure_time_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_exposure_time_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                                 ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_long_exposure_time(self, value):
        self.__lib.slcam_set_long_exposure_time(self.__h, ctypes.c_int32(value))

    def get_long_exposure_time(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_long_exposure_time(self.__h, ctypes.byref(value))
        return value.value

    def get_long_exposure_time_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_long_exposure_time_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                                      ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_exposure_gain(self, value):
        self.__lib.slcam_set_exposure_gain(self.__h, value)

    def get_exposure_gain(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_exposure_gain(self.__h, ctypes.byref(value))
        return value.value

    def get_exposure_gain_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_exposure_gain_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                                 ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_exposure_gain_unit(self, unit):
        self.__lib.slcam_set_exposure_gain_unit(self.__h, ctypes.c_int32(unit))  # SLcamExposureGainUnit

    def get_exposure_gain_unit(self):
        unit = ctypes.c_int32()
        self.__lib.slcam_get_exposure_gain_unit(self.__h, ctypes.byref(unit))  # SLcamExposureGainUnit
        return unit.value

    def set_gamma(self, value):
        self.__lib.slcam_set_gamma(self.__h, value)

    def get_gamma(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_gamma(self.__h, ctypes.byref(value))
        return value.value

    def get_gamma_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_gamma_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                         ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_white_balance_mode(self, mode):
        self.__lib.slcam_set_white_balance_mode(self.__h, mode)  # SLcamWhiteBalanceMode

    def get_white_balance_mode(self):
        mode = ctypes.c_int32()
        self.__lib.slcam_get_white_balance_mode(self.__h, ctypes.byref(mode))
        return mode.value

    def set_white_balance_temperature(self, value):
        self.__lib.slcam_set_white_balance_temperature(self.__h, value)

    def get_white_balance_temperature(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_white_balance_temperature(self.__h, ctypes.byref(value))
        return value.value

    def get_white_balance_temperature_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_white_balance_temperature_range(self.__h, ctypes.byref(min_value), ctypes.byref(max_value),
                                                             ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_white_balance_component_red(self, value):
        self.__lib.slcam_set_white_balance_component_red(self.__h, ctypes.c_int32(value))

    def get_white_balance_component_red(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_white_balance_component_red(self.__h, ctypes.byref(value))
        return value.value

    def get_white_balance_component_red_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_white_balance_component_red_range(self.__h, ctypes.byref(min_value),
                                                               ctypes.byref(max_value),
                                                               ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_white_balance_component_green(self, value):
        self.__lib.slcam_set_white_balance_component_green(self.__h, ctypes.c_int32(value))

    def get_white_balance_component_green(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_white_balance_component_green(self.__h, ctypes.byref(value))
        return value.value

    def get_white_balance_component_green_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_white_balance_component_green_range(self.__h, ctypes.byref(min_value),
                                                                 ctypes.byref(max_value),
                                                                 ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_white_balance_component_blue(self, value):
        self.__lib.slcam_set_white_balance_component_blue(self.__h, ctypes.c_int32(value))

    def get_white_balance_component_blue(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_white_balance_component_blue(self.__h, ctypes.byref(value))
        return value.value

    def get_white_balance_component_blue_range(self):
        min_value = ctypes.c_int32()
        max_value = ctypes.c_int32()
        def_value = ctypes.c_int32()
        step_value = ctypes.c_int32()
        self.__lib.slcam_get_white_balance_component_blue_range(self.__h, ctypes.byref(min_value),
                                                                ctypes.byref(max_value),
                                                                ctypes.byref(def_value), ctypes.byref(step_value))
        return SLcamRangeInfo(min_value.value, max_value.value, def_value.value, step_value.value)

    def set_auto_white_balance_region(self, x, y, w, h):
        self.__lib.slcam_set_auto_whitebalance_region(self.__h, ctypes.c_int32(x), ctypes.c_int32(y),
                                                      ctypes.c_int32(w), ctypes.c_int32(h))

    def get_auto_white_balance_region(self):
        x = ctypes.c_int32()
        y = ctypes.c_int32()
        w = ctypes.c_int32()
        h = ctypes.c_int32()
        self.__lib.slcam_get_auto_whitebalance_region(self.__h, ctypes.byref(x), ctypes.byref(y),
                                                      ctypes.byref(w), ctypes.byref(h))
        return x.value, y.value, w.value, h.value

    def set_auto_exposure_region(self, x, y, w, h):
        self.__lib.slcam_set_auto_exposure_region(self.__h, ctypes.c_int32(x), ctypes.c_int32(y),
                                                  ctypes.c_int32(w), ctypes.c_int32(h))

    def get_auto_exposure_region(self):
        x = ctypes.c_int32()
        y = ctypes.c_int32()
        w = ctypes.c_int32()
        h = ctypes.c_int32()
        self.__lib.slcam_get_auto_exposure_region(self.__h, ctypes.byref(x), ctypes.byref(y), ctypes.byref(w),
                                                  ctypes.byref(h))
        return x.value, y.value, w.value, h.value

    def set_scene(self, flag):
        self.__lib.slcam_set_scene(self.__h, ctypes.c_int32(flag))

    def get_scene(self):
        flag = ctypes.c_int32()
        self.__lib.slcam_get_scene(self.__h, ctypes.byref(flag))
        return flag.value

    def set_drc(self, value):
        self.__lib.slcam_set_drc(self.__h, ctypes.c_int32(value))

    def get_drc(self):
        value = ctypes.c_int32()
        self.__lib.slcam_get_drc(self.__h, ctypes.byref(value))
        return value.value

    def set_cac(self, enable):
        self.__lib.slcam_set_cac(self.__h, ctypes.c_int32(enable))

    def get_cac(self):
        enable = ctypes.c_int32()
        self.__lib.slcam_get_cac(self.__h, ctypes.byref(enable))
        return enable.value

    def set_ldci(self, enable):
        self.__lib.slcam_set_ldci(self.__h, ctypes.c_int32(enable))

    def get_ldci(self):
        enable = ctypes.c_int32()
        self.__lib.slcam_get_ldci(self.__h, ctypes.byref(enable))
        return enable.value

    def set_bayer_shp(self, enable):
        self.__lib.slcam_set_bayer_shp(self.__h, ctypes.c_int32(enable))

    def get_bayer_shp(self):
        enable = ctypes.c_int32()
        self.__lib.slcam_get_bayer_shp(self.__h, ctypes.byref(enable))
        return enable.value

    def set_led(self, ledIndex, part_index, level):
        self.__lib.slcam_set_led(self.__h, ctypes.c_int32(ledIndex), ctypes.c_int32(part_index), ctypes.c_int32(level))

    def get_led(self, led_index, part_index):
        level = ctypes.c_int32()
        self.__lib.slcam_get_led(self.__h, ctypes.c_int32(led_index), ctypes.c_int32(part_index), ctypes.byref(level))
        return level.value

    def get_led_info(self):
        led_number = ctypes.c_int32()
        part_number = ctypes.c_int32()
        min_level = ctypes.c_int32()
        max_level = ctypes.c_int32()
        def_level = ctypes.c_int32()
        step_level = ctypes.c_int32()

        self.__lib.slcam_get_led_info(self.__h, ctypes.byref(led_number), ctypes.byref(part_number),
                                      ctypes.byref(min_level), ctypes.byref(max_level),
                                      ctypes.byref(def_level), ctypes.byref(step_level))

        return {
            'led_number': led_number.value,
            'part_number': part_number.value,
            'min_level': min_level.value,
            'max_level': max_level.value,
            'def_level': def_level.value,
            'step_level': step_level.value
        }

    def set_a3xx_all_led(self, up, left, right, down):
        self.__lib.slcam_set_a3xx_all_led(self.__h, ctypes.c_int32(up), ctypes.c_int32(left),
                                          ctypes.c_int32(right), ctypes.c_int32(down))

    def get_a3xx_all_led(self):
        up = ctypes.c_int32()
        left = ctypes.c_int32()
        right = ctypes.c_int32()
        down = ctypes.c_int32()

        self.__lib.slcam_get_a3xx_all_led(self.__h, ctypes.byref(up), ctypes.byref(left),
                                          ctypes.byref(right), ctypes.byref(down))

        return up.value, left.value, right.value, down.value

    def set_a3xx_led(self, part_index, enable, level):
        self.__lib.slcam_set_a3xx_led(self.__h, ctypes.c_int32(part_index), ctypes.c_int32(enable),
                                      ctypes.c_int32(level))

    def get_a3xx_led(self, part_index):
        enable = ctypes.c_int32()
        level = ctypes.c_int32()

        self.__lib.slcam_get_a3xx_led(self.__h, ctypes.c_int32(part_index), ctypes.byref(enable), ctypes.byref(level))

        return enable.value, level.value

    def set_defog(self, enable):
        self.__lib.slcam_set_defog(self.__h, ctypes.c_int32(enable))

    def get_defog(self):
        enable = ctypes.c_int32()
        self.__lib.slcam_get_defog(self.__h, ctypes.byref(enable))
        return enable.value

    def set_user_id(self, user_id):
        self.__lib.slcam_set_user_id(self.__h, ctypes.c_int32(user_id))

    def get_user_id(self):
        user_id = ctypes.c_int32()
        self.__lib.slcam_get_user_id(self.__h, ctypes.byref(user_id))
        return user_id.value

    def set_windows_version(self, windowsVersion):
        if self.__lib.slcam_set_windows_version:
            self.__lib.slcam_set_windows_version(self.__h, ctypes.c_int32(windowsVersion))

    def get_model_suffix(self):
        suffix = ctypes.create_string_buffer(256)
        self.__lib.slcam_get_model_suffix(self.__h, suffix)
        return suffix.value.decode('utf-8')

    def upgrade(self, file_path, callback, ctx):
        self.__progress_fun = callback
        self.__progress_ctx = ctx
        self.__progress_cb = self.__ProgressCallback(self.__progress_handler)
        self.__lib.slcam_upgrade(self.__h, ctypes.c_char_p(file_path.encode('utf-8')), callback, ctx)

    def __progress_handler(self, percent, ctx):
        if self.__progress_fun:
            self.__progress_fun(percent, self.__progress_ctx)

    def set_device_name(self, name):
        self.__lib.slcam_set_device_name(self.__h, ctypes.c_char_p(name.encode('utf-8')))

    def get_unique_id(self):
        unique_id = (ctypes.c_uint8 * 256)()
        read_len = ctypes.c_int32()
        self.__lib.slcam_get_unique_id(self.__h, unique_id, 256, ctypes.byref(read_len))
        return bytes(unique_id[:read_len.value])

    def get_gamma_mode(self):
        mode = ctypes.c_int32()
        self.__lib.slcam_get_gamma_mode(self.__h, ctypes.byref(mode))
        return mode.value

    def set_gamma_mode(self, mode):
        self.__lib.slcam_set_gamma_mode(self.__h, ctypes.c_int32(mode))  # SLcamGammaMode

    def set_gamma_bezier_curve(self, points):
        points_array = (SLPoint * len(points))(*points)
        self.__lib.slcam_set_gamma_bezier_curve(self.__h, points_array, len(points))

    def get_gamma_bezier_curve(self, size):
        points_array = (SLPoint * size)()
        read_size = ctypes.c_int32()
        self.__lib.slcam_get_gamma_bezier_curve(self.__h, points_array, size, ctypes.byref(read_size))
        return points_array[:read_size.value]

    def set_roi_region(self, enable, x, y, width, height):
        self.__lib.slcam_set_roi_region(self.__h, ctypes.c_int32(enable), ctypes.c_int32(x), ctypes.c_int32(y),
                                        ctypes.c_int32(width), ctypes.c_int32(height))

    def get_roi_region(self):
        enable = ctypes.c_int32()
        x = ctypes.c_int32()
        y = ctypes.c_int32()
        width = ctypes.c_int32()
        height = ctypes.c_int32()
        self.__lib.slcam_get_roi_region(self.__h, ctypes.byref(enable), ctypes.byref(x), ctypes.byref(y),
                                        ctypes.byref(width), ctypes.byref(height))
        return enable.value, x.value, y.value, width.value, height.value

    def set_sharpness_enhancement(self, enable):
        self.__lib.slcam_set_sharpness_enhancement(self.__h, ctypes.c_int32(enable))

    def get_sharpness_enhancement(self):
        enable = ctypes.c_int32()
        self.__lib.slcam_get_sharpness_enhancement(self.__h, ctypes.byref(enable))
        return enable.value

    @classmethod
    def __init_lib(cls):
        if cls.__lib is None:
            try:  # Firstly try to load the library in the directory where this file is located
                if cls.__load_dll_path is None:
                    dir = os.path.dirname(os.path.realpath(__file__))
                else:
                    dir = cls.__load_dll_path

                if sys.platform == 'win32':
                    cls.__lib = ctypes.cdll.LoadLibrary(os.path.join(dir, 'slcam.dll'))
                elif sys.platform.startswith('linux'):
                    cls.__lib = ctypes.cdll.LoadLibrary(os.path.join(dir, 'libslcam.so'))
                else:
                    cls.__lib = ctypes.cdll.LoadLibrary(os.path.join(dir, 'libslcam.dylib'))
            except Exception:
                pass

            if cls.__lib is None:
                if sys.platform == 'win32':
                    cls.__lib = ctypes.cdll.LoadLibrary('slcam.dll')
                elif sys.platform.startswith('linux'):
                    cls.__lib = ctypes.cdll.LoadLibrary('libslcam.so')
                else:
                    cls.__lib = ctypes.cdll.LoadLibrary('libslcam.dylib')

            cls.__lib.slcam_api_init.argtypes = None
            cls.__lib.slcam_api_init.restype = ctypes.c_int32
            cls.__lib.slcam_api_init.errcheck = cls.__errcheck

            cls.__lib.slcam_api_destroy.argtypes = None
            cls.__lib.slcam_api_destroy.restype = ctypes.c_int32
            cls.__lib.slcam_api_destroy.errcheck = cls.__errcheck

            cls.__lib.slcam_log_set_level.argtypes = [ctypes.c_int32]
            cls.__lib.slcam_log_set_level.restype = ctypes.c_int32
            cls.__lib.slcam_log_set_level.errcheck = cls.__errcheck

            cls.__lib.slcam_log_set_pattern.argtypes = [ctypes.c_char_p]
            cls.__lib.slcam_log_set_pattern.restype = ctypes.c_int32
            cls.__lib.slcam_log_set_pattern.errcheck = cls.__errcheck

            cls.__lib.slcam_log_set_callback.argtypes = [cls.__LogCallback]
            cls.__lib.slcam_log_set_callback.restype = ctypes.c_int32
            cls.__lib.slcam_log_set_callback.errcheck = cls.__errcheck

            cls.__lib.slcam_enum_devices.argtypes = [ctypes.POINTER(SLcamDevInfos)]
            cls.__lib.slcam_enum_devices.restype = ctypes.c_int32
            # cls.__lib.slcam_enum_devices.errcheck = cls.__errcheck

            cls.__lib.slcam_open.argtypes = [ctypes.c_char_p, ctypes.POINTER(HSLcam)]
            cls.__lib.slcam_open.restype = ctypes.c_int32
            cls.__lib.slcam_open.errcheck = cls.__errcheck

            cls.__lib.slcam_close.argtypes = [ctypes.c_void_p]
            cls.__lib.slcam_close.restype = ctypes.c_int32
            cls.__lib.slcam_close.errcheck = cls.__errcheck

            cls.__lib.slcam_get_capture_capabilities.argtypes = [ctypes.c_void_p,
                                                                 ctypes.POINTER(SLcamVideoCaptureCapabilities)]
            cls.__lib.slcam_get_capture_capabilities.restype = ctypes.c_int32
            cls.__lib.slcam_get_capture_capabilities.errcheck = cls.__errcheck

            cls.__lib.slcam_set_capture_context.argtypes = [ctypes.c_void_p, ctypes.POINTER(SLcamCaptureContext)]
            cls.__lib.slcam_set_capture_context.restype = ctypes.c_int32
            cls.__lib.slcam_set_capture_context.errcheck = cls.__errcheck

            cls.__lib.slcam_read_frame.argtypes = [ctypes.c_void_p, ctypes.POINTER(SLcamVideoFrame)]
            cls.__lib.slcam_read_frame.restype = ctypes.c_int32
            # cls.__lib.slcam_read_frame.errcheck = cls.__errcheck

            cls.__lib.slcam_free_frame.argtypes = [ctypes.POINTER(SLcamVideoFrame)]
            cls.__lib.slcam_free_frame.restype = ctypes.c_int32
            cls.__lib.slcam_free_frame.errcheck = cls.__errcheck

            cls.__lib.slcam_file_save_image.argtypes = [ctypes.c_void_p, SLcamFileSaveInfo]
            cls.__lib.slcam_file_save_image.restype = ctypes.c_int32
            cls.__lib.slcam_file_save_image.errcheck = cls.__errcheck

            cls.__lib.slcam_record_start.argtypes = [ctypes.c_void_p, SLcamRecordSaveInfo]
            cls.__lib.slcam_record_start.restype = ctypes.c_int32
            cls.__lib.slcam_record_start.errcheck = cls.__errcheck

            cls.__lib.slcam_record_append_frame.argtypes = [ctypes.c_void_p, ctypes.POINTER(SLcamVideoFrame)]
            cls.__lib.slcam_record_append_frame.restype = ctypes.c_int32
            cls.__lib.slcam_record_append_frame.errcheck = cls.__errcheck

            cls.__lib.slcam_record_stop.argtypes = [ctypes.c_void_p]
            cls.__lib.slcam_record_stop.restype = ctypes.c_int32
            cls.__lib.slcam_record_stop.errcheck = cls.__errcheck

            cls.__lib.slcam_set_focus_mode.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_focus_mode.restype = ctypes.c_int32
            cls.__lib.slcam_set_focus_mode.errcheck = cls.__errcheck

            cls.__lib.slcam_get_focus_mode.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_focus_mode.restype = ctypes.c_int32
            cls.__lib.slcam_get_focus_mode.errcheck = cls.__errcheck

            cls.__lib.slcam_get_focus_mode_ex.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_focus_mode_ex.restype = ctypes.c_int32
            cls.__lib.slcam_get_focus_mode_ex.errcheck = cls.__errcheck

            cls.__lib.slcam_get_focus_state.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_focus_state.restype = ctypes.c_int32
            cls.__lib.slcam_get_focus_state.errcheck = cls.__errcheck

            cls.__lib.slcam_set_focus.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_focus.restype = ctypes.c_int32
            cls.__lib.slcam_set_focus.errcheck = cls.__errcheck

            cls.__lib.slcam_get_focus.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_focus.restype = ctypes.c_int32
            cls.__lib.slcam_get_focus.errcheck = cls.__errcheck

            cls.__lib.slcam_get_focus_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                        ctypes.POINTER(ctypes.c_int32), ctypes.POINTER(ctypes.c_int32),
                                                        ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_focus_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_focus_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_auto_focus_range.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32]
            cls.__lib.slcam_set_auto_focus_range.restype = ctypes.c_int32
            cls.__lib.slcam_set_auto_focus_range.errcheck = cls.__errcheck

            cls.__lib.slcam_get_auto_focus_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                             ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_auto_focus_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_auto_focus_range.errcheck = cls.__errcheck

            cls.__lib.slcam_increase_focus.argtypes = [ctypes.c_void_p]
            cls.__lib.slcam_increase_focus.restype = ctypes.c_int32
            cls.__lib.slcam_increase_focus.errcheck = cls.__errcheck

            cls.__lib.slcam_decrease_focus.argtypes = [ctypes.c_void_p]
            cls.__lib.slcam_decrease_focus.restype = ctypes.c_int32
            cls.__lib.slcam_decrease_focus.errcheck = cls.__errcheck

            cls.__lib.slcam_set_focus_region_v1.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32]
            cls.__lib.slcam_set_focus_region_v1.restype = ctypes.c_int32
            cls.__lib.slcam_set_focus_region_v1.errcheck = cls.__errcheck

            cls.__lib.slcam_get_focus_region_v1.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_focus_region_v1.restype = ctypes.c_int32
            cls.__lib.slcam_get_focus_region_v1.errcheck = cls.__errcheck

            cls.__lib.slcam_set_focus_region_v2.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32,
                                                            ctypes.c_int32, ctypes.c_int32]
            cls.__lib.slcam_set_focus_region_v2.restype = ctypes.c_int32
            cls.__lib.slcam_set_focus_region_v2.errcheck = cls.__errcheck

            cls.__lib.slcam_get_focus_region_v2.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_focus_region_v2.restype = ctypes.c_int32
            cls.__lib.slcam_get_focus_region_v2.errcheck = cls.__errcheck

            cls.__lib.slcam_get_focus_region_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                               ctypes.POINTER(ctypes.c_int32),
                                                               ctypes.POINTER(ctypes.c_int32),
                                                               ctypes.POINTER(ctypes.c_int32),
                                                               ctypes.POINTER(ctypes.c_int32),
                                                               ctypes.POINTER(ctypes.c_int32),
                                                               ctypes.POINTER(ctypes.c_int32),
                                                               ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_focus_region_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_focus_region_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_focus_region_v3.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32,
                                                            ctypes.c_int32, ctypes.c_int32, ctypes.c_int32]
            cls.__lib.slcam_set_focus_region_v3.restype = ctypes.c_int32
            cls.__lib.slcam_set_focus_region_v3.errcheck = cls.__errcheck

            cls.__lib.slcam_get_focus_region_v3.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_focus_region_v3.restype = ctypes.c_int32
            cls.__lib.slcam_get_focus_region_v3.errcheck = cls.__errcheck

            cls.__lib.slcam_get_version.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_version.restype = ctypes.c_int32
            cls.__lib.slcam_get_version.errcheck = cls.__errcheck

            cls.__lib.slcam_set_power_line_frequency.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_power_line_frequency.restype = ctypes.c_int32
            cls.__lib.slcam_set_power_line_frequency.errcheck = cls.__errcheck

            cls.__lib.slcam_get_power_line_frequency.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_power_line_frequency.restype = ctypes.c_int32
            cls.__lib.slcam_get_power_line_frequency.errcheck = cls.__errcheck

            cls.__lib.slcam_set_mirror.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_mirror.restype = ctypes.c_int32
            cls.__lib.slcam_set_mirror.errcheck = cls.__errcheck

            cls.__lib.slcam_get_mirror.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_mirror.restype = ctypes.c_int32
            cls.__lib.slcam_get_mirror.errcheck = cls.__errcheck

            cls.__lib.slcam_set_flip.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_flip.restype = ctypes.c_int32
            cls.__lib.slcam_set_flip.errcheck = cls.__errcheck

            cls.__lib.slcam_get_flip.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_flip.restype = ctypes.c_int32
            cls.__lib.slcam_get_flip.errcheck = cls.__errcheck

            cls.__lib.slcam_set_hue.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_hue.restype = ctypes.c_int32
            cls.__lib.slcam_set_hue.errcheck = cls.__errcheck

            cls.__lib.slcam_get_hue.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_hue.restype = ctypes.c_int32
            cls.__lib.slcam_get_hue.errcheck = cls.__errcheck

            cls.__lib.slcam_get_hue_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                      ctypes.POINTER(ctypes.c_int32),
                                                      ctypes.POINTER(ctypes.c_int32), ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_hue_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_hue_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_sharpness.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_sharpness.restype = ctypes.c_int32
            cls.__lib.slcam_set_sharpness.errcheck = cls.__errcheck

            cls.__lib.slcam_get_sharpness.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_sharpness.restype = ctypes.c_int32
            cls.__lib.slcam_get_sharpness.errcheck = cls.__errcheck

            cls.__lib.slcam_get_sharpness_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32),
                                                            ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_sharpness_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_sharpness_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_saturation.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_saturation.restype = ctypes.c_int32
            cls.__lib.slcam_set_saturation.errcheck = cls.__errcheck

            cls.__lib.slcam_get_saturation.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_saturation.restype = ctypes.c_int32
            cls.__lib.slcam_get_saturation.errcheck = cls.__errcheck

            cls.__lib.slcam_get_saturation_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                             ctypes.POINTER(ctypes.c_int32),
                                                             ctypes.POINTER(ctypes.c_int32),
                                                             ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_saturation_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_saturation_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_contrast.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_contrast.restype = ctypes.c_int32
            cls.__lib.slcam_set_contrast.errcheck = cls.__errcheck

            cls.__lib.slcam_get_contrast.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_contrast.restype = ctypes.c_int32
            cls.__lib.slcam_get_contrast.errcheck = cls.__errcheck

            cls.__lib.slcam_get_contrast_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                           ctypes.POINTER(ctypes.c_int32),
                                                           ctypes.POINTER(ctypes.c_int32),
                                                           ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_contrast_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_contrast_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_zoom.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_zoom.restype = ctypes.c_int32
            cls.__lib.slcam_set_zoom.errcheck = cls.__errcheck

            cls.__lib.slcam_get_zoom.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_zoom.restype = ctypes.c_int32
            cls.__lib.slcam_get_zoom.errcheck = cls.__errcheck

            cls.__lib.slcam_get_ozoom_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                        ctypes.POINTER(ctypes.c_int32),
                                                        ctypes.POINTER(ctypes.c_int32), ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_ozoom_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_ozoom_range.errcheck = cls.__errcheck

            cls.__lib.slcam_get_dzoom_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                        ctypes.POINTER(ctypes.c_int32),
                                                        ctypes.POINTER(ctypes.c_int32), ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_dzoom_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_dzoom_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_zoom_relative.argtypes = [ctypes.c_void_p, ctypes.c_int8, ctypes.c_uint8,
                                                          ctypes.c_uint8]
            cls.__lib.slcam_set_zoom_relative.restype = ctypes.c_int32
            cls.__lib.slcam_set_zoom_relative.errcheck = cls.__errcheck

            cls.__lib.slcam_get_zoom_relative.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int8),
                                                          ctypes.POINTER(ctypes.c_uint8),
                                                          ctypes.POINTER(ctypes.c_uint8)]
            cls.__lib.slcam_get_zoom_relative.restype = ctypes.c_int32
            cls.__lib.slcam_get_zoom_relative.errcheck = cls.__errcheck

            cls.__lib.slcam_set_zoom_speed.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_zoom_speed.restype = ctypes.c_int32
            cls.__lib.slcam_set_zoom_speed.errcheck = cls.__errcheck

            cls.__lib.slcam_get_zoom_speed.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_zoom_speed.restype = ctypes.c_int32
            cls.__lib.slcam_get_zoom_speed.errcheck = cls.__errcheck

            cls.__lib.slcam_set_exposure_mode.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_exposure_mode.restype = ctypes.c_int32
            cls.__lib.slcam_set_exposure_mode.errcheck = cls.__errcheck

            cls.__lib.slcam_get_exposure_mode.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_exposure_mode.restype = ctypes.c_int32
            cls.__lib.slcam_get_exposure_mode.errcheck = cls.__errcheck

            cls.__lib.slcam_set_exposure_compensation.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_exposure_compensation.restype = ctypes.c_int32
            cls.__lib.slcam_set_exposure_compensation.errcheck = cls.__errcheck

            cls.__lib.slcam_get_exposure_compensation.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_exposure_compensation.restype = ctypes.c_int32
            cls.__lib.slcam_get_exposure_compensation.errcheck = cls.__errcheck

            cls.__lib.slcam_get_exposure_compensation_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                                        ctypes.POINTER(ctypes.c_int32),
                                                                        ctypes.POINTER(ctypes.c_int32),
                                                                        ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_exposure_compensation_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_exposure_compensation_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_exposure_time.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_exposure_time.restype = ctypes.c_int32
            cls.__lib.slcam_set_exposure_time.errcheck = cls.__errcheck

            cls.__lib.slcam_get_exposure_time.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_exposure_time.restype = ctypes.c_int32
            cls.__lib.slcam_get_exposure_time.errcheck = cls.__errcheck

            cls.__lib.slcam_get_exposure_time_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                                ctypes.POINTER(ctypes.c_int32),
                                                                ctypes.POINTER(ctypes.c_int32),
                                                                ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_exposure_time_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_exposure_time_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_long_exposure_time.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_long_exposure_time.restype = ctypes.c_int32
            cls.__lib.slcam_set_long_exposure_time.errcheck = cls.__errcheck

            cls.__lib.slcam_get_long_exposure_time.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_long_exposure_time.restype = ctypes.c_int32
            cls.__lib.slcam_get_long_exposure_time.errcheck = cls.__errcheck

            cls.__lib.slcam_get_long_exposure_time_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                                     ctypes.POINTER(ctypes.c_int32),
                                                                     ctypes.POINTER(ctypes.c_int32),
                                                                     ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_long_exposure_time_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_long_exposure_time_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_exposure_gain.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_exposure_gain.restype = ctypes.c_int32
            cls.__lib.slcam_set_exposure_gain.errcheck = cls.__errcheck

            cls.__lib.slcam_get_exposure_gain.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_exposure_gain.restype = ctypes.c_int32
            cls.__lib.slcam_get_exposure_gain.errcheck = cls.__errcheck

            cls.__lib.slcam_get_exposure_gain_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                                ctypes.POINTER(ctypes.c_int32),
                                                                ctypes.POINTER(ctypes.c_int32),
                                                                ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_exposure_gain_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_exposure_gain_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_exposure_gain_unit.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_exposure_gain_unit.restype = ctypes.c_int32
            cls.__lib.slcam_set_exposure_gain_unit.errcheck = cls.__errcheck

            cls.__lib.slcam_get_exposure_gain_unit.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_exposure_gain_unit.restype = ctypes.c_int32
            cls.__lib.slcam_get_exposure_gain_unit.errcheck = cls.__errcheck

            cls.__lib.slcam_set_gamma.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_gamma.restype = ctypes.c_int32
            cls.__lib.slcam_set_gamma.errcheck = cls.__errcheck

            cls.__lib.slcam_get_gamma.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_gamma.restype = ctypes.c_int32
            cls.__lib.slcam_get_gamma.errcheck = cls.__errcheck

            cls.__lib.slcam_get_gamma_range.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                        ctypes.POINTER(ctypes.c_int32),
                                                        ctypes.POINTER(ctypes.c_int32), ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_gamma_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_gamma_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_white_balance_mode.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_white_balance_mode.restype = ctypes.c_int32
            cls.__lib.slcam_set_white_balance_mode.errcheck = cls.__errcheck

            cls.__lib.slcam_get_white_balance_mode.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_white_balance_mode.restype = ctypes.c_int32
            cls.__lib.slcam_get_white_balance_mode.errcheck = cls.__errcheck

            cls.__lib.slcam_set_white_balance_temperature.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_white_balance_temperature.restype = ctypes.c_int32
            cls.__lib.slcam_set_white_balance_temperature.errcheck = cls.__errcheck

            cls.__lib.slcam_get_white_balance_temperature.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_white_balance_temperature.restype = ctypes.c_int32
            cls.__lib.slcam_get_white_balance_temperature.errcheck = cls.__errcheck

            cls.__lib.slcam_get_white_balance_temperature_range.argtypes = [ctypes.c_void_p,
                                                                            ctypes.POINTER(ctypes.c_int32),
                                                                            ctypes.POINTER(ctypes.c_int32),
                                                                            ctypes.POINTER(ctypes.c_int32),
                                                                            ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_white_balance_temperature_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_white_balance_temperature_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_white_balance_component_red.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_white_balance_component_red.restype = ctypes.c_int32
            cls.__lib.slcam_set_white_balance_component_red.errcheck = cls.__errcheck

            cls.__lib.slcam_get_white_balance_component_red.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_white_balance_component_red.restype = ctypes.c_int32
            cls.__lib.slcam_get_white_balance_component_red.errcheck = cls.__errcheck

            cls.__lib.slcam_get_white_balance_component_red_range.argtypes = [ctypes.c_void_p,
                                                                              ctypes.POINTER(ctypes.c_int32),
                                                                              ctypes.POINTER(ctypes.c_int32),
                                                                              ctypes.POINTER(ctypes.c_int32),
                                                                              ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_white_balance_component_red_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_white_balance_component_red_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_white_balance_component_green.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_white_balance_component_green.restype = ctypes.c_int32
            cls.__lib.slcam_set_white_balance_component_green.errcheck = cls.__errcheck

            cls.__lib.slcam_get_white_balance_component_green.argtypes = [ctypes.c_void_p,
                                                                          ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_white_balance_component_green.restype = ctypes.c_int32
            cls.__lib.slcam_get_white_balance_component_green.errcheck = cls.__errcheck

            cls.__lib.slcam_get_white_balance_component_green_range.argtypes = [ctypes.c_void_p,
                                                                                ctypes.POINTER(ctypes.c_int32),
                                                                                ctypes.POINTER(ctypes.c_int32),
                                                                                ctypes.POINTER(ctypes.c_int32),
                                                                                ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_white_balance_component_green_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_white_balance_component_green_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_white_balance_component_blue.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_white_balance_component_blue.restype = ctypes.c_int32
            cls.__lib.slcam_set_white_balance_component_blue.errcheck = cls.__errcheck

            cls.__lib.slcam_get_white_balance_component_blue.argtypes = [ctypes.c_void_p,
                                                                         ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_white_balance_component_blue.restype = ctypes.c_int32
            cls.__lib.slcam_get_white_balance_component_blue.errcheck = cls.__errcheck

            cls.__lib.slcam_get_white_balance_component_blue_range.argtypes = [ctypes.c_void_p,
                                                                               ctypes.POINTER(ctypes.c_int32),
                                                                               ctypes.POINTER(ctypes.c_int32),
                                                                               ctypes.POINTER(ctypes.c_int32),
                                                                               ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_white_balance_component_blue_range.restype = ctypes.c_int32
            cls.__lib.slcam_get_white_balance_component_blue_range.errcheck = cls.__errcheck

            cls.__lib.slcam_set_auto_exposure_region.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32,
                                                                 ctypes.c_int32, ctypes.c_int32]
            cls.__lib.slcam_set_auto_exposure_region.restype = ctypes.c_int32
            cls.__lib.slcam_set_auto_exposure_region.errcheck = cls.__errcheck

            cls.__lib.slcam_get_auto_exposure_region.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                                 ctypes.POINTER(ctypes.c_int32),
                                                                 ctypes.POINTER(ctypes.c_int32),
                                                                 ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_auto_exposure_region.restype = ctypes.c_int32
            cls.__lib.slcam_get_auto_exposure_region.errcheck = cls.__errcheck

            cls.__lib.slcam_set_auto_whitebalance_region.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32,
                                                                     ctypes.c_int32, ctypes.c_int32]
            cls.__lib.slcam_set_auto_whitebalance_region.restype = ctypes.c_int32
            cls.__lib.slcam_set_auto_whitebalance_region.errcheck = cls.__errcheck

            cls.__lib.slcam_get_auto_whitebalance_region.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                                     ctypes.POINTER(ctypes.c_int32),
                                                                     ctypes.POINTER(ctypes.c_int32),
                                                                     ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_auto_whitebalance_region.restype = ctypes.c_int32
            cls.__lib.slcam_get_auto_whitebalance_region.errcheck = cls.__errcheck

            cls.__lib.slcam_set_scene.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_scene.restype = ctypes.c_int32
            cls.__lib.slcam_set_scene.errcheck = cls.__errcheck

            cls.__lib.slcam_get_scene.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_scene.restype = ctypes.c_int32
            cls.__lib.slcam_get_scene.errcheck = cls.__errcheck

            cls.__lib.slcam_set_drc.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_drc.restype = ctypes.c_int32
            cls.__lib.slcam_set_drc.errcheck = cls.__errcheck

            cls.__lib.slcam_get_drc.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_drc.restype = ctypes.c_int32
            cls.__lib.slcam_get_drc.errcheck = cls.__errcheck

            cls.__lib.slcam_set_cac.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_cac.restype = ctypes.c_int32
            cls.__lib.slcam_set_cac.errcheck = cls.__errcheck

            cls.__lib.slcam_get_cac.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_cac.restype = ctypes.c_int32
            cls.__lib.slcam_get_cac.errcheck = cls.__errcheck

            cls.__lib.slcam_set_ldci.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_ldci.restype = ctypes.c_int32
            cls.__lib.slcam_set_ldci.errcheck = cls.__errcheck

            cls.__lib.slcam_get_ldci.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_ldci.restype = ctypes.c_int32
            cls.__lib.slcam_get_ldci.errcheck = cls.__errcheck

            cls.__lib.slcam_set_bayer_shp.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_bayer_shp.restype = ctypes.c_int32
            cls.__lib.slcam_set_bayer_shp.errcheck = cls.__errcheck

            cls.__lib.slcam_set_led.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32, ctypes.c_int32]
            cls.__lib.slcam_set_led.restype = ctypes.c_int32
            cls.__lib.slcam_set_led.errcheck = cls.__errcheck

            cls.__lib.slcam_get_led.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                ctypes.POINTER(ctypes.c_int32),
                                                ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_led.restype = ctypes.c_int32
            cls.__lib.slcam_get_led.errcheck = cls.__errcheck

            cls.__lib.slcam_get_led_info.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                     ctypes.POINTER(ctypes.c_int32),
                                                     ctypes.POINTER(ctypes.c_int32), ctypes.POINTER(ctypes.c_int32),
                                                     ctypes.POINTER(ctypes.c_int32), ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_led_info.restype = ctypes.c_int32
            cls.__lib.slcam_get_led_info.errcheck = cls.__errcheck

            cls.__lib.slcam_set_a3xx_all_led.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32,
                                                         ctypes.c_int32, ctypes.c_int32]
            cls.__lib.slcam_set_a3xx_all_led.restype = ctypes.c_int32
            cls.__lib.slcam_set_a3xx_all_led.errcheck = cls.__errcheck

            cls.__lib.slcam_get_a3xx_all_led.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                         ctypes.POINTER(ctypes.c_int32),
                                                         ctypes.POINTER(ctypes.c_int32), ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_a3xx_all_led.restype = ctypes.c_int32
            cls.__lib.slcam_get_a3xx_all_led.errcheck = cls.__errcheck

            cls.__lib.slcam_set_a3xx_led.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32, ctypes.c_int32]
            cls.__lib.slcam_set_a3xx_led.restype = ctypes.c_int32
            cls.__lib.slcam_set_a3xx_led.errcheck = cls.__errcheck

            cls.__lib.slcam_get_a3xx_led.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.POINTER(ctypes.c_int32),
                                                     ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_a3xx_led.restype = ctypes.c_int32
            cls.__lib.slcam_get_a3xx_led.errcheck = cls.__errcheck

            cls.__lib.slcam_set_defog.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_defog.restype = ctypes.c_int32
            cls.__lib.slcam_set_defog.errcheck = cls.__errcheck

            cls.__lib.slcam_get_defog.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_defog.restype = ctypes.c_int32
            cls.__lib.slcam_get_defog.errcheck = cls.__errcheck

            cls.__lib.slcam_set_user_id.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_user_id.restype = ctypes.c_int32
            cls.__lib.slcam_set_user_id.errcheck = cls.__errcheck

            cls.__lib.slcam_get_user_id.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_user_id.restype = ctypes.c_int32
            cls.__lib.slcam_get_user_id.errcheck = cls.__errcheck

            if sys.platform == 'win32':
                cls.__lib.slcam_set_windows_version.argtypes = [ctypes.c_void_p, ctypes.c_int32]
                cls.__lib.slcam_set_windows_version.restype = ctypes.c_int32
                cls.__lib.slcam_set_windows_version.errcheck = cls.__errcheck

            cls.__lib.slcam_get_model_suffix.argtypes = [ctypes.c_void_p, ctypes.c_char_p]
            cls.__lib.slcam_get_model_suffix.restype = ctypes.c_int32
            cls.__lib.slcam_get_model_suffix.errcheck = cls.__errcheck

            cls.__lib.slcam_upgrade.argtypes = [ctypes.c_void_p, ctypes.c_char_p, cls.__ProgressCallback,
                                                ctypes.c_void_p]
            cls.__lib.slcam_upgrade.restype = ctypes.c_int32
            cls.__lib.slcam_upgrade.errcheck = cls.__errcheck

            cls.__lib.slcam_set_device_name.argtypes = [ctypes.c_void_p, ctypes.c_char_p]
            cls.__lib.slcam_set_device_name.restype = ctypes.c_int32
            cls.__lib.slcam_set_device_name.errcheck = cls.__errcheck

            cls.__lib.slcam_get_unique_id.argtypes = [ctypes.c_void_p, ctypes.c_char_p, ctypes.c_int32,
                                                      ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_unique_id.restype = ctypes.c_int32
            cls.__lib.slcam_get_unique_id.errcheck = cls.__errcheck

            cls.__lib.slcam_set_gamma_mode.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_gamma_mode.restype = ctypes.c_int32
            cls.__lib.slcam_set_gamma_mode.errcheck = cls.__errcheck

            cls.__lib.slcam_get_gamma_mode.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_gamma_mode.restype = ctypes.c_int32
            cls.__lib.slcam_get_gamma_mode.errcheck = cls.__errcheck

            cls.__lib.slcam_set_gamma_bezier_curve.argtypes = [ctypes.c_void_p, ctypes.POINTER(SLPoint), ctypes.c_int32]
            cls.__lib.slcam_set_gamma_bezier_curve.restype = ctypes.c_int32
            cls.__lib.slcam_set_gamma_bezier_curve.errcheck = cls.__errcheck

            cls.__lib.slcam_get_gamma_bezier_curve.argtypes = [ctypes.c_void_p, ctypes.POINTER(SLPoint), ctypes.c_int32,
                                                               ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_gamma_bezier_curve.restype = ctypes.c_int32
            cls.__lib.slcam_get_gamma_bezier_curve.errcheck = cls.__errcheck

            cls.__lib.slcam_set_roi_region.argtypes = [ctypes.c_void_p, ctypes.c_int32, ctypes.c_int32, ctypes.c_int32,
                                                       ctypes.c_int32]
            cls.__lib.slcam_set_roi_region.restype = ctypes.c_int32
            cls.__lib.slcam_set_roi_region.errcheck = cls.__errcheck

            cls.__lib.slcam_get_roi_region.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32),
                                                       ctypes.POINTER(ctypes.c_int32),
                                                       ctypes.POINTER(ctypes.c_int32), ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_roi_region.restype = ctypes.c_int32
            cls.__lib.slcam_get_roi_region.errcheck = cls.__errcheck

            cls.__lib.slcam_set_sharpness_enhancement.argtypes = [ctypes.c_void_p, ctypes.c_int32]
            cls.__lib.slcam_set_sharpness_enhancement.restype = ctypes.c_int32
            cls.__lib.slcam_set_sharpness_enhancement.errcheck = cls.__errcheck

            cls.__lib.slcam_get_sharpness_enhancement.argtypes = [ctypes.c_void_p, ctypes.POINTER(ctypes.c_int32)]
            cls.__lib.slcam_get_sharpness_enhancement.restype = ctypes.c_int32
            cls.__lib.slcam_get_sharpness_enhancement.errcheck = cls.__errcheck
