#ifndef SLCAM_H_
#define SLCAM_H_

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(_WIN32)  // Windows

#ifdef SLCAM_EXPORT
#define SLCAM_API __declspec(dllexport)
#else
#define SLCAM_API __declspec(dllimport)
#endif

#else  // Linux / MacOS

#define SLCAM_API

#endif

typedef struct _SLcam *HSLcam;
#define SLCAM_MAX_DEVICES 16
#define SLCAM_MAX_CAP_SIZE 32

// Return Code
enum
{
    SLCAMRET_SUCCESS = 0x00000000,
    SLCAMRET_FAILURE = 0x80000000
};

typedef enum SLcamModel : uint16_t
{
    SLCAM_MODEL_M112U = 0x0112,
    SLCAM_MODEL_M114U = 0x0114,
    SLCAM_MODEL_M202U = 0x0202,
    SLCAM_MODEL_M122 = 0x0122,
    SLCAM_MODEL_M124 = 0x0124,
    SLCAM_MODEL_M212 = 0x0212,
    SLCAM_MODEL_C310 = 0xC310,
    SLCAM_MODEL_C311 = 0xC311,
    SLCAM_MODEL_C312 = 0xC312,
    SLCAM_MODEL_C313 = 0xC313,
    SLCAM_MODEL_C314 = 0xC314,
    SLCAM_MODEL_U106 = 0x1106,
    SLCAM_MODEL_U108 = 0x1108,
    SLCAM_MODEL_U112 = 0x1112,
    SLCAM_MODEL_U120 = 0x1120,
    SLCAM_MODEL_A311U = 0xA311,
    SLCAM_MODEL_A312U = 0xA312,
    SLCAM_MODEL_A313 = 0xA313,
    SLCAM_MODEL_A314 = 0xA314,
    SLCAM_MODEL_A321U = 0xA321,
    SLCAM_MODEL_A322U = 0xA322,
    SLCAM_MODEL_A323 = 0xA323,
    SLCAM_MODEL_A324 = 0xA324,
    SLCAM_MODEL_U405 = 0x1405,
    SLCAM_MODEL_U406 = 0x1406,
    SLCAM_MODEL_U408 = 0x1408,
    SLCAM_MODEL_U202 = 0x1202,
    SLCAM_MODEL_U205 = 0x1205,
    SLCAM_MODEL_U208 = 0x1208,
    SLCAM_MODEL_U304 = 0x1304,
    SLCAM_MODEL_U305 = 0x1305,
    SLCAM_MODEL_U306 = 0x1306,
    SLCAM_MODEL_B201 = 0x00F9,
    SLCAM_MODEL_L311 = 0x2311,
    SLCAM_MODEL_L313 = 0x2313,
    SLCAM_MODEL_L314 = 0x2314,
    SLCAM_MODEL_UNSUPPORT = 0xFFFF
} SLcamModel;

typedef enum SLcamLogLevel
{
    SLCAM_LOG_TRACE,
    SLCAM_LOG_DEBUG,
    SLCAM_LOG_INFO,
    SLCAM_LOG_WARNING,
    SLCAM_LOG_ERROR,
    SLCAM_LOG_CRITICAL,
    SLCAM_LOG_OFF
} SLcamLogLevel;

typedef enum SLcamFocusMode
{
    SLCAM_FOCUS_MODE_MANUAL = 0,
    SLCAM_FOCUS_MODE_AUTO,
    SLCAM_FOCUS_MODE_ONCE
} SLcamFocusMode;

typedef enum SLcamExposureMode
{
    SLCAM_EXPOSURE_MODE_MANUAL = 0,
    SLCAM_EXPOSURE_MODE_AUTO
} SLcamExposureMode;

typedef enum SLcamWhiteBalanceMode
{
    SLCAM_WHITE_BALANCE_MODE_MANUAL = 0,
    SLCAM_WHITE_BALANCE_MODE_AUTO
} SLcamWhiteBalanceMode;

typedef enum SLcamPowerLineFrequence
{
    SLCAM_POWER_LINE_FREQUENCE_60HZ = 0,
    SLCAM_POWER_LINE_FREQUENCE_50HZ
} SLcamPowerLineFrequence;

typedef enum SLcamExposureGainUnit
{
    SLCAM_EXPOSURE_GAIN_MAGNIFICATION = 0,
    SLCAM_EXPOSURE_GAIN_DB
} SLcamGainUnit;

typedef enum SLcamVideoFormat
{
    SLCAM_VIDEO_FORMAT_UNKNOWN = -1,
    SLCAM_VIDEO_FORMAT_I420,
    SLCAM_VIDEO_FORMAT_J420,
    SLCAM_VIDEO_FORMAT_IYUV,
    SLCAM_VIDEO_FORMAT_RGB24,
    SLCAM_VIDEO_FORMAT_BGR24,
    SLCAM_VIDEO_FORMAT_ABGR,
    SLCAM_VIDEO_FORMAT_ARGB,
    SLCAM_VIDEO_FORMAT_RGBA,
    SLCAM_VIDEO_FORMAT_BGRA,
    SLCAM_VIDEO_FORMAT_RGB565,
    SLCAM_VIDEO_FORMAT_YUY2,
    SLCAM_VIDEO_FORMAT_UYVY,
    SLCAM_VIDEO_FORMAT_MJPEG,
    SLCAM_VIDEO_FORMAT_H264,
    SLCAM_VIDEO_FORMAT_H265,
    SLCAM_VIDEO_FORMAT_NV12
} SLcamVideoFormat;

typedef enum SLcamPixFormat
{
    SLCAM_PIX_FORMAT_UNKNOWN = -1,
    SLCAM_PIX_FORMAT_I420,
    SLCAM_PIX_FORMAT_J420,
    SLCAM_PIX_FORMAT_IYUV,
    SLCAM_PIX_FORMAT_RGB24,
    SLCAM_PIX_FORMAT_BGR24,
    SLCAM_PIX_FORMAT_ABGR,
    SLCAM_PIX_FORMAT_ARGB,
    SLCAM_PIX_FORMAT_RGBA,
    SLCAM_PIX_FORMAT_BGRA,
    SLCAM_PIX_FORMAT_RGB565,
    SLCAM_PIX_FORMAT_YUY2,
    SLCAM_PIX_FORMAT_UYVY,
    SLCAM_PIX_FORMAT_NV12,
    SLCAM_PIX_FORMAT_GRAY8
} SLcamPixFormat;

typedef enum SLcamImgFormat
{
    SLCAM_IMG_FORMAT_UNKNOWN = -1,
    SLCAM_IMG_FORMAT_PNG,
    SLCAM_IMG_FORMAT_JPG,
    SLCAM_IMG_FORMAT_BMP
} SLcamImgFormat;

typedef struct SLcamVideoResolution
{
    int32_t width;
    int32_t height;
} SLcamVideoResolution;

typedef struct SLcamVideoFrame
{
    uint8_t *data[4];
    SLcamPixFormat fmt;
    int32_t width;
    int32_t height;
    int32_t linesize[4];
    int64_t pts;
} SLcamVideoFrame;

typedef struct SLcamCaptureContext
{
    char uniqueName[128];
    SLcamVideoResolution resolution;
    SLcamVideoFormat capFmt;
    SLcamPixFormat readFmt;
    int32_t fps;
} SLcamCaptureContext;

typedef struct SLcamVideoCaptureCapability
{
    SLcamVideoResolution resolution;
    int32_t maxFps;
    SLcamVideoFormat videoFmt;
} SLcamVideoCaptureCapability;

typedef struct SLcamVideoCaptureCapabilities
{
    SLcamVideoCaptureCapability videoCaps[SLCAM_MAX_CAP_SIZE];
    int32_t capNum;
} SLcamVideoCaptureCapabilities;

/**
 * Low speed   USB1.0
 * Full speed  USB1.1
 * High speed  USB2.0
 * Super speed USB3.0
 */
typedef enum SLcamDevUsbSpeed
{
    SLCAM_USB_LOW_SPEED = 0,
    SLCAM_USB_FULL_SPEED,
    SLCAM_USB_HIGH_SPEED,
    SLCAM_USB_SUPER_SPEED
} SLcamDevUsbSpeed;

typedef struct SLcamDevInfo
{
    char name[128];
    char uniqueName[128];
    uint16_t vendorId;
    SLcamModel model;
    SLcamDevUsbSpeed speed;
} SLcamDevInfo;

typedef struct SLcamDevInfos
{
    SLcamDevInfo cameras[SLCAM_MAX_DEVICES];
    int32_t cameraNum;
} SLcamDevInfos;

typedef struct SLcamFileSaveInfo
{
    SLcamImgFormat format;
    char *savePath;
    SLcamVideoFrame *frame;
} SLcamFileSaveInfo;

typedef struct SLcamRecordSaveInfo
{
    char *savePath;
    int32_t width;
    int32_t height;
    int32_t fps;
} SLcamRecordSaveInfo;

typedef struct SLPoint
{
    uint8_t x;
    uint8_t y;
} SLPoint;

typedef enum SLcamGammaMode
{
    SLCAM_GAMMA_MODE_NORMAL = 0,
    SLCAM_GAMMA_MODE_BEZIER
} SLcamGammaMode;

typedef void (*SLcamLogCallback)(int32_t level, const char *msg);
typedef void (*SLcamProgressCallback)(int32_t percent, void *ctx);

SLCAM_API int32_t slcam_api_init();
SLCAM_API int32_t slcam_api_destroy();

SLCAM_API int32_t slcam_log_set_level(SLcamLogLevel level);
SLCAM_API int32_t slcam_log_set_pattern(const char *pattern);
SLCAM_API int32_t slcam_log_set_callback(SLcamLogCallback callback);

SLCAM_API int32_t slcam_enum_devices(SLcamDevInfos *info);

SLCAM_API int32_t slcam_open(const char *uniqueName, HSLcam *cam);
SLCAM_API int32_t slcam_get_capture_capabilities(HSLcam cam, SLcamVideoCaptureCapabilities *capabilities);
SLCAM_API int32_t slcam_set_capture_context(HSLcam cam, const SLcamCaptureContext *ctx);
SLCAM_API int32_t slcam_close(HSLcam cam);
SLCAM_API int32_t slcam_read_frame(HSLcam cam, SLcamVideoFrame *frame);
SLCAM_API int32_t slcam_free_frame(SLcamVideoFrame *frame);

SLCAM_API int32_t slcam_file_save_image(HSLcam cam, SLcamFileSaveInfo fileSaveInfo);

SLCAM_API int32_t slcam_record_start(HSLcam cam, SLcamRecordSaveInfo recordSaveInfo);
SLCAM_API int32_t slcam_record_append_frame(HSLcam cam, const SLcamVideoFrame *frame);
SLCAM_API int32_t slcam_record_stop(HSLcam cam);

/**
 * Set focus mode
 * @param mode  0 manual, 1 auto, 2 once/trigger once
 */
SLCAM_API int32_t slcam_set_focus_mode(HSLcam cam, int32_t mode);
SLCAM_API int32_t slcam_get_focus_mode(HSLcam cam, int32_t *mode);
SLCAM_API int32_t slcam_get_focus_mode_ex(HSLcam cam, int32_t *mode);

/**
 * Get focus state
 * @param state  0 focusing, 1 focus done
 */
SLCAM_API int32_t slcam_get_focus_state(HSLcam cam, int32_t *state);

SLCAM_API int32_t slcam_set_focus(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_focus(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_focus_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_auto_focus_range(HSLcam cam, int32_t minValue, int32_t maxValue);
SLCAM_API int32_t slcam_get_auto_focus_range(HSLcam cam, int32_t *minValue, int32_t *maxValue);

/**
 * Increase/Decrease one focus value. Only for M Serise.
 */
SLCAM_API int32_t slcam_increase_focus(HSLcam cam);
SLCAM_API int32_t slcam_decrease_focus(HSLcam cam);

/**
 * Set focus region, in auto focus mode(old version)
 * Support all focus camera
 * @param x     x pos
 * @param y     y pos
 */
SLCAM_API int32_t slcam_set_focus_region_v1(HSLcam cam, int32_t x, int32_t y);

/**
 * Get focus region(old version)
 * Support all focus camera
 */
SLCAM_API int32_t slcam_get_focus_region_v1(HSLcam cam, int32_t *x, int32_t *y);

/**
 * Set focus region, in auto focus mode
 * Support specific version camera (c31x 1.4.0/A311u 1.0.0/m202u 1.2.0/m112u 1.1.1)
 * @param enable 0 disable, 1 enable
 * @param mode   0 small,   1 big
 *                 (small  0 <= x <= 16, 0 <= y <= 14)
 *                 (big    0 <= x <= 4,  0 <= y <= 4)
 * @param  x     x pos
 * @param  y     y pos
 */
SLCAM_API int32_t slcam_set_focus_region_v2(HSLcam cam, int32_t enable, int32_t mode, int32_t x, int32_t y);

/**
 * Get focus region
 * Support specific version camera (c31x 1.4.0/A311u 1.0.0/m202u 1.2.0/m112u 1.1.1)
 */
SLCAM_API int32_t slcam_get_focus_region_v2(HSLcam cam, int32_t *enable, int32_t *mode, int32_t *x, int32_t *y);

/** Get focus region range
 *  Only get the range of small mode
 */
SLCAM_API int32_t slcam_get_focus_region_range(HSLcam cam, int32_t *minX, int32_t *minY, int32_t *maxX, int32_t *maxY, int32_t *defX, int32_t *defY, int32_t *stepX, int32_t *stepY);

/**
 * Set focus region, in auto focus mode
 * @param enable   0 disable, 1 enable
 * @param  x       x pos
 * @param  y       y pos
 * @param  width   width
 * @param  height  height
 */
SLCAM_API int32_t slcam_set_focus_region_v3(HSLcam cam, int32_t enable, int32_t x, int32_t y, int32_t width, int32_t height);

/**
 * Get focus region
 */
SLCAM_API int32_t slcam_get_focus_region_v3(HSLcam cam, int32_t *enable, int32_t *x, int32_t *y, int32_t *width, int32_t *height);

/**
 * Get firmware version
 * C31xu/U1xx/M112u/M202u: real_version = double(version) / 100;
 * B201: real_version = double(version & 0xFFFF) / 100;
 */
SLCAM_API int32_t slcam_get_version(HSLcam cam, int32_t *version);

/**
 * Set power line frequence
 * @param flag  0 60hz, 1 50hz
 */
SLCAM_API int32_t slcam_set_power_line_frequency(HSLcam cam, int32_t flag);

/**
 * Get power line frequence
 * @param flag  0 60hz, 1 50hz
 */
SLCAM_API int32_t slcam_get_power_line_frequency(HSLcam cam, int32_t *flag);

/**
 * Set horizontal flip
 * @param enable  0 disable, 1 enable
 */
SLCAM_API int32_t slcam_set_mirror(HSLcam cam, int32_t enable);

/**
 * Get horizontal flip
 * @param enable 0 disable, 1 enable
 */
SLCAM_API int32_t slcam_get_mirror(HSLcam cam, int32_t *enable);

/**
 * Set vertical flip
 * @param enable  0 disable, 1 enable
 */
SLCAM_API int32_t slcam_set_flip(HSLcam cam, int32_t enable);

/**
 * Get vertical flip
 * @param enable  0 disable, 1 enable
 */
SLCAM_API int32_t slcam_get_flip(HSLcam cam, int32_t *enable);

SLCAM_API int32_t slcam_set_hue(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_hue(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_hue_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_sharpness(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_sharpness(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_sharpness_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_saturation(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_saturation(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_saturation_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_contrast(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_contrast(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_contrast_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

/**
 * Set zoom and dzoom value, value range(zoomMin + dzoomMin, zoomMax + dzoomMax) 
 * NOTICE: When value > zoomMax, enable dzoom, dzoomValue = value - zoomValue
 *         dzoom: digital zoom
 */
SLCAM_API int32_t slcam_set_zoom(HSLcam cam, int32_t value);
/**
 * Get zoom and dzoom value
 * When value > zoomMax, zoomValue = zoomMax, dzoomValue = value - zoomValue
 * When value <= zoomMax, zoomValue = value, dzoomValue = 0
 */
SLCAM_API int32_t slcam_get_zoom(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_ozoom_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);
SLCAM_API int32_t slcam_get_dzoom_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

/**
 * Set zoom relative
 * @param zoomRel      0 stop, 1 zoom out, -1 zoom in
 * @param digitalZoom  invaild
 * @param speed         invaild
 */
SLCAM_API int32_t slcam_set_zoom_relative(HSLcam cam, int8_t zoomRel, uint8_t digitalZoom, uint8_t speed);
SLCAM_API int32_t slcam_get_zoom_relative(HSLcam cam, int8_t *zoomRel, uint8_t *digitalZoom, uint8_t *speed);

/**
 * Set zoom speed
 * @param speed  0 low, 1 medium, 2 high
 */
SLCAM_API int32_t slcam_set_zoom_speed(HSLcam cam, int32_t speed);
/**
 * Get zoom speed
 * @param speed  0 low, 1 medium, 2 high
 */
SLCAM_API int32_t slcam_get_zoom_speed(HSLcam cam, int32_t *speed);

/**
 * Set exposure mode
 * @param mode  0 manual, 1 auto
 */
SLCAM_API int32_t slcam_set_exposure_mode(HSLcam cam, int32_t mode);

/**
 * Get exposure mode
 * @param mode  0 manual, 1 auto
 */
SLCAM_API int32_t slcam_get_exposure_mode(HSLcam cam, int32_t *mode);

SLCAM_API int32_t slcam_set_exposure_compensation(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_exposure_compensation(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_exposure_compensation_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_exposure_time(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_exposure_time(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_exposure_time_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_long_exposure_time(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_long_exposure_time(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_long_exposure_time_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_exposure_gain(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_exposure_gain(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_exposure_gain_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

/**
 * Set gain unit
 * @param unit  0 magnification, 1 db
 */
SLCAM_API int32_t slcam_set_exposure_gain_unit(HSLcam cam, int32_t unit);

/**
 * Get gain unit
 * @param unit  0 magnification, 1 db
 */
SLCAM_API int32_t slcam_get_exposure_gain_unit(HSLcam cam, int32_t *unit);

SLCAM_API int32_t slcam_set_gamma(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_gamma(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_gamma_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

/**
 * Set white balance mode
 * @param mode  0 manual, 1 auto
 */
SLCAM_API int32_t slcam_set_white_balance_mode(HSLcam cam, int32_t mode);

/**
 * Get white balance mode
 * @param mode  0 manual, 1 auto
 */
SLCAM_API int32_t slcam_get_white_balance_mode(HSLcam cam, int32_t *mode);

SLCAM_API int32_t slcam_set_white_balance_temperature(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_white_balance_temperature(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_white_balance_temperature_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_white_balance_component_red(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_white_balance_component_red(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_white_balance_component_red_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_white_balance_component_green(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_white_balance_component_green(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_white_balance_component_green_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_white_balance_component_blue(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_white_balance_component_blue(HSLcam cam, int32_t *value);
SLCAM_API int32_t slcam_get_white_balance_component_blue_range(HSLcam cam, int32_t *minValue, int32_t *maxValue, int32_t *defValue, int32_t *stepValue);

SLCAM_API int32_t slcam_set_auto_exposure_region(HSLcam cam, int32_t x, int32_t y, int32_t w, int32_t h);
SLCAM_API int32_t slcam_get_auto_exposure_region(HSLcam cam, int32_t *x, int32_t *y, int32_t *w, int32_t *h);

SLCAM_API int32_t slcam_set_auto_whitebalance_region(HSLcam cam, int32_t x, int32_t y, int32_t w, int32_t h);
SLCAM_API int32_t slcam_get_auto_whitebalance_region(HSLcam cam, int32_t *x, int32_t *y, int32_t *w, int32_t *h);

/**
 * Change scene
 * @param flag  0 default scene, 1 metallography scene
 */
SLCAM_API int32_t slcam_set_scene(HSLcam cam, int32_t flag);
SLCAM_API int32_t slcam_get_scene(HSLcam cam, int32_t *flag);

/**
 * Dynamic Range Compression
 * @param value range: 0-127
 */
SLCAM_API int32_t slcam_set_drc(HSLcam cam, int32_t value);
SLCAM_API int32_t slcam_get_drc(HSLcam cam, int32_t *value);

/**
 * Chromatic Abberation Correction, for depurple
 * @param enable 0 disable, 1 enable
 */
SLCAM_API int32_t slcam_set_cac(HSLcam cam, int32_t enable);
SLCAM_API int32_t slcam_get_cac(HSLcam cam, int32_t *enable);

/**
 * Local Dynamic Contrast Improvement
 * @param enable 0 disable, 1 enable
 */
SLCAM_API int32_t slcam_set_ldci(HSLcam cam, int32_t enable);
SLCAM_API int32_t slcam_get_ldci(HSLcam cam, int32_t *enable);

/**
 * Weak Texture Enhancement
 * @param enable 0 disable, 1 enable
 */
SLCAM_API int32_t slcam_set_bayer_shp(HSLcam cam, int32_t enable);
SLCAM_API int32_t slcam_get_bayer_shp(HSLcam cam, int32_t *enable);

/**
 * LED control
 * @param ledIndex  led index
 * @param partIndex part index
 * @param level      led brightness
 */
SLCAM_API int32_t slcam_set_led(HSLcam cam, int32_t ledIndex, int32_t partIndex, int32_t level);
SLCAM_API int32_t slcam_get_led(HSLcam cam, int32_t ledIndex, int32_t partIndex, int32_t *level);
SLCAM_API int32_t slcam_get_led_info(HSLcam cam, int32_t *ledNumber, int32_t *partNumber, int32_t *minLevel, int32_t *maxLevel, int32_t *defLevel, int32_t *stepLevel);

/**
 * A3XX LED control
 * @param up,left,right,down  brightness of the led postion
 * partIndex for led pos: 0-up, 1-left, 2-right, 3-down
 */
SLCAM_API int32_t slcam_set_a3xx_all_led(HSLcam cam, int32_t up, int32_t left, int32_t right, int32_t down);
SLCAM_API int32_t slcam_get_a3xx_all_led(HSLcam cam, int32_t *up, int32_t *left, int32_t *right, int32_t *down);

/**
 * A3XX LED control
 * @param partIndex part index
 * @param enable     led enable state, 0-close, 1-open
 * @param level      led brightness
 */
SLCAM_API int32_t slcam_set_a3xx_led(HSLcam cam, int32_t partIndex, int32_t enable, int32_t level);
SLCAM_API int32_t slcam_get_a3xx_led(HSLcam cam, int32_t partIndex, int32_t *enable, int32_t *level);

/**
 * Set defog mode, heat to prevent fogging up
 * @param enable  0 disable, 1 enable
 */
SLCAM_API int32_t slcam_set_defog(HSLcam cam, int32_t enable);

/**
 * Get defog mode
 * @param enable  0 disable, 1 enable
 */
SLCAM_API int32_t slcam_get_defog(HSLcam cam, int32_t *enable);

/**
 * Set user id
 * @param userID  range(0x0-0x3FFFFFF)
 */
SLCAM_API int32_t slcam_set_user_id(HSLcam cam, int32_t userId);

/**
 * Get user id
 */
SLCAM_API int32_t slcam_get_user_id(HSLcam cam, int32_t *userId);

/** NOTE DO NOT USE THIS!!!!!!!
 *  Only use for M series test
 */
SLCAM_API int32_t slcam_focus_cure_test(HSLcam cam);
SLCAM_API int32_t slcam_focus_cure_check(HSLcam cam);
SLCAM_API int32_t slcam_set_encrypt(HSLcam cam, int32_t enable);
SLCAM_API int32_t slcam_get_encrypt(HSLcam cam, int32_t *enable);

/**
 * Tell camera current windows version. 
 * @param windowsVersion  7 windows7, 10 windows10, 11 windows11, other <windows7
 */
SLCAM_API int32_t slcam_set_windows_version(HSLcam cam, int32_t windowsVersion);

/**
 * Get model suffix
 * @param suffix model suffix
 * For example, U205A's suffix is 'A', U205's suffix is 'NUL'
 * Only one char, lowercase
 */
SLCAM_API int32_t slcam_get_model_suffix(HSLcam cam, char *suffix);

/**
 * Firmware upgrade (U408 only)
 * @param filePath upgrade file full path
 * @param callback progress percent callback
 * @param ctx      custom ctx
 */
SLCAM_API int32_t slcam_upgrade(HSLcam cam, const char *filePath, SLcamProgressCallback callback, void *ctx);

/**
 * Set device name (U408 only)
 * @param name new device name 
 */
SLCAM_API int32_t slcam_set_device_name(HSLcam cam, const char *name);

/**
 * Get unique id
 * @param uniqueId  unique id (max length 60)
 * @param len        input uniqueId length
 * @param readLen   uniqueId actual length
 */
SLCAM_API int32_t slcam_get_unique_id(HSLcam cam, uint8_t *uniqueId, int32_t len, int32_t *readLen);

/**
 * get gamma mode
 * @param mode  0 ,1 bezier_curve
 */
SLCAM_API int32_t slcam_get_gamma_mode(HSLcam cam, int32_t *mode);

/**
 * set gamma mode
 * @param mode  0 ,1 bezier_curve
 */
SLCAM_API int32_t slcam_set_gamma_mode(HSLcam cam, int32_t mode);

/**
 * @brief Set Gamma Bezier Curve.
 * 
 * Sets the Gamma Bezier curve on the device with the provided points.
 * 
 * @param points  The array of SLPoint defining the curve.
 * @param size    The number of points (max 25).
 */
SLCAM_API int32_t slcam_set_gamma_bezier_curve(HSLcam cam, const SLPoint *points, int32_t size);

/**
 * @brief Get Gamma Bezier Curve.
 * 
 * Retrieves the current Gamma Bezier curve from the device.
 * 
 * @param points  The array to store retrieved points.
 * @param size    The number of points (max 25).
 * @param readSize   The pointer to store the number of points.

 */
SLCAM_API int32_t slcam_get_gamma_bezier_curve(HSLcam cam, SLPoint *points, int32_t size, int32_t *readSize);

/**
 * Set roi region, called only after 'set_capture_context' has succeeded
 * @param enable   0 disable, 1 enable
 * @param  x       x pos, should be even
 * @param  y       y pos, should be even
 * @param  width   width, should be a multiple of 8
 * @param  height  height, should be a multiple of 8
 */
SLCAM_API int32_t slcam_set_roi_region(HSLcam cam, int32_t enable, int32_t x, int32_t y, int32_t width, int32_t height);

/**
 * Get roi region, called only after 'set_capture_context' has succeeded
 */
SLCAM_API int32_t slcam_get_roi_region(HSLcam cam, int32_t *enable, int32_t *x, int32_t *y, int32_t *width, int32_t *height);

/**
 * Set sharpness enhancement
 * @param enable 0 disable, 1 enable
 */
SLCAM_API int32_t slcam_set_sharpness_enhancement(HSLcam cam, int32_t enable);

/**
 * Get sharpness enhancement
 * @param enable 0 disable, 1 enable
 */
SLCAM_API int32_t slcam_get_sharpness_enhancement(HSLcam cam, int32_t *enable);

#ifdef __cplusplus
}
#endif

#endif  // SLCAM_H_
